/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.repository.redis

import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.Vertx
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.redis.client.redisOptionsOf
import io.vertx.redis.client.RedisClientType.CLUSTER
import io.vertx.redis.client.RedisClientType.STANDALONE

/**
 * This class contains logic for get and performing retryable redis commands.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */

private val logger = KotlinLogging.logger {}

object RedisUtils {
    @Suppress("unused")
    fun getRedisClient(config: JsonObject): GenghisRedisClient = getRedisClient(Vertx.currentContext().owner(), config)

    fun getRedisClient(
        vertx: Vertx,
        config: JsonObject,
    ): GenghisRedisClient {
        val redisServer = config.getString("redis_host") ?: "localhost"
        val redisPort = config.getInteger("redis_port") ?: 6380
        val tls = config.getBoolean("redis_tls") ?: false
        val clustered = config.getBoolean("redis_clustered") ?: false
        val replication = config.getBoolean("redis_replication") ?: false
        val additionalConnections = config.getJsonArray("redis_connections")?.map { it.toString() } ?: emptyList()
        val redisOptions =
            redisOptionsOf(
                maxPoolSize = 48,
                maxPoolWaiting = 96,
                type = if (clustered) CLUSTER else STANDALONE,
            )
        additionalConnections.forEach {
            redisOptions.addConnectionString(it)
        }
        redisOptions.setConnectionString("redis${if (tls) "s" else ""}://$redisServer:$redisPort")

        return GenghisRedisClient(
            vertx,
            GenghisRedisConfiguration(
                redisServer = redisServer,
                redisPort = redisPort,
                clustered = clustered,
                tls = tls,
                replication = replication,
                redisOptions = redisOptions,
            ),
        )
    }
}
