/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

@file:Suppress(
    "BooleanMethodIsAlwaysInverted",
    "ClassName",
    "unused",
)

package com.genghis.tools.repository.utils

import com.fasterxml.jackson.annotation.JsonIgnore
import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.utils.AggregateFunctions.COUNT
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.codegen.annotations.Fluent
import io.vertx.core.json.JsonObject

private val logger = KotlinLogging.logger { }

/**
 * This class defines an aggregation function with the field, the function, and any grouping parameters.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
@Suppress("BooleanMethodIsAlwaysInverted", "unused")
data class AggregateFunction(
    val function: AggregateFunctions,
    val field: String? = null,
    val groupBy: List<GroupingConfiguration> = emptyList(),
) {
    @JsonIgnore
    val validationError: JsonObject = JsonObject()

    val isMin: Boolean
        get() = function == AggregateFunctions.MIN

    val isMax: Boolean
        get() = function == AggregateFunctions.MAX

    val isAverage: Boolean
        get() = function == AggregateFunctions.AVG

    val isSum: Boolean
        get() = function == AggregateFunctions.SUM

    val isCount: Boolean
        get() = function == COUNT

    enum class TIMEUNIT_DATE {
        HOUR,
        TWELVE_HOUR,
        DAY,
        WEEK,
        MONTH,
        YEAR,
    }

    @Suppress("unused")
    class AggregateFunctionBuilder internal constructor() {
        private var function: AggregateFunctions? = null
        private var field: String? = null
        private var groupBy: List<GroupingConfiguration> = emptyList()

        fun build(): AggregateFunction {
            if (function == null) throw IllegalArgumentException("Function cannot be null!")
            if (field == null && function != COUNT) throw IllegalArgumentException("Field cannot be null!")

            return AggregateFunction(
                function = function!!,
                field = field,
                groupBy = groupBy,
            )
        }

        @Fluent
        fun withAggregateFunction(function: AggregateFunctions): AggregateFunctionBuilder {
            this.function = function

            return this
        }

        @Fluent
        fun withField(field: String?): AggregateFunctionBuilder {
            this.field = field

            return this
        }

        @Fluent
        fun withGroupBy(groupBy: List<GroupingConfiguration>): AggregateFunctionBuilder {
            if (groupBy.size > 3) throw IllegalArgumentException("You can only group 3 levels deep!")

            val newGroup = this.groupBy.toMutableList()
            newGroup.addAll(groupBy)
            this.groupBy = newGroup

            return this
        }

        @Suppress(
            "unused",
        )
        @Fluent
        fun addGroupBy(groupBy: GroupingConfiguration): AggregateFunctionBuilder {
            if (this.groupBy.size == 3) throw IllegalArgumentException("You can only group 3 levels deep!")

            val newGroup = this.groupBy.toMutableList()
            newGroup.add(groupBy)
            this.groupBy = newGroup

            return this
        }
    }

    fun hasGrouping(): Boolean = groupBy.isNotEmpty()

    fun <E : ETagable> validateFieldForFunction(modelType: Class<E>): Boolean {
        if (!isMin && !isMax && !isAverage) return true

        if (field == null) {
            val errorMessage = "Field name cannot be null..."

            when (function) {
                AggregateFunctions.MIN -> validationError.put("min_error", errorMessage)
                AggregateFunctions.MAX -> validationError.put("max_error", errorMessage)
                AggregateFunctions.AVG -> validationError.put("avg_error", errorMessage)
                else -> throw IllegalStateException("Illegal aggreation null!")
            }

            return false
        }

        return when {
            hasGrouping() ->
                groupBy
                    .map { groupingConfiguration ->
                        groupingConfiguration.validate(modelType, field, validationError)
                    }.any { !it }
            else -> validationError.isEmpty
        }
    }

    companion object {
        fun builder(init: AggregateFunctionBuilder.() -> Unit): AggregateFunction {
            val builder = AggregateFunctionBuilder()
            builder.init()

            return builder.build()
        }
    }
}
