/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.dynamodb.operators

import com.genghis.tools.repository.dynamodb.DynamoDBRepository
import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.DynamoDBModel
import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.models.ModelUtils
import com.genghis.tools.repository.repository.cache.CacheManager
import com.genghis.tools.repository.repository.etag.ETagManager
import com.genghis.tools.repository.utils.AggregateFunction
import com.genghis.tools.repository.utils.AggregateFunctions
import com.genghis.tools.repository.utils.AggregateFunctions.MAX
import com.genghis.tools.repository.utils.AggregateFunctions.MIN
import com.genghis.tools.repository.utils.DynamoDBQuery
import com.genghis.tools.repository.utils.GroupingConfiguration
import com.genghis.tools.repository.utils.toJson
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.json.Json
import io.vertx.core.json.Json.encodePrettily
import io.vertx.core.json.JsonArray
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.core.json.jsonObjectOf
import java.time.Duration
import java.util.AbstractMap.SimpleEntry
import java.util.Comparator.comparing
import java.util.Comparator.comparingLong
import java.util.Date
import java.util.NoSuchElementException
import java.util.Objects
import java.util.function.BiFunction
import java.util.stream.Collectors.averagingDouble
import java.util.stream.Collectors.counting
import java.util.stream.Collectors.groupingBy
import java.util.stream.Collectors.summingDouble
import java.util.stream.Collectors.toList
import java.util.stream.Collectors.toMap
import java.util.stream.IntStream
import kotlin.math.ceil
import kotlin.math.floor

private val logger = KotlinLogging.logger { }

/**
 * This class defines the aggregate operations for the DynamoDBRepository.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
class DynamoDBAggregates<E>(
    private val TYPE: Class<E>,
    private val db: DynamoDBRepository<E>,
    private val HASH_IDENTIFIER: String,
    private val IDENTIFIER: String?,
    private val cacheManager: CacheManager<E>?,
    private val eTagManager: ETagManager<E>?,
) where E : ETagable, E : Cacheable, E : DynamoDBModel, E : Model {
    suspend fun aggregation(query: DynamoDBQuery): String {
        if (logger.isDebugEnabled()) {
            logger.debug {
                "QueryPack is: ${encodePrettily(query)}, projections: ${query.projections}, ids: ${
                    query.identifiers.toJson().encodePrettily()
                }"
            }
        }

        return when (query.aggregate!!.function) {
            MIN -> findItemsWithMinOfField(query)
            MAX -> findItemsWithMaxOfField(query)
            AggregateFunctions.AVG -> avgField(query)
            AggregateFunctions.SUM -> sumField(query)
            AggregateFunctions.COUNT -> countItems(query)
        }
    }

    private suspend fun findItemsWithMinOfField(query: DynamoDBQuery): String =
        performMinOrMaxAggregation(
            query,
            "MIN",
        ) { r, f -> getAllItemsWithLowestValue(r, f) }

    private suspend fun findItemsWithMaxOfField(query: DynamoDBQuery): String =
        performMinOrMaxAggregation(
            query,
            "MAX",
        ) { r, f -> getAllItemsWithHighestValue(r, f) }

    private fun calculateGroupingPageToken(
        groupingParam: List<GroupingConfiguration>?,
        projs: MutableSet<Set<String>>,
        finalProjections: Set<String>,
    ) {
        groupingParam
            ?.stream()
            ?.filter { param -> finalProjections.none { p -> p == param.groupBy } }
            ?.forEach { groupByParam ->
                val newProjectionArray = arrayOfNulls<String>(finalProjections.size + 1)
                IntStream.range(0, finalProjections.size).forEach { i ->
                    newProjectionArray[i] = finalProjections.elementAt(i)
                }
                newProjectionArray[finalProjections.size] = groupByParam.groupBy
                projs.add(newProjectionArray.filterNotNull().toSet())
            }
    }

    private fun calculateGroupings(
        aggregateFunction: AggregateFunction?,
        minItems: List<E>,
    ): JsonObject =
        performGroupingAndSorting(
            minItems,
            aggregateFunction!!,
        ) { items, groupingConfigurations ->
            if (groupingConfigurations.size > 3) throw IllegalArgumentException("GroupBy size of three is max!")
            val levelOne = groupingConfigurations[0]
            val levelTwo = if (groupingConfigurations.size > 1) groupingConfigurations[1] else null
            val levelThree = if (groupingConfigurations.size > 2) groupingConfigurations[2] else null

            when {
                levelTwo == null ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy {
                                calculateGroupingKey(it, levelOne)
                            },
                        )

                levelThree == null ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                groupingBy {
                                    calculateGroupingKey(it, levelTwo)
                                },
                            ),
                        )

                else ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                groupingBy(
                                    {
                                        calculateGroupingKey<E>(it, levelTwo)
                                    },
                                    groupingBy {
                                        calculateGroupingKey(it, levelThree)
                                    },
                                ),
                            ),
                        )
            }
        }

    private fun getAllItemsWithLowestValue(
        records: List<E>,
        field: String,
    ): List<E> {
        val result = ArrayList<E>()
        val min = ArrayList<E>()

        records.forEach {
            if (min.isEmpty() ||
                db.extractValueAsDouble(db.checkAndGetField(field), it) <
                db.extractValueAsDouble(db.checkAndGetField(field), min[0])
            ) {
                min.add(it)
                result.clear()
                result.add(it)
            } else if (min.isNotEmpty() &&
                db
                    .extractValueAsDouble(db.checkAndGetField(field), it)
                    .compareTo(db.extractValueAsDouble(db.checkAndGetField(field), min[0])) == 0
            ) {
                result.add(it)
            }
        }

        return result
    }

    private suspend fun performMinOrMaxAggregation(
        query: DynamoDBQuery,
        command: String,
        valueExtractor: BiFunction<List<E>, String, List<E>>,
    ): String {
        val hashCode =
            when {
                query.aggregate!!.groupBy.isEmpty() -> 0
                else -> query.aggregate.groupBy.hashCode()
            }
        val aggregateFunction = query.aggregate
        val field = aggregateFunction.field
        val newEtagKeyPostfix = "_" + field + "_" + command
        val baseEtagKey = query.baseEtagKey()
        val etagKey = baseEtagKey + newEtagKeyPostfix + hashCode
        val cacheKey = baseEtagKey + newEtagKeyPostfix + hashCode
        val groupingParam = query.aggregate.groupBy
        val projections = query.projections

        return runCatching {
            cacheManager?.checkAggregationCache(cacheKey, TYPE.simpleName) ?: throw NoSuchElementException()
        }.recoverCatching {
            val projs = mutableSetOf(projections)

            calculateGroupingPageToken(groupingParam, projs, projections)

            val finalProjections2 = if (projs.isEmpty()) setOf() else projs.last()

            if (field != null) {
                if (finalProjections2.none { p -> p.equals(field, ignoreCase = true) }) {
                    val newProjectionArray = arrayOfNulls<String>(finalProjections2.size + 1)
                    IntStream
                        .range(0, finalProjections2.size)
                        .forEach { i -> newProjectionArray[i] = finalProjections2.elementAt(i) }
                    newProjectionArray[finalProjections2.size] = field
                    projs.add(newProjectionArray.filterNotNull().toSet())
                }
            }

            logger.debug { "Projections: ${projs.last()}" }

            val records =
                db
                    .index(
                        query.copy(
                            autoPaginate = true,
                            projections = projs.last(),
                        ),
                    ).itemList.items

            return when {
                records.isEmpty() ->
                    setEtagAndCacheAndReturnContent(
                        etagKey,
                        query.identifiers.hashCode(),
                        cacheKey,
                        JsonObject().put("error", "Empty table!").encode(),
                    )

                else ->
                    when {
                        query.aggregate.hasGrouping() -> {
                            val maxItems = getAllItemsWithHighestValue(records, field!!)
                            val aggregatedItems = calculateGroupings(aggregateFunction, maxItems)

                            setEtagAndCacheAndReturnContent(
                                etagKey,
                                query.identifiers.hashCode(),
                                cacheKey,
                                aggregatedItems.encode(),
                            )
                        }

                        else -> {
                            val items = JsonArray()
                            valueExtractor
                                .apply(records, field!!)
                                .stream()
                                .map { o -> o.toJsonFormat() }
                                .forEach { items.add(it) }

                            setEtagAndCacheAndReturnContent(
                                etagKey,
                                query.identifiers.hashCode(),
                                cacheKey,
                                items.encode(),
                            )
                        }
                    }
            }
        }.getOrThrow()
    }

    private fun addIdentifiers(projections: Set<String>): Set<String> {
        val projs = addHashIdentifierToProjections(projections)

        return addRangeIdentifierToProjections(projs)
    }

    private fun addHashIdentifierToProjections(projections: Set<String>): Set<String> =
        when {
            projections.none { p -> p.equals(HASH_IDENTIFIER, ignoreCase = true) } -> {
                val newProjectionArray = arrayOfNulls<String>(projections.size + 1)
                IntStream.range(0, projections.size).forEach { i -> newProjectionArray[i] = projections.elementAt(i) }
                newProjectionArray[projections.size] = HASH_IDENTIFIER

                newProjectionArray.filterNotNull().toSet()
            }

            else -> projections
        }

    private fun addRangeIdentifierToProjections(projections: Set<String>): Set<String> =
        when {
            IDENTIFIER != null && projections.none { p -> p.equals(IDENTIFIER, ignoreCase = true) } -> {
                val newProjectionArray = arrayOfNulls<String>(projections.size + 1)
                IntStream.range(0, projections.size).forEach { i -> newProjectionArray[i] = projections.elementAt(i) }
                newProjectionArray[projections.size] = IDENTIFIER

                newProjectionArray.filterNotNull().toSet()
            }

            else -> projections
        }

    private fun getAllItemsWithHighestValue(
        records: List<E>,
        field: String,
    ): List<E> {
        val result = ArrayList<E>()
        val max = ArrayList<E>()

        records.forEach {
            when {
                max.isEmpty() ||
                    db.extractValueAsDouble(db.checkAndGetField(field), it) >
                    db.extractValueAsDouble(db.checkAndGetField(field), max[0]) -> {
                    max.add(it)
                    result.clear()
                    result.add(it)
                }

                max.isNotEmpty() &&
                    db
                        .extractValueAsDouble(db.checkAndGetField(field), it)
                        .compareTo(db.extractValueAsDouble(db.checkAndGetField(field), max[0])) == 0 -> result.add(it)
            }
        }

        return result
    }

    private suspend fun avgField(query: DynamoDBQuery): String {
        val hashCode =
            when (query.aggregate!!.groupBy.isEmpty()) {
                true -> 0
                else -> query.aggregate.groupBy.hashCode()
            }
        val aggregateFunction = query.aggregate
        val field = aggregateFunction.field
        val newEtagKeyPostfix = "_" + field + "_AVG"
        val baseEtagKey = query.baseEtagKey()
        val etagKey = baseEtagKey + newEtagKeyPostfix + hashCode
        val cacheKey = baseEtagKey + newEtagKeyPostfix + hashCode
        val groupingParam = query.aggregate.groupBy

        return runCatching {
            cacheManager?.checkAggregationCache(cacheKey, TYPE.simpleName) ?: throw NoSuchElementException()
        }.recoverCatching {
            val projections = mutableSetOf(setOf(field).filterNotNull().toSet())
            val finalProjections = projections.first()

            calculateGroupingPageToken(groupingParam, projections, finalProjections)

            val records =
                db
                    .index(
                        query.copy(
                            autoPaginate = true,
                            projections = finalProjections,
                        ),
                    ).itemList.items

            when {
                records.isEmpty() ->
                    setEtagAndCacheAndReturnContent(
                        etagKey,
                        query.identifiers.hashCode(),
                        cacheKey,
                        JsonObject().put("error", "Empty table!").encode(),
                    )

                else -> {
                    val avg: JsonObject

                    when {
                        query.aggregate.hasGrouping() ->
                            avg = avgGrouping(records, aggregateFunction, field)

                        else -> {
                            avg = JsonObject()

                            records
                                .stream()
                                .mapToDouble { r ->
                                    db.extractValueAsDouble(
                                        db.checkAndGetField(
                                            field!!,
                                        ),
                                        r,
                                    )
                                }.filter { Objects.nonNull(it) }
                                .average()
                                .ifPresent { value -> avg.put("avg", value) }

                            if (avg.size() == 0) {
                                avg.put("avg", 0.0)
                            }
                        }
                    }

                    setEtagAndCacheAndReturnContent(
                        etagKey,
                        query.identifiers.hashCode(),
                        cacheKey,
                        avg.encode(),
                    )
                }
            }
        }.getOrThrow()
    }

    private fun avgGrouping(
        result: List<E>,
        aggregateFunction: AggregateFunction?,
        field: String?,
    ): JsonObject =
        performGroupingAndSorting(
            result,
            aggregateFunction!!,
        ) { items, groupingConfigurations ->
            require(groupingConfigurations.size <= 3) { "GroupBy size of three is max!" }

            val levelOne = groupingConfigurations[0]
            val levelTwo = if (groupingConfigurations.size > 1) groupingConfigurations[1] else null
            val levelThree = if (groupingConfigurations.size > 2) groupingConfigurations[2] else null

            when {
                levelTwo == null ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                averagingDouble {
                                    db.extractValueAsDouble(db.checkAndGetField(field!!), it)
                                },
                            ),
                        )

                levelThree == null ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                groupingBy(
                                    {
                                        calculateGroupingKey<E>(it, levelTwo)
                                    },
                                    averagingDouble {
                                        db.extractValueAsDouble(db.checkAndGetField(field!!), it)
                                    },
                                ),
                            ),
                        )

                else ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                groupingBy(
                                    {
                                        calculateGroupingKey<E>(it, levelTwo)
                                    },
                                    groupingBy(
                                        {
                                            calculateGroupingKey<E>(it, levelThree)
                                        },
                                        summingDouble {
                                            db.extractValueAsDouble(db.checkAndGetField(field!!), it)
                                        },
                                    ),
                                ),
                            ),
                        )
            }
        }

    private suspend fun sumField(query: DynamoDBQuery): String {
        val hashCode =
            when (query.aggregate!!.groupBy.isEmpty()) {
                true -> 0
                else -> query.aggregate.groupBy.hashCode()
            }
        val aggregateFunction = query.aggregate
        val field = aggregateFunction.field
        val newEtagKeyPostfix = "_" + field + "_SUM"
        val baseEtagKey = query.baseEtagKey()
        val etagKey = baseEtagKey + newEtagKeyPostfix + hashCode
        val cacheKey = baseEtagKey + newEtagKeyPostfix + hashCode
        val groupingParam = query.aggregate.groupBy

        return runCatching {
            cacheManager?.checkAggregationCache(cacheKey, TYPE.simpleName) ?: throw NoSuchElementException()
        }.recoverCatching {
            val projections = mutableSetOf(setOf(field).filterNotNull().toSet())
            val finalProjections = projections.first()

            calculateGroupingPageToken(groupingParam, projections, finalProjections)

            val records =
                db
                    .index(
                        query.copy(
                            autoPaginate = true,
                            projections = projections.last(),
                        ),
                    ).itemList.items

            when {
                records.isEmpty() ->
                    setEtagAndCacheAndReturnContent(
                        etagKey,
                        query.identifiers.hashCode(),
                        cacheKey,
                        JsonObject().put("error", "Empty table!").encode(),
                    )

                else -> {
                    val sum =
                        if (aggregateFunction.hasGrouping()) {
                            sumGrouping(records, aggregateFunction, field)
                        } else {
                            JsonObject().put(
                                "sum",
                                records
                                    .stream()
                                    .mapToDouble { r ->
                                        db.extractValueAsDouble(
                                            db.checkAndGetField(
                                                field!!,
                                            ),
                                            r,
                                        )
                                    }.filter { Objects.nonNull(it) }
                                    .sum(),
                            )
                        }

                    setEtagAndCacheAndReturnContent(
                        etagKey,
                        query.identifiers.hashCode(),
                        cacheKey,
                        sum.encode(),
                    )
                }
            }
        }.getOrThrow()
    }

    private fun sumGrouping(
        result: List<E>,
        aggregateFunction: AggregateFunction?,
        field: String?,
    ): JsonObject =
        performGroupingAndSorting(
            result,
            aggregateFunction!!,
        ) { items, groupingConfigurations ->
            require(groupingConfigurations.size <= 3) { "GroupBy size of three is max!" }

            val levelOne = groupingConfigurations[0]
            val levelTwo = if (groupingConfigurations.size > 1) groupingConfigurations[1] else null
            val levelThree = if (groupingConfigurations.size > 2) groupingConfigurations[2] else null

            when {
                levelTwo == null ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                summingDouble {
                                    db.extractValueAsDouble(db.checkAndGetField(field!!), it)
                                },
                            ),
                        )

                levelThree == null ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                groupingBy(
                                    {
                                        calculateGroupingKey<E>(it, levelTwo)
                                    },
                                    summingDouble {
                                        db.extractValueAsDouble(db.checkAndGetField(field!!), it)
                                    },
                                ),
                            ),
                        )

                else ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                groupingBy(
                                    {
                                        calculateGroupingKey<E>(it, levelTwo)
                                    },
                                    groupingBy(
                                        {
                                            calculateGroupingKey<E>(it, levelThree)
                                        },
                                        summingDouble {
                                            db.extractValueAsDouble(db.checkAndGetField(field!!), it)
                                        },
                                    ),
                                ),
                            ),
                        )
            }
        }

    private suspend fun countItems(query: DynamoDBQuery): String {
        val newEtagKeyPostfix = "_COUNT"
        val baseEtagKey = query.baseEtagKey()
        val etagKey = "$baseEtagKey$newEtagKeyPostfix${query.aggregate!!.groupBy.hashCode()}"
        val cacheKey = "$baseEtagKey$newEtagKeyPostfix${query.aggregate.groupBy.hashCode()}"

        return runCatching {
            cacheManager?.checkAggregationCache(cacheKey, TYPE.simpleName) ?: throw NoSuchElementException()
        }.recoverCatching {
            val aggregateFunction = query.aggregate
            val projections =
                when {
                    !aggregateFunction.hasGrouping() -> setOf("etag")
                    else ->
                        aggregateFunction.groupBy
                            .map { it.groupBy }
                            .distinct()
                            .filterNotNull()
                            .toSet()
                }

            val result =
                db
                    .index(
                        query.copy(
                            autoPaginate = true,
                            projections = projections,
                        ),
                    ).itemList.items

            val count =
                when {
                    aggregateFunction.hasGrouping() -> countGrouping(result, aggregateFunction)
                    else -> jsonObjectOf("count" to result.size)
                }

            setEtagAndCacheAndReturnContent(
                etagKey,
                query.identifiers.hashCode(),
                cacheKey,
                count.encode(),
            )
        }.getOrThrow()
    }

    private fun countGrouping(
        result: List<E>,
        aggregateFunction: AggregateFunction?,
    ): JsonObject =
        performGroupingAndSorting(
            result,
            aggregateFunction!!,
        ) { items, groupingConfigurations ->
            require(groupingConfigurations.size <= 3) { "GroupBy size of three is max!" }

            val levelOne = groupingConfigurations[0]
            val levelTwo = if (groupingConfigurations.size > 1) groupingConfigurations[1] else null
            val levelThree = if (groupingConfigurations.size > 2) groupingConfigurations[2] else null

            when {
                levelTwo == null ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy({
                                calculateGroupingKey<E>(it, levelOne)
                            }, counting()),
                        )

                levelThree == null ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                groupingBy({
                                    calculateGroupingKey<E>(it, levelTwo)
                                }, counting()),
                            ),
                        )

                else ->
                    items
                        .parallelStream()
                        .collect(
                            groupingBy(
                                {
                                    calculateGroupingKey<E>(it, levelOne)
                                },
                                groupingBy(
                                    {
                                        calculateGroupingKey<E>(it, levelTwo)
                                    },
                                    groupingBy({
                                        calculateGroupingKey<E>(it, levelThree)
                                    }, counting()),
                                ),
                            ),
                        )
            }
        }

    @Suppress("UNCHECKED_CAST")
    private fun performGroupingAndSorting(
        items: List<E>,
        aggregateFunction: AggregateFunction,
        mappingFunction: BiFunction<List<E>, List<GroupingConfiguration>, Map<String, *>>,
    ): JsonObject {
        val groupingConfigurations = aggregateFunction.groupBy

        require(groupingConfigurations.size <= 3) { "GroupBy size of three is max!" }

        val levelOne = groupingConfigurations[0]
        val levelTwo = if (groupingConfigurations.size > 1) groupingConfigurations[1] else null
        val levelThree = if (groupingConfigurations.size > 2) groupingConfigurations[2] else null
        val collect = mappingFunction.apply(items, groupingConfigurations)

        val funcName = aggregateFunction.function.name.lowercase()

        logger.debug { "Map is: ${encodePrettily(collect)} with size: ${collect.size}" }

        @Suppress("SENSELESS_COMPARISON")
        when {
            collect != null -> {
                val totalGroupCount = collect.size
                val levelOneStream: Map<String, *> =
                    when {
                        levelOne.hasGroupRanging() -> doRangedSorting(collect, levelOne)
                        else -> doNormalSorting(collect, levelOne)
                    }

                when (levelTwo) {
                    null -> return when {
                        levelOne.hasGroupRanging() ->
                            doRangedGrouping(
                                funcName,
                                levelOneStream,
                                levelOne,
                                totalGroupCount,
                            )

                        else -> doNormalGrouping(funcName, levelOneStream, totalGroupCount)
                    }

                    else -> {
                        val levelTwoStream =
                            levelOneStream.entries.stream().map { e ->
                                val entry = e as Map.Entry<String, *>
                                val superGroupedItems = entry.value as Map<String, *>
                                val totalSubGroupCount = superGroupedItems.size

                                when (levelThree) {
                                    null ->
                                        when {
                                            levelTwo.hasGroupRanging() ->
                                                SimpleEntry(
                                                    entry.key,
                                                    doRangedGrouping(
                                                        funcName,
                                                        doRangedSorting(superGroupedItems, levelTwo),
                                                        levelTwo,
                                                        totalSubGroupCount,
                                                    ),
                                                )

                                            else ->
                                                SimpleEntry(
                                                    entry.key,
                                                    doNormalGrouping(
                                                        funcName,
                                                        doNormalSorting(superGroupedItems, levelTwo),
                                                        totalSubGroupCount,
                                                    ),
                                                )
                                        }

                                    else -> {
                                        val levelTwoMap: Map<*, *> =
                                            when {
                                                levelTwo.hasGroupRanging() -> doRangedSorting(superGroupedItems, levelTwo)
                                                else -> doNormalSorting(superGroupedItems, levelTwo)
                                            }

                                        val levelThreeStream =
                                            levelTwoMap.entries.stream().map { subE ->
                                                val subEntry = subE as Map.Entry<String, *>
                                                val subSuperGroupedItems = subEntry.value as Map<String, *>

                                                val totalSubSuperGroupCount = subSuperGroupedItems.size

                                                when {
                                                    levelThree.hasGroupRanging() ->
                                                        SimpleEntry(
                                                            subEntry.key,
                                                            doRangedGrouping(
                                                                funcName,
                                                                doRangedSorting(
                                                                    subSuperGroupedItems,
                                                                    levelThree,
                                                                ),
                                                                levelThree,
                                                                totalSubSuperGroupCount,
                                                            ),
                                                        )

                                                    else ->
                                                        SimpleEntry(
                                                            subEntry.key,
                                                            doNormalGrouping(
                                                                funcName,
                                                                doNormalSorting(
                                                                    subSuperGroupedItems,
                                                                    levelThree,
                                                                ),
                                                                totalSubSuperGroupCount,
                                                            ),
                                                        )
                                                }
                                            }

                                        val levelThreeMap =
                                            levelThreeStream.collect(
                                                toMap(
                                                    { it.key as String },
                                                    { it.value },
                                                    { e1, _ -> e1 },
                                                    { mapOf() },
                                                ),
                                            )

                                        when {
                                            levelTwo.hasGroupRanging() ->
                                                SimpleEntry<Any, JsonObject>(
                                                    entry.key,
                                                    doRangedGrouping(funcName, levelThreeMap, levelTwo, totalSubGroupCount),
                                                )

                                            else ->
                                                SimpleEntry<Any, JsonObject>(
                                                    entry.key,
                                                    doNormalGrouping(funcName, levelThreeMap, totalSubGroupCount),
                                                )
                                        }
                                    }
                                }
                            }

                        val levelTwoMap =
                            levelTwoStream.collect(
                                toMap(
                                    { it.key as String },
                                    { it.value },
                                    { e1, _ -> e1 },
                                    { mapOf() },
                                ),
                            )

                        return when {
                            levelOne.hasGroupRanging() ->
                                doRangedGrouping(funcName, levelTwoMap, levelOne, totalGroupCount)

                            else -> doNormalGrouping(funcName, levelTwoMap, totalGroupCount)
                        }
                    }
                }
            }

            else -> throw InternalError()
        }
    }

    private fun doNormalGrouping(
        aggregationFunctionKey: String,
        mapStream: Map<String, *>,
        totalGroupCount: Int,
    ): JsonObject {
        val results = JsonArray()
        mapStream.forEach { (key, value) ->
            results.add(
                JsonObject()
                    .put("groupByKey", key)
                    .put(aggregationFunctionKey, value),
            )
        }

        return JsonObject()
            .put("totalGroupCount", totalGroupCount)
            .put("count", results.size())
            .put("results", results)
    }

    private fun doRangedGrouping(
        aggregationFunctionKey: String,
        mapStream: Map<String, *>,
        groupingConfiguration: GroupingConfiguration,
        totalGroupCount: Int,
    ): JsonObject {
        val results = JsonArray()
        mapStream.forEach { (key, value) ->
            val rangeObject = JsonObject(key)
            val resultObject =
                JsonObject()
                    .put("floor", rangeObject.getLong("floor"))
                    .put("ceil", rangeObject.getLong("ceil"))
                    .put(aggregationFunctionKey, value)

            results.add(resultObject)
        }

        return JsonObject()
            .put("totalGroupCount", totalGroupCount)
            .put("count", results.size())
            .put(
                "rangeGrouping",
                JsonObject()
                    .put("unit", groupingConfiguration.groupByUnit)
                    .put("range", groupingConfiguration.groupByRange),
            ).put("results", results)
    }

    private fun <T> doNormalSorting(
        collect: Map<String, T>,
        groupingConfiguration: GroupingConfiguration,
    ): Map<String, T> {
        val asc = groupingConfiguration.groupingSortOrder.equals("asc", ignoreCase = true)

        when {
            collect.isEmpty() ->
                return collect.entries
                    .map { e -> SimpleEntry(e.key, e.value) }
                    .fold(LinkedHashMap()) { accumulator, item ->
                        accumulator[item.key] = item.value
                        accumulator
                    }

            else -> {
                val next = collect.entries.iterator().next()
                val isCollection = next.value is Collection<*> || next.value is Map<*, *>
                val keyIsRanged = groupingConfiguration.hasGroupRanging()

                val comp =
                    when {
                        keyIsRanged ->
                            comparing(
                                { e -> JsonObject(e.key) },
                                comparing { keyOne -> keyOne.getLong("floor") },
                            )

                        isCollection -> comparing<SimpleEntry<String, T>, String> { it.key }
                        else -> comparing { Json.encode(it.value) }
                    }

                val sorted =
                    collect.entries
                        .map { e -> SimpleEntry(e.key, e.value) }
                        .sortedWith(if (asc) comp else comp.reversed())

                return when {
                    groupingConfiguration.isFullList ->
                        sorted.fold(LinkedHashMap()) { accumulator, item ->
                            accumulator[item.key] = item.value
                            accumulator
                        }

                    else ->
                        sorted
                            .take(groupingConfiguration.groupingListLimit)
                            .fold(LinkedHashMap()) { accumulator, item ->
                                accumulator[item.key] = item.value
                                accumulator
                            }
                }
            }
        }
    }

    private fun <T> doRangedSorting(
        collect: Map<String, T>,
        groupingConfiguration: GroupingConfiguration,
    ): Map<String, T> {
        val asc = groupingConfiguration.groupingSortOrder.equals("asc", ignoreCase = true)
        val comp = comparingLong<SimpleEntry<String, T>> { e -> JsonObject(e.key).getLong("floor") }

        val sorted =
            collect.entries
                .stream()
                .map { e -> SimpleEntry(e.key, e.value) }
                .sorted(if (asc) comp else comp.reversed())

        return when {
            groupingConfiguration.isFullList ->
                sorted
                    .collect(toList())
                    .fold(LinkedHashMap()) { accumulator, item ->
                        accumulator[item.key] = item.value
                        accumulator
                    }

            else ->
                sorted
                    .limit(groupingConfiguration.groupingListLimit.toLong())
                    .collect(toList())
                    .fold(LinkedHashMap()) { accumulator, item ->
                        accumulator[item.key] = item.value
                        accumulator
                    }
        }
    }

    private fun <T> calculateGroupingKey(
        item: T,
        groupingConfiguration: GroupingConfiguration?,
    ): String {
        val groupingKey: String?

        try {
            groupingKey = db.getFieldAsString(groupingConfiguration!!.groupBy!!, item as Any)
            @Suppress("SENSELESS_COMPARISON")
            if (groupingKey == null) throw UnknownError("Cannot find field!")
        } catch (e: NullPointerException) {
            throw UnknownError("Field is null!")
        }

        when {
            groupingConfiguration.hasGroupRanging() -> {
                val groupByRangeUnit = groupingConfiguration.groupByUnit
                val groupByRangeRange = groupingConfiguration.groupByRange
                var groupingValue: Long? = null
                var rangingValue: Double? = null

                when {
                    groupByRangeUnit!!.equals("INTEGER", ignoreCase = true) -> {
                        groupingValue = java.lang.Long.parseLong(groupByRangeRange!!.toString())
                        val value: Long =
                            try {
                                java.lang.Long.parseLong(groupingKey)
                            } catch (nfe: NumberFormatException) {
                                java.lang.Double
                                    .parseDouble(groupingKey)
                                    .toLong()
                            }

                        rangingValue = ceil((value / groupingValue).toDouble())
                    }

                    groupByRangeUnit.equals("DATE", ignoreCase = true) -> {
                        val date: Date = db.getFieldAsObject(groupingConfiguration.groupBy!!, item as Any)
                        groupingValue = getTimeRangeFromDateUnit(groupByRangeRange!!.toString())

                        rangingValue = ceil((date.time / groupingValue).toDouble())
                    }
                }

                return when {
                    rangingValue != null ->
                        JsonObject()
                            .put("floor", floor(rangingValue).toLong() * groupingValue!!)
                            .put("base", groupingValue)
                            .put("ratio", rangingValue)
                            .put("ceil", (ceil(rangingValue).toLong() + 1L) * groupingValue)
                            .encode()

                    else -> throw UnknownError("Cannot find field!")
                }
            }

            else -> return groupingKey
        }
    }

    private fun getTimeRangeFromDateUnit(groupByRangeRange: String): Long =
        when (AggregateFunction.TIMEUNIT_DATE.valueOf(groupByRangeRange.uppercase())) {
            AggregateFunction.TIMEUNIT_DATE.HOUR -> Duration.ofHours(1).toMillis()
            AggregateFunction.TIMEUNIT_DATE.TWELVE_HOUR -> Duration.ofHours(12).toMillis()
            AggregateFunction.TIMEUNIT_DATE.DAY -> Duration.ofDays(1).toMillis()
            AggregateFunction.TIMEUNIT_DATE.WEEK -> Duration.ofDays(7).toMillis()
            AggregateFunction.TIMEUNIT_DATE.MONTH -> Duration.ofDays(30).toMillis()
            AggregateFunction.TIMEUNIT_DATE.YEAR -> Duration.ofDays(365).toMillis()
        }

    private suspend fun setEtagAndCacheAndReturnContent(
        etagKey: String,
        hash: Int,
        cacheKey: String,
        content: String,
    ): String {
        val etagItemListHashKey = TYPE.simpleName + "_" + hash + "_" + "itemListEtags"
        val newEtag = ModelUtils.returnNewEtag(content.hashCode().toLong())

        return runCatching {
            cacheManager?.replaceAggregationCache(content) { cacheKey }
        }.onFailure {
            logger.error(it) { "Cache failed on agg!" }
        }.getOrThrow()
            .let {
                when {
                    eTagManager != null -> {
                        eTagManager.replaceAggregationEtag(
                            etagItemListHashKey,
                            etagKey,
                            newEtag,
                        )

                        content
                    }

                    else -> content
                }
            }
    }
}
