/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.dynamodb.operators

import com.genghis.tools.repository.dynamodb.DynamoDBRepository
import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.DynamoDBModel
import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.utils.FilterParameter
import com.genghis.tools.repository.utils.OrderByParameter
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.json.DecodeException
import io.vertx.core.json.EncodeException
import io.vertx.core.json.Json.encodePrettily
import io.vertx.core.json.JsonObject
import software.amazon.awssdk.enhanced.dynamodb.Expression
import software.amazon.awssdk.enhanced.dynamodb.Key
import software.amazon.awssdk.enhanced.dynamodb.model.QueryConditional
import software.amazon.awssdk.enhanced.dynamodb.model.QueryEnhancedRequest
import software.amazon.awssdk.enhanced.dynamodb.model.ScanEnhancedRequest
import software.amazon.awssdk.services.dynamodb.model.AttributeValue
import software.amazon.awssdk.services.dynamodb.model.ComparisonOperator
import software.amazon.awssdk.services.dynamodb.model.ComparisonOperator.BEGINS_WITH
import software.amazon.awssdk.services.dynamodb.model.ComparisonOperator.BETWEEN
import software.amazon.awssdk.services.dynamodb.model.ComparisonOperator.EQ
import software.amazon.awssdk.services.dynamodb.model.ComparisonOperator.GE
import software.amazon.awssdk.services.dynamodb.model.ComparisonOperator.GT
import software.amazon.awssdk.services.dynamodb.model.ComparisonOperator.LE
import software.amazon.awssdk.services.dynamodb.model.ComparisonOperator.LT
import java.util.Arrays
import java.util.Base64
import java.util.concurrent.ConcurrentLinkedQueue
import java.util.concurrent.atomic.AtomicInteger
import java.util.stream.Collectors.joining
import java.util.stream.Collectors.toList
import java.util.stream.IntStream

private val logger = KotlinLogging.logger { }

/**
 * This class is used to define the parameters for various operations for the DynamoDBRepository.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
class DynamoDBParameters<E>(
    private val db: DynamoDBRepository<E>,
    private val HASH_IDENTIFIER: String,
    private val IDENTIFIER: String?,
    private val PAGINATION_IDENTIFIER: String?,
) where E : ETagable, E : Cacheable, E : DynamoDBModel, E : Model {
    internal fun applyParameters(
        hashValue: AttributeValue,
        peek: OrderByParameter?,
        params: Map<String, List<FilterParameter>>?,
        builder: QueryEnhancedRequest.Builder,
        gsi: JsonObject?,
    ): QueryEnhancedRequest.Builder {
        logger.debug { encodePrettily(params) }

        params?.let {
            val keyConditionString = arrayOf("")
            val ean = HashMap<String, String>()
            val eav = HashMap<String, AttributeValue?>()
            val paramCount = intArrayOf(0)
            val count = intArrayOf(0)
            val paramSize = params.keys.size

            params.keys
                .mapNotNull {
                    params[it]
                }.forEach { paramList ->
                    val orderCounter = intArrayOf(0)
                    keyConditionString[0] += "("

                    when {
                        paramList.size > 1 &&
                            isRangeKey(peek, paramList[0], gsi) &&
                            !isIllegalRangedKeyQueryParams(paramList) ->
                            buildMultipleRangeKeyCondition(
                                hashValue,
                                builder,
                                paramList,
                            )

                        else -> {
                            paramList.forEach { param ->
                                val fieldName =
                                    when {
                                        param.field!![param.field!!.length - 2] == '_' ->
                                            param.field!!.substring(0, param.field!!.length - 2)
                                        else -> param.field
                                    }
                                if (fieldName != null) ean["#name" + count[0]] = fieldName

                                when {
                                    param.isEq ->
                                        if (isRangeKey(peek, gsi, fieldName, param)) {
                                            buildRangeKeyCondition(
                                                hashValue,
                                                builder,
                                                params.size,
                                                EQ,
                                                db.createAttributeValue(fieldName, param.eq!!.toString()),
                                            )
                                        } else {
                                            val curCount = count[0]++
                                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.eq!!.toString())

                                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                                curCount + " = :val" + curCount
                                        }

                                    param.nonExistant != null -> {
                                        val curCount = count[0]++

                                        when (param.nonExistant!!) {
                                            true -> {
                                                eav[":val$curCount"] = AttributeValue.fromNul(true)
                                                keyConditionString[0] += "${
                                                    if (orderCounter[0] == 0) "" else " ${param.type} "
                                                }attribute_not_exists(#name$curCount) or #name$curCount = :val$curCount"
                                            }
                                            else ->
                                                keyConditionString[0] += (
                                                    if (orderCounter[0] == 0) "" else " ${param.type} "
                                                ) + "attribute_exists(#name$curCount)"
                                        }
                                    }

                                    param.isNe ->
                                        if (isRangeKey(peek, gsi, fieldName, param)) {
                                            buildRangeKeyCondition(
                                                hashValue,
                                                builder,
                                                params.size,
                                                ComparisonOperator.NE,
                                                db.createAttributeValue(fieldName, param.eq!!.toString()),
                                            )
                                        } else {
                                            val curCount = count[0]++
                                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.ne!!.toString())

                                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                                curCount + " <> :val" + curCount
                                        }

                                    param.isBetween -> {
                                        val curCount = count[0]
                                        eav[":val$curCount"] = db.createAttributeValue(fieldName, param.gt!!.toString())
                                        eav[":val" + (curCount + 1)] =
                                            db.createAttributeValue(fieldName, param.lt!!.toString())

                                        keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                            curCount + " > :val" + curCount + " and " +
                                            " #name" + curCount + " < " + " :val" + (curCount + 1)

                                        count[0] = curCount + 2
                                    }

                                    param.isGt ->
                                        if (isRangeKey(peek, gsi, fieldName, param)) {
                                            buildRangeKeyCondition(
                                                hashValue,
                                                builder,
                                                1,
                                                GT,
                                                db.createAttributeValue(fieldName, param.gt!!.toString()),
                                            )
                                        } else {
                                            val curCount = count[0]++
                                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.gt!!.toString())

                                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                                curCount + " > :val" + curCount
                                        }

                                    param.isLt ->
                                        if (isRangeKey(peek, gsi, fieldName, param)) {
                                            buildRangeKeyCondition(
                                                hashValue,
                                                builder,
                                                1,
                                                LT,
                                                db.createAttributeValue(fieldName, param.lt!!.toString()),
                                            )
                                        } else {
                                            val curCount = count[0]++
                                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.lt!!.toString())

                                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                                curCount + " < :val" + curCount
                                        }

                                    param.isInclusiveBetween -> {
                                        val curCount = count[0]
                                        eav[":val$curCount"] = db.createAttributeValue(fieldName, param.ge!!.toString())
                                        eav[":val" + (curCount + 1)] =
                                            db.createAttributeValue(fieldName, param.le!!.toString())

                                        keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                            curCount + " >= :val" + curCount + " and " +
                                            " #name" + curCount + " <= " + " :val" + (curCount + 1)

                                        count[0] = curCount + 2
                                    }

                                    param.isGeLtVariableBetween -> {
                                        val curCount = count[0]
                                        eav[":val$curCount"] = db.createAttributeValue(fieldName, param.ge!!.toString())
                                        eav[":val" + (curCount + 1)] =
                                            db.createAttributeValue(fieldName, param.lt!!.toString())

                                        keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                            curCount + " >= :val" + curCount + " and " +
                                            " #name" + curCount + " < " + " :val" + (curCount + 1)

                                        count[0] = curCount + 2
                                    }

                                    param.isLeGtVariableBetween -> {
                                        val curCount = count[0]
                                        eav[":val$curCount"] = db.createAttributeValue(fieldName, param.le!!.toString())
                                        eav[":val" + (curCount + 1)] =
                                            db.createAttributeValue(fieldName, param.gt!!.toString())

                                        keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                            curCount + " <= :val" + curCount + " and " +
                                            " #name" + curCount + " > " + " :val" + (curCount + 1)

                                        count[0] = curCount + 2
                                    }

                                    param.isGe ->
                                        if (isRangeKey(peek, gsi, fieldName, param)) {
                                            buildRangeKeyCondition(
                                                hashValue,
                                                builder,
                                                1,
                                                GE,
                                                db.createAttributeValue(fieldName, param.ge!!.toString()),
                                            )
                                        } else {
                                            val curCount = count[0]++
                                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.ge!!.toString())

                                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                                curCount + " >= :val" + curCount
                                        }

                                    param.isLe ->
                                        if (isRangeKey(peek, gsi, fieldName, param)) {
                                            buildRangeKeyCondition(
                                                hashValue,
                                                builder,
                                                1,
                                                LE,
                                                db.createAttributeValue(fieldName, param.le!!.toString()),
                                            )
                                        } else {
                                            val curCount = count[0]++
                                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.le!!.toString())

                                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                                curCount + " <= :val" + curCount
                                        }

                                    param.isContains -> {
                                        val curCount = count[0]++
                                        eav[":val$curCount"] =
                                            db.createAttributeValue(fieldName, param.contains!!.toString())

                                        keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") +
                                            "contains(" + "#name" + curCount + ", :val" + curCount + ")"
                                    }

                                    param.isNotContains -> {
                                        val curCount = count[0]++
                                        eav[":val$curCount"] =
                                            db.createAttributeValue(fieldName, param.notContains!!.toString())

                                        keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") +
                                            "not contains(" + "#name" + curCount + ", :val" + curCount + ")"
                                    }

                                    param.isBeginsWith ->
                                        if (isRangeKey(peek, gsi, fieldName, param)) {
                                            buildRangeKeyCondition(
                                                hashValue,
                                                builder,
                                                params.size,
                                                BEGINS_WITH,
                                                db.createAttributeValue(fieldName, param.beginsWith!!.toString()),
                                            )
                                        } else {
                                            val curCount = count[0]++
                                            eav[":val$curCount"] =
                                                db.createAttributeValue(fieldName, param.beginsWith!!.toString())

                                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") +
                                                "begins_with(" + "#name" + curCount + ", :val" + curCount + ")"
                                        }

                                    param.isIn -> {
                                        val inList = inQueryToStringChain(fieldName!!, param.inParam)

                                        val keys = ConcurrentLinkedQueue<String>()
                                        val valCounter = AtomicInteger()
                                        inList.l().forEach { av ->
                                            val currentValKey = ":inVal" + valCounter.getAndIncrement()
                                            keys.add(currentValKey)
                                            eav[currentValKey] = av
                                        }

                                        val curCount = count[0]++

                                        keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                            curCount + " IN (" + keys.stream().collect(joining(", ")) + ")"
                                    }
                                }

                                orderCounter[0]++
                            }

                            when {
                                keyConditionString[0].endsWith("(") ->
                                    when {
                                        keyConditionString[0].equals("(", ignoreCase = true) -> keyConditionString[0] = ""
                                        else ->
                                            keyConditionString[0] =
                                                keyConditionString[0].substring(0, keyConditionString[0].lastIndexOf(")"))
                                    }

                                else -> {
                                    keyConditionString[0] += ")"

                                    if (paramSize > 1 && paramCount[0] < paramSize - 1) {
                                        keyConditionString[0] += " AND "
                                    }
                                }
                            }

                            paramCount[0]++
                        }
                    }
                }

            if (ean.size > 0 && eav.size > 0) {
                builder.filterExpression(
                    Expression
                        .builder()
                        .expression(keyConditionString[0])
                        .expressionNames(ean)
                        .expressionValues(eav)
                        .build(),
                )
            }

            logger.debug { "NAMES: ${encodePrettily(ean)}" }

            if (logger.isDebugEnabled()) {
                logger.debug {
                    "Values: ${
                        encodePrettily(
                            eav.map {
                                it.key to it.value?.toString()
                            },
                        )
                    }"
                }
            }
        }

        return builder
    }

    private fun inQueryToStringChain(
        fieldName: String,
        `in`: Array<Any>?,
    ): AttributeValue =
        AttributeValue.fromL(
            Arrays
                .stream(`in`!!)
                .map { o -> db.createAttributeValue(fieldName, o.toString()) }
                .collect(toList())
                .toList(),
        )

    private fun isRangeKey(
        peek: OrderByParameter?,
        param: FilterParameter,
        gsi: JsonObject?,
    ): Boolean =
        isRangeKey(
            peek,
            gsi,
            if (param.field!![param.field!!.length - 2] == '_') {
                param.field!!.substring(0, param.field!!.length - 2)
            } else {
                param.field
            },
            param,
        )

    private fun isRangeKey(
        peek: OrderByParameter?,
        gsi: JsonObject?,
        fieldName: String?,
        param: FilterParameter,
    ): Boolean =
        when {
            gsi != null -> {
                val gsiRange = gsi.getString("range", "NONE")

                when {
                    gsiRange == "NONE" -> false
                    peek != null ->
                        peek.field.equals(
                            gsiRange!!,
                            ignoreCase = true,
                        )
                    else -> param.field!!.equals(gsiRange!!, ignoreCase = true)
                }
            }
            else ->
                peek != null &&
                    peek.field.equals(
                        fieldName!!,
                        ignoreCase = true,
                    ) ||
                    peek == null &&
                    param.field!!.equals(PAGINATION_IDENTIFIER!!, ignoreCase = true)
        }

    internal fun applyParameters(params: Map<String, List<FilterParameter>>?): ScanEnhancedRequest.Builder {
        val filterExpression = ScanEnhancedRequest.builder()

        if (params != null) {
            logger.debug { encodePrettily(params) }

            val keyConditionString = arrayOf("")
            val ean = HashMap<String, String>()
            val eav = HashMap<String, AttributeValue?>()
            val paramCount = intArrayOf(0)
            val count = intArrayOf(0)
            val paramSize = params.keys.size

            params.keys.stream().map<List<FilterParameter>> { params[it] }.forEach { paramList ->
                keyConditionString[0] += "("
                val orderCounter = intArrayOf(0)

                paramList.forEach { param ->
                    val fieldName =
                        if (param.field!![param.field!!.length - 2] == '_') {
                            param.field!!.substring(0, param.field!!.length - 2)
                        } else {
                            param.field
                        }
                    if (fieldName != null) ean["#name" + count[0]] = fieldName

                    when {
                        param.isEq -> {
                            val curCount = count[0]++
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.eq!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " = :val" + curCount
                        }

                        param.nonExistant != null -> {
                            val curCount = count[0]++

                            when (param.nonExistant!!) {
                                true -> {
                                    eav[":val$curCount"] = AttributeValue.fromNul(true)
                                    keyConditionString[0] += "${
                                        if (orderCounter[0] == 0) "" else " ${param.type} "
                                    }attribute_not_exists(#name$curCount) or #name$curCount = :val$curCount"
                                }
                                else ->
                                    keyConditionString[0] += (
                                        if (orderCounter[0] == 0) "" else " ${param.type} "
                                    ) + "attribute_exists(#name$curCount)"
                            }
                        }

                        param.isNe -> {
                            val curCount = count[0]++
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.ne!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " <> :val" + curCount
                        }

                        param.isBetween -> {
                            val curCount = count[0]
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.gt!!.toString())
                            eav[":val" + (curCount + 1)] = db.createAttributeValue(fieldName, param.lt!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " > :val" + curCount + " and " +
                                " #name" + curCount + " < " + " :val" + (curCount + 1)

                            count[0] = curCount + 2
                        }

                        param.isGt -> {
                            val curCount = count[0]++
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.gt!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " > :val" + curCount
                        }

                        param.isLt -> {
                            val curCount = count[0]++
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.lt!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " < :val" + curCount
                        }

                        param.isInclusiveBetween -> {
                            val curCount = count[0]
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.ge!!.toString())
                            eav[":val" + (curCount + 1)] = db.createAttributeValue(fieldName, param.le!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " >= :val" + curCount + " and " +
                                " #name" + curCount + " <= " + " :val" + (curCount + 1)

                            count[0] = curCount + 2
                        }

                        param.isGeLtVariableBetween -> {
                            val curCount = count[0]
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.ge!!.toString())
                            eav[":val" + (curCount + 1)] = db.createAttributeValue(fieldName, param.lt!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " >= :val" + curCount + " and " +
                                " #name" + curCount + " < " + " :val" + (curCount + 1)

                            count[0] = curCount + 2
                        }

                        param.isLeGtVariableBetween -> {
                            val curCount = count[0]
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.le!!.toString())
                            eav[":val" + (curCount + 1)] = db.createAttributeValue(fieldName, param.gt!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " <= :val" + curCount + " and " +
                                " #name" + curCount + " > " + " :val" + (curCount + 1)

                            count[0] = curCount + 2
                        }

                        param.isGe -> {
                            val curCount = count[0]++
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.ge!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " >= :val" + curCount
                        }

                        param.isLe -> {
                            val curCount = count[0]++
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.le!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " <= :val" + curCount
                        }

                        param.isContains -> {
                            val curCount = count[0]++
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.contains!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") +
                                "contains(" + "#name" + curCount + ", :val" + curCount + ")"
                        }

                        param.isNotContains -> {
                            val curCount = count[0]++
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.notContains!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") +
                                "not contains(" + "#name" + curCount + ", :val" + curCount + ")"
                        }

                        param.isBeginsWith -> {
                            val curCount = count[0]++
                            eav[":val$curCount"] = db.createAttributeValue(fieldName, param.beginsWith!!.toString())

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") +
                                "begins_with(" + "#name" + curCount + ", :val" + curCount + ")"
                        }

                        param.isIn -> {
                            val inList = inQueryToStringChain(fieldName!!, param.inParam)

                            val keys = ConcurrentLinkedQueue<String>()
                            val valCounter = AtomicInteger()
                            inList.l().forEach { av ->
                                val currentValKey = ":inVal" + valCounter.getAndIncrement()
                                keys.add(currentValKey)
                                eav[currentValKey] = av
                            }

                            val curCount = count[0]++

                            keyConditionString[0] += (if (orderCounter[0] == 0) "" else " " + param.type + " ") + "#name" +
                                curCount + " IN (" + keys.stream().collect(joining(", ")) + ")"
                        }
                    }

                    orderCounter[0]++
                }

                keyConditionString[0] += ")"

                if (paramSize > 1 && paramCount[0] < paramSize - 1) {
                    keyConditionString[0] += " AND "
                }

                paramCount[0]++
            }

            if (ean.size > 0 && eav.size > 0) {
                filterExpression.filterExpression(
                    Expression
                        .builder()
                        .expression(keyConditionString[0])
                        .expressionNames(ean)
                        .expressionValues(eav)
                        .build(),
                )
            }

            if (logger.isDebugEnabled()) {
                logger.debug { "SCAN EXPRESSION IS: ${encodePrettily(keyConditionString)}" }
            }
            if (logger.isDebugEnabled()) {
                logger.debug { "NAMES: ${encodePrettily(ean)}" }
            }
            if (logger.isDebugEnabled()) {
                logger.debug {
                    "Values: ${
                        encodePrettily(
                            eav.map {
                                it.key to it.value?.toString()
                            },
                        )
                    }"
                }
            }
        }

        return filterExpression
    }

    private fun buildRangeKeyCondition(
        hashValue: AttributeValue,
        requestBuilder: QueryEnhancedRequest.Builder,
        count: Int,
        comparator: ComparisonOperator,
        vararg attributeValues: AttributeValue,
    ) {
        if (count > 1) {
            require(comparator == BETWEEN) {
                "Between is only valid argument for multicount!"
            }

            requestBuilder
                .queryConditional(
                    QueryConditional
                        .sortBetween(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(attributeValues.first())
                                .build(),
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(attributeValues.last())
                                .build(),
                        ),
                ).build()
        } else {
            when (comparator) {
                GE ->
                    requestBuilder.queryConditional(
                        QueryConditional.sortGreaterThanOrEqualTo(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(attributeValues.first())
                                .build(),
                        ),
                    )
                GT ->
                    requestBuilder.queryConditional(
                        QueryConditional.sortGreaterThan(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(attributeValues.first())
                                .build(),
                        ),
                    )
                LE ->
                    requestBuilder.queryConditional(
                        QueryConditional.sortLessThanOrEqualTo(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(attributeValues.first())
                                .build(),
                        ),
                    )
                LT ->
                    requestBuilder.queryConditional(
                        QueryConditional.sortLessThan(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(attributeValues.first())
                                .build(),
                        ),
                    )
                BEGINS_WITH ->
                    requestBuilder.queryConditional(
                        QueryConditional.sortBeginsWith(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(attributeValues.first())
                                .build(),
                        ),
                    )
                else -> throw IllegalArgumentException("Cannot build rangekey with this")
            }.build()
        }
    }

    private fun buildMultipleRangeKeyCondition(
        hashValue: AttributeValue,
        filterExpression: QueryEnhancedRequest.Builder,
        paramList: List<FilterParameter>,
    ) {
        if (paramList.size > 2) throw IllegalArgumentException("Cannot query on more than two params on a range key!")

        val paramOne = paramList[0]
        val paramTwo = paramList[1]

        when {
            paramOne.isGt && paramTwo.isLt -> {
                val paramOneAsString = paramList[0].gt.toString()
                val paramTwoAsString = paramList[1].lt.toString()
                val paramOneAsLong = paramOneAsString.toLongOrNull()
                val paramTwoAsLong = paramTwoAsString.toLongOrNull()

                filterExpression.queryConditional(
                    QueryConditional
                        .sortBetween(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramOneAsLong?.let {
                                        AttributeValue.fromN((it + 1L).toString())
                                    } ?: AttributeValue.fromS(paramOneAsString),
                                ).build(),
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramTwoAsLong?.let {
                                        AttributeValue.fromN((it - 1L).toString())
                                    } ?: AttributeValue.fromS(paramTwoAsString),
                                ).build(),
                        ),
                )
            }

            paramOne.isLt && paramTwo.isGt -> {
                val paramOneAsString = paramList[0].lt.toString()
                val paramTwoAsString = paramList[1].gt.toString()
                val paramOneAsLong = paramOneAsString.toLongOrNull()
                val paramTwoAsLong = paramTwoAsString.toLongOrNull()

                filterExpression.queryConditional(
                    QueryConditional
                        .sortBetween(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramTwoAsLong?.let {
                                        AttributeValue.fromN((it + 1L).toString())
                                    } ?: AttributeValue.fromS(paramTwoAsString),
                                ).build(),
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramOneAsLong?.let {
                                        AttributeValue.fromN((it - 1L).toString())
                                    } ?: AttributeValue.fromS(paramOneAsString),
                                ).build(),
                        ),
                )
            }

            paramOne.isLe && paramTwo.isGe -> {
                val paramOneAsString = paramList[0].le.toString()
                val paramTwoAsString = paramList[1].ge.toString()
                val paramOneAsLong = paramOneAsString.toLongOrNull()
                val paramTwoAsLong = paramTwoAsString.toLongOrNull()

                filterExpression.queryConditional(
                    QueryConditional
                        .sortBetween(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramTwoAsLong?.let {
                                        AttributeValue.fromN(it.toString())
                                    } ?: AttributeValue.fromS(paramTwoAsString),
                                ).build(),
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramOneAsLong?.let {
                                        AttributeValue.fromN(it.toString())
                                    } ?: AttributeValue.fromS(paramOneAsString),
                                ).build(),
                        ),
                )
            }

            paramOne.isGe && paramTwo.isLe -> {
                val paramOneAsString = paramList[0].ge.toString()
                val paramTwoAsString = paramList[1].le.toString()
                val paramOneAsLong = paramOneAsString.toLongOrNull()
                val paramTwoAsLong = paramTwoAsString.toLongOrNull()

                filterExpression.queryConditional(
                    QueryConditional
                        .sortBetween(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramOneAsLong?.let {
                                        AttributeValue.fromN(it.toString())
                                    } ?: AttributeValue.fromS(paramOneAsString),
                                ).build(),
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramTwoAsLong?.let {
                                        AttributeValue.fromN(it.toString())
                                    } ?: AttributeValue.fromS(paramTwoAsString),
                                ).build(),
                        ),
                )
            }

            paramOne.isGe && paramTwo.isLt -> {
                val paramOneAsString = paramList[0].ge.toString()
                val paramTwoAsString = paramList[1].lt.toString()
                val paramOneAsLong = paramOneAsString.toLongOrNull()
                val paramTwoAsLong = paramTwoAsString.toLongOrNull()

                filterExpression.queryConditional(
                    QueryConditional
                        .sortBetween(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramOneAsLong?.let {
                                        AttributeValue.fromN(it.toString())
                                    } ?: AttributeValue.fromS(paramOneAsString),
                                ).build(),
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramTwoAsLong?.let {
                                        AttributeValue.fromN((it - 1L).toString())
                                    } ?: AttributeValue.fromS(paramTwoAsString),
                                ).build(),
                        ),
                )
            }

            paramOne.isLt && paramTwo.isGe -> {
                val paramOneAsString = paramList[0].lt.toString()
                val paramTwoAsString = paramList[1].ge.toString()
                val paramOneAsLong = paramOneAsString.toLongOrNull()
                val paramTwoAsLong = paramTwoAsString.toLongOrNull()

                filterExpression.queryConditional(
                    QueryConditional
                        .sortBetween(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramTwoAsLong?.let {
                                        AttributeValue.fromN(it.toString())
                                    } ?: AttributeValue.fromS(paramTwoAsString),
                                ).build(),
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramOneAsLong?.let {
                                        AttributeValue.fromN((it - 1L).toString())
                                    } ?: AttributeValue.fromS(paramOneAsString),
                                ).build(),
                        ),
                )
            }

            paramOne.isLe && paramTwo.isGt -> {
                val paramOneAsString = paramList[0].le.toString()
                val paramTwoAsString = paramList[1].gt.toString()
                val paramOneAsLong = paramOneAsString.toLongOrNull()
                val paramTwoAsLong = paramTwoAsString.toLongOrNull()

                filterExpression.queryConditional(
                    QueryConditional
                        .sortBetween(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramTwoAsLong?.let {
                                        AttributeValue.fromN((it + 1L).toString())
                                    } ?: AttributeValue.fromS(paramTwoAsString),
                                ).build(),
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramOneAsLong?.let {
                                        AttributeValue.fromN(it.toString())
                                    } ?: AttributeValue.fromS(paramOneAsString),
                                ).build(),
                        ),
                )
            }

            paramOne.isGt && paramTwo.isLe -> {
                val paramOneAsString = paramList[0].gt.toString()
                val paramTwoAsString = paramList[1].le.toString()
                val paramOneAsLong = paramOneAsString.toLongOrNull()
                val paramTwoAsLong = paramTwoAsString.toLongOrNull()

                filterExpression.queryConditional(
                    QueryConditional
                        .sortBetween(
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramOneAsLong?.let {
                                        AttributeValue.fromN((it + 1L).toString())
                                    } ?: AttributeValue.fromS(paramOneAsString),
                                ).build(),
                            Key
                                .builder()
                                .partitionValue(hashValue)
                                .sortValue(
                                    paramTwoAsLong?.let {
                                        AttributeValue.fromN(it.toString())
                                    } ?: AttributeValue.fromS(paramTwoAsString),
                                ).build(),
                        ),
                )
            }

            else -> throw IllegalArgumentException("This is an invalid query!")
        }
    }

    internal fun applyOrderBy(
        orderBy: OrderByParameter?,
        queryBuilder: QueryEnhancedRequest.Builder,
    ): QueryEnhancedRequest.Builder {
        when {
            orderBy == null -> queryBuilder.scanIndexForward(false)
            else -> queryBuilder.scanIndexForward(orderBy.isAsc)
        }

        return queryBuilder
    }

    internal fun isIllegalRangedKeyQueryParams(nameParams: List<FilterParameter>): Boolean =
        nameParams
            .stream()
            .anyMatch { it.isIllegalRangedKeyParam }

    internal fun buildProjections(
        projections: Set<String>,
        indexName: String?,
    ): Set<String> {
        var newProjections = projections.toMutableSet()
        if (newProjections.isEmpty()) return newProjections
        val finalProjections = newProjections

        if (finalProjections.none { p -> p.equals(HASH_IDENTIFIER, ignoreCase = true) }) {
            val newProjectionArray = arrayOfNulls<String>(finalProjections.size + 1)
            IntStream.range(0, finalProjections.size).forEach { i ->
                newProjectionArray[i] = finalProjections.elementAt(i)
            }
            newProjectionArray[finalProjections.size] = HASH_IDENTIFIER
            newProjections = newProjectionArray.filterNotNull().toMutableSet()
        }

        val finalProjections2 = newProjections

        if (db.hasRangeKey()) {
            if (finalProjections2.none { p -> p.equals(IDENTIFIER, ignoreCase = true) }) {
                val newProjectionArray = arrayOfNulls<String>(finalProjections2.size + 1)
                IntStream.range(0, finalProjections2.size).forEach { i ->
                    newProjectionArray[i] = finalProjections2.elementAt(i)
                }
                newProjectionArray[finalProjections2.size] = IDENTIFIER
                newProjections = newProjectionArray.filterNotNull().toMutableSet()
            }
        }

        val finalProjections3 = newProjections

        if (finalProjections3.none { p -> p.equals(indexName, ignoreCase = true) }) {
            val newProjectionArray = arrayOfNulls<String>(finalProjections3.size + 1)
            IntStream.range(0, finalProjections3.size).forEach { i ->
                newProjectionArray[i] = finalProjections3.elementAt(i)
            }
            newProjectionArray[finalProjections3.size] = indexName
            newProjections = newProjectionArray.filterNotNull().toMutableSet()
        }

        return newProjections.toSet()
    }

    internal fun createPageTokenMap(encodedPageToken: String?): MutableMap<String, AttributeValue>? {
        var pageTokenMap: MutableMap<String, AttributeValue>? = null
        var pageToken: JsonObject? = null
        var splitToken: List<String>? = null

        if (encodedPageToken != null) {
            if (encodedPageToken != "END_OF_LIST") {
                val pageTokens = String(Base64.getUrlDecoder().decode(encodedPageToken))
                splitToken = pageTokens.split(":".toRegex(), 3)

                pageToken =
                    try {
                        val pageTokenJson = JsonObject(String(Base64.getUrlDecoder().decode(splitToken[1])))

                        splitToken.getOrNull(2)?.let {
                            pageTokenJson.put("totalScannedCountForAllPages", it.toInt())
                        }

                        pageTokenJson
                    } catch (e: EncodeException) {
                        null
                    } catch (e: DecodeException) {
                        null
                    }
            }
        }

        if (pageToken != null) {
            pageTokenMap = HashMap()

            pageToken.map.filter { it.key != "totalScannedCountForAllPages" }.forEach { (k, v) ->
                val fieldName =
                    db
                        .getField(k)
                        .type.simpleName
                        .lowercase()

                logger.debug { "$k : $fieldName" }

                pageTokenMap[k] =
                    when {
                        fieldName == "long" || fieldName == "klong" -> AttributeValue.fromN(v.toString())
                        fieldName == "int" || fieldName == "kint" -> AttributeValue.fromN(v.toString())
                        else -> AttributeValue.fromS(v.toString())
                    }
            }

            pageTokenMap["totalScannedCountForAllPages"] =
                AttributeValue.fromN(
                    pageToken.getInteger("totalScannedCountForAllPages").toString(),
                )
        }

        if (pageTokenMap != null) {
            splitToken?.let {
                pageTokenMap["oldPageToken"] = AttributeValue.fromS(splitToken[0])
            }
        }

        logger.debug {
            "PageTokenMap is: ${
                encodePrettily(
                    pageTokenMap?.map {
                        "${it.key} : ${
                            when (it.value.type()) {
                                AttributeValue.Type.N -> "${it.value.n().toLong()} as N"
                                else -> it.value.s()
                            }
                        }"
                    },
                )
            }"
        }

        return pageTokenMap
    }

    internal fun createNewPageToken(
        lastEvaluatedKey: Map<String, AttributeValue>,
        totalScannedCountForAllPages: Int,
    ): String {
        logger.debug {
            "Last key is: ${
                encodePrettily(
                    lastEvaluatedKey.map {
                        "${it.key} : ${
                            when (it.value.type()) {
                                AttributeValue.Type.N -> "${it.value.n().toLong()} as N"
                                else -> it.value.s()
                            }
                        }"
                    },
                )
            }"
        }

        val pageToken = JsonObject()
        lastEvaluatedKey.forEach { (k, v) ->
            pageToken.put(
                k,
                when {
                    v.s() != null -> v.s()
                    v.n() != null -> v.n()
                    v.b() != null -> v.b()
                    else -> v.s()
                },
            )
        }
        pageToken.put("totalScannedCountForAllPages", totalScannedCountForAllPages)

        logger.debug { "PageToken is: ${pageToken.encodePrettily()}" }

        return Base64.getUrlEncoder().encodeToString(pageToken.encode().toByteArray())
    }
}
