/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

@file:Suppress("UNCHECKED_CAST")

package com.genghis.tools.repository.dynamodb.operators

import com.genghis.tools.repository.dynamodb.DynamoDBRepository
import com.genghis.tools.repository.dynamodb.DynamoDBRepository.Companion.PAGINATION_INDEX
import com.genghis.tools.repository.dynamodb.hash
import com.genghis.tools.repository.dynamodb.range
import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.DynamoDBModel
import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.repository.cache.CacheManager
import com.genghis.tools.repository.repository.etag.ETagManager
import com.genghis.tools.repository.repository.results.ItemListResult
import com.genghis.tools.repository.repository.results.ItemResult
import com.genghis.tools.repository.utils.DynamoDBQuery
import com.genghis.tools.repository.utils.OrderByDirection.ASC
import com.genghis.tools.repository.utils.OrderByDirection.DESC
import com.genghis.tools.repository.utils.PageTokens
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.AsyncResult
import io.vertx.core.Future
import io.vertx.core.Future.failedFuture
import io.vertx.core.Future.succeededFuture
import io.vertx.core.Handler
import io.vertx.core.Promise
import io.vertx.core.json.Json.encode
import io.vertx.core.json.Json.encodePrettily
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.core.json.jsonObjectOf
import io.vertx.kotlin.coroutines.awaitResult
import io.vertx.kotlin.coroutines.coAwait
import io.vertx.serviceproxy.ServiceException
import kotlinx.coroutines.delay
import software.amazon.awssdk.enhanced.dynamodb.DynamoDbAsyncIndex
import software.amazon.awssdk.enhanced.dynamodb.DynamoDbAsyncTable
import software.amazon.awssdk.enhanced.dynamodb.Expression
import software.amazon.awssdk.enhanced.dynamodb.Key
import software.amazon.awssdk.enhanced.dynamodb.internal.client.DefaultDynamoDbAsyncTable
import software.amazon.awssdk.enhanced.dynamodb.model.GetItemEnhancedRequest
import software.amazon.awssdk.enhanced.dynamodb.model.Page
import software.amazon.awssdk.enhanced.dynamodb.model.QueryConditional
import software.amazon.awssdk.enhanced.dynamodb.model.QueryEnhancedRequest
import software.amazon.awssdk.enhanced.dynamodb.model.ReadBatch
import software.amazon.awssdk.enhanced.dynamodb.model.ScanEnhancedRequest
import software.amazon.awssdk.services.dynamodb.model.AttributeValue
import software.amazon.awssdk.services.dynamodb.model.AttributeValue.fromN
import software.amazon.awssdk.services.dynamodb.model.AttributeValue.fromS
import java.lang.reflect.Method
import java.util.Base64
import java.util.concurrent.atomic.AtomicLong
import java.util.function.Function
import kotlin.math.ceil
import kotlin.math.min

private val logger = KotlinLogging.logger { }

/**
 * This class defines the read operations for the DynamoDBRepository.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
class DynamoDBReader<E>(
    private val TYPE: Class<E>,
    private val db: DynamoDBRepository<E>,
    private val HASH_IDENTIFIER: String,
    private val HASH_FIELD: Method?,
    private val IDENTIFIER: String?,
    private val IDENTIFIER_FIELD: Method?,
    private val PAGINATION_IDENTIFIER: String?,
    private val GSI_KEY_MAP: Map<String, JsonObject>,
    private val dbParams: DynamoDBParameters<E>,
    private val cacheManager: CacheManager<E>?,
    private val etagManager: ETagManager<E>?,
)
    where E : ETagable, E : Cacheable, E : DynamoDBModel, E : Model {
    private val mapper: DynamoDbAsyncTable<E> = db.dynamoDbTable

    private val paginationIndex: String?
        get() = if (PAGINATION_IDENTIFIER != null && PAGINATION_IDENTIFIER != "") PAGINATION_INDEX else null

    suspend fun show(query: DynamoDBQuery): ItemResult<E> {
        val startTime = AtomicLong()
        startTime.set(System.nanoTime())

        val preOperationTime = AtomicLong()
        val operationTime = AtomicLong()
        val postOperationTime = AtomicLong()

        require(query.identifiers.hash != null) {
            "Must supply hash value for read"
        }

        val hash = query.identifiers.hash
        val range = query.identifiers.ranges.firstOrNull()
        val projections = query.projections
        val cacheId =
            TYPE.simpleName + "_" + hash + (if (range == null) "" else "/$range") +
                if (projections.isNotEmpty()) "/projection/" + projections.hashCode() else ""

        return runCatching {
            cacheManager?.checkObjectCache(cacheId, TYPE.simpleName) ?: throw NoSuchElementException()
        }.map {
            val result = ItemResult(it, true)

            if (logger.isDebugEnabled()) {
                logger.debug { "Served cached version of: $cacheId" }
            }

            result
        }.recoverCatching {
            runCatching {
                runCatching {
                    awaitResult<E?> {
                        fetchItem(
                            startTime = startTime,
                            preOperationTime = preOperationTime,
                            operationTime = operationTime,
                            hash = hash,
                            range = range,
                            GSI = query.identifiers.gsi,
                            consistent = query.consistent,
                            resultHandler = it,
                        )
                    } ?: throw NoSuchElementException("${db.modelName}: $hash - $range not found!")
                }.getOrElse {
                    query.identifiers.gsi?.let {
                        delay(2000)

                        awaitResult {
                            fetchItem(
                                startTime = startTime,
                                preOperationTime = preOperationTime,
                                operationTime = operationTime,
                                hash = hash,
                                range = range,
                                GSI = query.identifiers.gsi,
                                consistent = query.consistent,
                                resultHandler = it,
                            )
                        }
                    } ?: throw NoSuchElementException("${db.modelName}: $hash - $range not found on second query!")
                }
            }.recoverCatching {
                when (it) {
                    is NoSuchElementException -> {
                        postOperationTime.set(System.nanoTime() - startTime.get())

                        throw it
                    }
                    else -> {
                        logger.error(it) { "Error in read!" }

                        postOperationTime.set(System.nanoTime() - startTime.get())

                        throw ServiceException(
                            500,
                            jsonObjectOf("error" to it.message).encodePrettily(),
                            jsonObjectOf("error" to it.message),
                        )
                    }
                }
            }.mapCatching { item ->
                val cacheFuture = Promise.promise<E>()

                when {
                    cacheManager?.isObjectCacheAvailable() == true ->
                        cacheManager.replaceObjectCache(
                            cacheId = cacheId,
                            item = item,
                            future = cacheFuture,
                            projections = projections,
                        )

                    else -> cacheFuture.tryComplete(item)
                }

                runCatching {
                    cacheFuture.future().coAwait()
                }.mapCatching {
                    postOperationTime.set(System.nanoTime() - startTime.get())

                    returnTimedResult(
                        readResult = it,
                        preOperationTime = preOperationTime,
                        operationTime = operationTime,
                        postOperationTime = postOperationTime,
                    )
                }.getOrThrow()
            }.getOrThrow()
        }.getOrThrow()
    }

    private fun fetchItem(
        startTime: AtomicLong,
        preOperationTime: AtomicLong,
        operationTime: AtomicLong,
        hash: String,
        range: String?,
        GSI: String?,
        consistent: Boolean,
        resultHandler: Handler<AsyncResult<E?>>,
    ) = runCatching {
        when {
            db.hasRangeKey() ->
                when {
                    range == null && GSI != null -> {
                        if (logger.isDebugEnabled()) {
                            logger.debug { "Loading ranged item without range key!" }
                        }

                        fetchHashItem(
                            hash = hash,
                            startTime = startTime,
                            preOperationTime = preOperationTime,
                            operationTime = operationTime,
                            GSI = GSI,
                            consistent = consistent,
                            resultHandler = resultHandler,
                        )
                    }

                    else ->
                        when {
                            range.isNullOrBlank() -> resultHandler.handle(failedFuture(NoSuchElementException()))
                            else ->
                                fetchHashAndRangeItem(
                                    hash = hash,
                                    range = range,
                                    startTime = startTime,
                                    preOperationTime = preOperationTime,
                                    operationTime = operationTime,
                                    GSI = GSI,
                                    consistent = consistent,
                                    resultHandler = resultHandler,
                                )
                        }
                }

            else ->
                fetchHashItem(
                    hash = hash,
                    startTime = startTime,
                    preOperationTime = preOperationTime,
                    operationTime = operationTime,
                    GSI = GSI,
                    consistent = consistent,
                    resultHandler = resultHandler,
                )
        }
    }.onFailure {
        logger.error {
            "$it : ${it.message} : ${it.stackTrace.joinToString {
                "${it.fileName}:${it.lineNumber}\n"
            }}"
        }

        resultHandler.handle(failedFuture(it))
    }.getOrElse { resultHandler.handle(succeededFuture(null)) }

    private fun fetchHashAndRangeItem(
        hash: String,
        range: String,
        startTime: AtomicLong,
        preOperationTime: AtomicLong,
        operationTime: AtomicLong,
        consistent: Boolean,
        GSI: String?,
        resultHandler: Handler<AsyncResult<E?>>,
    ) {
        val timeBefore = System.currentTimeMillis()

        preOperationTime.set(System.nanoTime() - startTime.get())

        val builder = Key.builder().partitionValue(hash)

        @Suppress("UNCHECKED_CAST")
        when (val indexMapper = GSI?.let { mapper.index(GSI) } ?: mapper) {
            is DefaultDynamoDbAsyncTable<*>,
            is DynamoDbAsyncTable<*>,
            -> {
                if (db.hasRangeKey()) {
                    range.toLongOrNull()?.let {
                        builder.sortValue(it)
                    } ?: builder.sortValue(range)
                }

                (indexMapper as DynamoDbAsyncTable<E>)
                    .getItem(
                        GetItemEnhancedRequest
                            .builder()
                            .key(builder.build())
                            .consistentRead(consistent)
                            .build(),
                    ).whenComplete { t, u ->
                        when (u) {
                            null -> {
                                operationTime.set(System.nanoTime() - startTime.get())

                                if (logger.isDebugEnabled()) {
                                    logger.debug { "Results received in: " + (System.currentTimeMillis() - timeBefore) + " ms" }
                                }

                                resultHandler.handle(succeededFuture(t))
                            }
                            else -> resultHandler.handle(failedFuture(u))
                        }
                    }
            }
            else -> {
                val key = QueryConditional.keyEqualTo(builder.build())
                var item: E? = null

                (indexMapper as DynamoDbAsyncIndex<E>)
                    .query(
                        QueryEnhancedRequest
                            .builder()
                            .consistentRead(consistent)
                            .queryConditional(key)
                            .limit(1)
                            .build(),
                    ).doAfterOnComplete {
                        resultHandler.handle(succeededFuture(item))
                    }.doAfterOnError {
                        resultHandler.handle(failedFuture(it))
                    }.doAfterOnCancel {
                        resultHandler.handle(failedFuture("CANCELLED"))
                    }.subscribe { page ->
                        operationTime.set(System.nanoTime() - startTime.get())

                        if (logger.isDebugEnabled()) {
                            logger.debug { "Results received in: " + (System.currentTimeMillis() - timeBefore) + " ms" }
                        }

                        page.items().firstOrNull()?.let { item = it }
                    }.exceptionally {
                        resultHandler.handle(failedFuture(it))

                        return@exceptionally null
                    }
            }
        }
    }

    @Throws(IllegalAccessException::class, InstantiationException::class)
    private fun fetchHashItem(
        hash: String,
        startTime: AtomicLong,
        preOperationTime: AtomicLong,
        operationTime: AtomicLong,
        GSI: String?,
        consistent: Boolean,
        resultHandler: Handler<AsyncResult<E?>>,
    ) {
        val key = Key.builder().partitionValue(hash).build()
        val query =
            QueryEnhancedRequest
                .builder()
                .consistentRead(if (GSI != null) false else consistent)
                .limit(1)
                .queryConditional(QueryConditional.keyEqualTo(key))
        val timeBefore = System.currentTimeMillis()

        preOperationTime.set(System.nanoTime() - startTime.get())

        @Suppress("UNCHECKED_CAST")
        when (val indexMapper = GSI?.let { mapper.index(GSI) } ?: mapper) {
            is DefaultDynamoDbAsyncTable<*>,
            is DynamoDbAsyncTable<*>,
            -> {
                var item: E? = null

                (indexMapper as DynamoDbAsyncTable<E>)
                    .query(query.build())
                    .doAfterOnComplete {
                        resultHandler.handle(succeededFuture(item))
                    }.doAfterOnError {
                        resultHandler.handle(failedFuture(it))
                    }.doAfterOnCancel {
                        resultHandler.handle(failedFuture("CANCELLED"))
                    }.subscribe { page ->
                        operationTime.set(System.nanoTime() - startTime.get())

                        if (logger.isDebugEnabled()) {
                            logger.debug { "Results received in: " + (System.currentTimeMillis() - timeBefore) + " ms" }
                        }

                        page.items().firstOrNull()?.let { item = it }
                    }.exceptionally {
                        resultHandler.handle(failedFuture(it))

                        return@exceptionally null
                    }
            }
            else -> {
                var item: E? = null

                (indexMapper as DynamoDbAsyncIndex<E>)
                    .query(query.build())
                    .doAfterOnComplete {
                        resultHandler.handle(succeededFuture(item))
                    }.doAfterOnError {
                        resultHandler.handle(failedFuture(it))
                    }.doAfterOnCancel {
                        resultHandler.handle(failedFuture("CANCELLED"))
                    }.subscribe { page ->
                        operationTime.set(System.nanoTime() - startTime.get())

                        if (logger.isDebugEnabled()) {
                            logger.debug { "Results received in: " + (System.currentTimeMillis() - timeBefore) + " ms" }
                        }

                        page.items().firstOrNull()?.let { item = it }
                    }.exceptionally {
                        resultHandler.handle(failedFuture(it))

                        return@exceptionally null
                    }
            }
        }
    }

    private fun returnTimedResult(
        readResult: E,
        preOperationTime: AtomicLong,
        operationTime: AtomicLong,
        postOperationTime: AtomicLong,
    ): ItemResult<E> {
        val eItemResult = ItemResult(readResult, false)
        eItemResult.preOperationProcessingTime = preOperationTime.get()
        eItemResult.operationProcessingTime = operationTime.get()
        eItemResult.postOperationProcessingTime = postOperationTime.get()

        return eItemResult
    }

    suspend fun index(query: DynamoDBQuery): ItemListResult<E> {
        val startTime = AtomicLong()
        startTime.set(System.nanoTime())

        val hash = query.identifiers.hash
        val baseEtagKey = query.baseEtagKey()
        val cacheId = "${TYPE.simpleName}_$hash/$baseEtagKey"

        if (logger.isDebugEnabled()) {
            logger.debug { "Running readAll with: $hash : $cacheId" }
        }

        return runCatching {
            val operationTime = AtomicLong()

            if (query.noCache) throw IllegalArgumentException("No cache")

            val result =
                cacheManager?.checkItemListCache(cacheId, query.projections, TYPE.simpleName)
                    ?: throw NoSuchElementException()
            operationTime.set(System.nanoTime() - startTime.get())

            if (logger.isDebugEnabled()) {
                logger.debug { "Served cached version of: $cacheId" }
            }

            ItemListResult(
                etagBase = baseEtagKey,
                count = result.meta.count,
                totalCount = result.meta.totalCount,
                items = result.items,
                paging = result.paging,
                projections = query.projections,
                isCacheHit = true,
                preOperationProcessingTime = startTime.get(),
                operationProcessingTime = operationTime.get(),
                postOperationProcessingTime = 0L,
            )
        }.recoverCatching {
            val orderBy = query.orderBy
            val filter = query.filter

            if (logger.isDebugEnabled()) {
                logger.debug { "Building expression with: " + encodePrettily(query) }
            }

            val queryBuilder = QueryEnhancedRequest.builder()
            val multiple = query.identifiers.ranges.isNotEmpty()

            if (!multiple && (orderBy != null || filter.isNotEmpty())) {
                dbParams.applyParameters(
                    fromS(hash),
                    orderBy,
                    filter,
                    queryBuilder,
                    query.identifiers.gsi?.let { GSI_KEY_MAP[it] },
                )
                dbParams.applyOrderBy(orderBy, queryBuilder)
            } else if (multiple) {
                dbParams.applyOrderBy(orderBy, queryBuilder)
            }

            if (logger.isDebugEnabled()) {
                logger.debug {
                    "Custom filter is: " +
                        "\nLimit: ${query.limit}" +
                        "\nExpression: ${queryBuilder?.build()?.filterExpression()?.expression()}" +
                        "\nAsc: ${queryBuilder?.build()?.scanIndexForward()}"
                }
            }

            returnDatabaseContent(query, cacheId, queryBuilder!!, startTime)
        }.getOrThrow()
    }

    private suspend fun returnDatabaseContent(
        query: DynamoDBQuery,
        cacheId: String,
        filteringExpression: QueryEnhancedRequest.Builder,
        startTime: AtomicLong,
    ): ItemListResult<E> {
        val etagKey = query.baseEtagKey()
        val filter = query.filter
        val nameParams = if (filter.isEmpty()) null else filter[PAGINATION_IDENTIFIER]
        val itemListFuture = Promise.promise<ItemListResult<E>>()

        if (logger.isDebugEnabled()) {
            logger.debug { "Do remoteRead" }
        }

        when {
            query.identifiers.hash == null ->
                runRootQuery(query, startTime, itemListFuture)

            filter.isNotEmpty() && nameParams != null && dbParams.isIllegalRangedKeyQueryParams(nameParams) ->
                runIllegalRangedKeyQueryAsScan(query, startTime, itemListFuture)

            else ->
                runStandardQuery(
                    query,
                    filteringExpression,
                    startTime,
                    itemListFuture,
                )
        }

        return runCatching {
            itemListFuture.future().coAwait()
        }.map { result ->
            val itemList = result.itemList

            if (logger.isDebugEnabled()) {
                logger.debug { "Constructed items!" }
            }

            when {
                cacheManager?.isItemListCacheAvailable() == true -> {
                    if (logger.isDebugEnabled()) {
                        logger.debug { "Constructing cache!" }
                    }

                    val cacheObject = itemList.toJson(query.projections)

                    if (logger.isDebugEnabled()) {
                        logger.debug { "Cache complete!" }
                    }

                    val content = cacheObject.encode()

                    if (logger.isDebugEnabled()) {
                        logger.debug { "Cache encoded!" }
                    }

                    runCatching {
                        cacheManager?.replaceItemListCache(content!!) { cacheId }
                    }.onFailure {
                        logger.warn(it) { "Failed itemlistcache replace" }
                    }

                    if (logger.isDebugEnabled()) {
                        logger.debug { "Setting: $etagKey with: ${itemList.meta.etag}" }
                    }

                    val etagItemListHashKey = "${TYPE.simpleName}_${query.identifiers.hashCode()}_itemListEtags"

                    etagManager?.setItemListEtags(etagItemListHashKey, etagKey, itemList)

                    result
                }

                else -> {
                    if (logger.isDebugEnabled()) {
                        logger.debug { "Setting: $etagKey with: ${itemList.meta.etag}" }
                    }

                    val etagItemListHashKey = "${TYPE.simpleName}_${query.identifiers.hashCode()}_itemListEtags"

                    etagManager?.setItemListEtags(etagItemListHashKey, etagKey, itemList)

                    result
                }
            }
        }.getOrThrow()
    }

    @Throws(InstantiationException::class, IllegalAccessException::class)
    private fun runStandardQuery(
        query: DynamoDBQuery,
        filteringExpression: QueryEnhancedRequest.Builder,
        startTime: AtomicLong,
        resultHandler: Promise<ItemListResult<E>>,
    ) {
        when {
            query.identifiers.ranges.isNotEmpty() ->
                standardMultipleQuery(
                    query = query,
                    startTime = startTime,
                    resultHandler = resultHandler,
                )

            else ->
                standardQuery(
                    query = query,
                    filteringExpression = filteringExpression,
                    startTime = startTime,
                    resultHandler = resultHandler,
                )
        }
    }

    private fun standardMultipleQuery(
        query: DynamoDBQuery,
        startTime: AtomicLong,
        resultHandler: Promise<ItemListResult<E>>,
    ) {
        val preOperationTime = AtomicLong()
        val operationTime = AtomicLong()
        val postOperationTime = AtomicLong()

        if (logger.isDebugEnabled()) {
            logger.debug { "Running multiple id query..." }
        }

        val keyPairsList =
            query.identifiers.ranges.map { id ->
                when {
                    hashOnlyModel() -> Key.builder().partitionValue(id).build()
                    else ->
                        keyBuilder(
                            partition = query.identifiers.hash!!,
                            sort = id,
                            sortNumber =
                                (
                                    query.identifiers.gsi?.let {
                                        db.getAlternativeIndexIdentifierReturnType(it)
                                    } ?: db.identifierField?.returnType
                                ) == String::class.java,
                        )
                }
            }

        val keyPairs = HashMap<Class<*>, List<Key>>()
        keyPairs[TYPE] = keyPairsList

        val timeBefore = System.currentTimeMillis()

        preOperationTime.set(System.nanoTime() - startTime.get())
        val indexedMapper =
            query.identifiers.gsi?.let {
                mapper.index(it)
            } ?: mapper
        val itemFutures =
            buildList {
                when (indexedMapper) {
                    is DefaultDynamoDbAsyncTable<*>,
                    is DynamoDbAsyncTable<*>,
                    -> {
                        keyPairs[TYPE]?.chunked(100)?.forEach {
                            val fut = Promise.promise<List<E>>()
                            val batchReader =
                                ReadBatch
                                    .builder(TYPE)
                                    .mappedTableResource(indexedMapper as DynamoDbAsyncTable<E>)
                            it.forEach { key ->
                                batchReader.addGetItem { it.key(key).consistentRead(true).build() }
                            }
                            val list = mutableListOf<E>()
                            val batchRequest = batchReader.build()

                            db.dynamoDbMapper
                                .batchGetItem {
                                    it.addReadBatch(batchRequest)
                                }.doAfterOnError {
                                    fut.fail(it)
                                }.doAfterOnCancel {
                                    fut.fail("CANCELLED")
                                }.doAfterOnComplete {
                                    fut.complete(list)
                                }.subscribe {
                                    val results = it.resultsForTable(indexedMapper)

                                    list.addAll(results)
                                }

                            add(fut.future())
                        }
                    }

                    else ->
                        keyPairs[TYPE]?.map {
                            val fut = Promise.promise<List<E>>()

                            @Suppress("UNCHECKED_CAST")
                            (indexedMapper as DynamoDbAsyncIndex<E>)
                                .query(QueryConditional.keyEqualTo(it))
                                .subscribe { page ->
                                    fut.complete(page.items().firstOrNull()?.let { listOf(it) } ?: emptyList())
                                }.exceptionally { error ->
                                    fut.fail(error)

                                    return@exceptionally null
                                }

                            add(fut.future())
                        } ?: emptyList<Future<*>>()
                }
            }

        Future.all<Any>(itemFutures).andThen { asyncResult ->
            when {
                asyncResult.failed() -> resultHandler.handle(failedFuture(asyncResult.cause()))
                else ->
                    runCatching {
                        operationTime.set(System.nanoTime() - preOperationTime.get())

                        val items = itemFutures.map { it.result() as List<E> }.flatten()
                        var pageCount = items.size
                        val desiredCount =
                            when (query.autoPaginate) {
                                true -> Int.MAX_VALUE
                                else -> query.limit
                            }

                        if (logger.isDebugEnabled()) {
                            logger.debug { "Results received in: ${System.currentTimeMillis() - timeBefore} ms" }
                        }

                        if (logger.isDebugEnabled()) {
                            logger.debug { "Items Returned for collection: $pageCount" }
                        }

                        var allItems =
                            items.sortedWith(
                                Comparator.comparing(
                                    { it.range(HASH_FIELD)!! },
                                    Comparator.comparingInt { query.identifiers.ranges.indexOf(it) },
                                ),
                            )
                        var oldPageToken: AttributeValue? = null
                        var totalScanned: Int? = null

                        if (query.pageToken != null) {
                            val pageTokenMap = getTokenMap(query.pageToken)
                            oldPageToken = pageTokenMap?.remove("oldPageToken")
                            totalScanned = pageTokenMap?.remove("totalScannedCountForAllPages")?.n()?.toInt()
                            val id = pageTokenMap!![IDENTIFIER]?.s()

                            val first =
                                allItems.firstOrNull { item ->
                                    if (logger.isDebugEnabled()) {
                                        logger.debug { "Id is: " + id + ", Range is: " + item.range(IDENTIFIER_FIELD) }
                                    }

                                    item.range(IDENTIFIER_FIELD) == id
                                }

                            if (first != null) {
                                allItems = allItems.subList(allItems.indexOf(first) + 1, allItems.size)
                            }
                        }

                        val itemList = allItems.take(desiredCount)

                        pageCount = allItems.size
                        val count = items.size
                        val pagingToken =
                            setScanPageToken(
                                query.pageToken,
                                allItems.size,
                                desiredCount,
                                totalScanned ?: items.size,
                                items,
                                query.identifiers.gsi,
                                query.identifiers.lsi,
                            )

                        returnTimedItemListResult(
                            baseEtagKey = query.baseEtagKey(),
                            resultHandler = resultHandler,
                            count = count,
                            totalCount = totalScanned ?: items.size,
                            previousPageToken = oldPageToken?.s(),
                            pageToken = query.pageToken,
                            nextPageToken = pagingToken,
                            itemList = itemList,
                            projections = query.projections,
                            preOperationTime = preOperationTime,
                            operationTime = operationTime,
                            postOperationTime = postOperationTime,
                        )
                    }.onFailure {
                        resultHandler.handle(failedFuture(it))
                    }.getOrNull()
            }
        }
    }

    private fun keyBuilder(
        partition: String,
        sort: String,
        sortNumber: Boolean,
    ) = when {
        sortNumber ->
            Key
                .builder()
                .partitionValue(partition)
                .sortValue(fromS(sort))
                .build()

        else ->
            Key
                .builder()
                .partitionValue(partition)
                .sortValue(fromN(sort))
                .build()
    }

    private fun returnTimedItemListResult(
        baseEtagKey: String?,
        resultHandler: Promise<ItemListResult<E>>,
        count: Int,
        totalCount: Int,
        previousPageToken: String?,
        pageToken: String?,
        nextPageToken: String,
        itemList: List<E>,
        projections: Set<String>,
        preOperationTime: AtomicLong,
        operationTime: AtomicLong,
        postOperationTime: AtomicLong,
    ) {
        postOperationTime.set(System.nanoTime() - operationTime.get())
        val pageTokens = PageTokens(self = pageToken, next = nextPageToken, previous = previousPageToken)
        val eItemListResult =
            ItemListResult(
                etagBase = baseEtagKey,
                count = count,
                totalCount = totalCount,
                items = itemList,
                paging = pageTokens,
                projections = projections,
                isCacheHit = false,
                preOperationProcessingTime = preOperationTime.get(),
                operationProcessingTime = operationTime.get(),
                postOperationProcessingTime = postOperationTime.get(),
            )

        resultHandler.handle(succeededFuture(eItemListResult))
    }

    @Throws(IllegalAccessException::class, InstantiationException::class)
    private fun standardQuery(
        query: DynamoDBQuery,
        filteringExpression: QueryEnhancedRequest.Builder,
        startTime: AtomicLong,
        resultHandler: Promise<ItemListResult<E>>,
    ) {
        val preOperationTime = AtomicLong()
        val operationTime = AtomicLong()
        val postOperationTime = AtomicLong()

        val effectivelyFinalProjections =
            dbParams.buildProjections(
                query.projections,
                when {
                    query.identifiers.ranges.isEmpty() -> paginationIndex
                    else -> query.identifiers.lsi
                },
            )

        logger.debug { "Running standard query..." }

        val oldPageToken: AttributeValue?
        val totalScanned: Int?

        val tokenMap = getTokenMap(query.pageToken)
        oldPageToken = tokenMap?.remove("oldPageToken")
        totalScanned = tokenMap?.remove("totalScannedCountForAllPages")?.n()?.toInt()

        if (tokenMap?.isNotEmpty() == true) {
            filteringExpression.exclusiveStartKey(tokenMap)
        }

        when (query.identifiers.gsi) {
            null ->
                if (filteringExpression.build().queryConditional() == null) {
                    filteringExpression.queryConditional(
                        QueryConditional.keyEqualTo(Key.builder().partitionValue(query.identifiers.hash).build()),
                    )
                }

            else ->
                if (filteringExpression.build().queryConditional() == null && query.identifiers.hash != null) {
                    setFilterExpressionKeyCondition(filteringExpression, query.identifiers.hash)
                }
        }

        filteringExpression.consistentRead(false)
        setProjectionsOnQueryExpression(filteringExpression, effectivelyFinalProjections)

        when {
            query.orderBy != null ->
                when (query.orderBy.direction) {
                    DESC -> filteringExpression.scanIndexForward(false)
                    else -> filteringExpression.scanIndexForward(true)
                }
            else -> filteringExpression.scanIndexForward(false)
        }

        val timeBefore = System.currentTimeMillis()
        preOperationTime.set(System.nanoTime() - startTime.get())
        val indexedMapper =
            query.identifiers.gsi?.let {
                mapper.index(it)
            } ?: query.identifiers.lsi?.let {
                mapper.index(it)
            } ?: mapper
        val request = filteringExpression.build()

        @Suppress("UNCHECKED_CAST")
        val queryPageResults =
            when (indexedMapper) {
                is DefaultDynamoDbAsyncTable<*>,
                is DynamoDbAsyncTable<*>,
                -> (indexedMapper as DynamoDbAsyncTable<E>).query(request)
                else -> (indexedMapper as DynamoDbAsyncIndex<E>).query(request)
            }
        val pages = mutableListOf<Page<E>>()

        queryPageResults
            .limit(if (query.autoPaginate) Integer.MAX_VALUE else ceil(query.limit.toDouble() / 100).toInt())
            .doAfterOnComplete {
                operationTime.set(System.nanoTime() - preOperationTime.get())

                val items = pages.map { it.items() }.flatten()

                logger.debug { "Results received in: " + (System.currentTimeMillis() - timeBefore) + " ms" }
                logger.debug { "Scanned items: " + items.size }

                val desiredCount =
                    when {
                        query.autoPaginate -> Int.MAX_VALUE
                        else -> query.limit
                    }
                val itemList = items.subList(0, min(desiredCount, items.size))
                val count = itemList.size
                val lastEvaluatedKey =
                    itemList.lastOrNull()?.let {
                        setLastKey(
                            lastKey = it,
                            gsi = query.identifiers.gsi,
                            lsi = query.identifiers.lsi,
                        )
                    }
                val pagingToken =
                    when {
                        lastEvaluatedKey == null ||
                            pages.last().lastEvaluatedKey() == null &&
                            items.size == itemList.size -> "END_OF_LIST"
                        else -> {
                            logger.debug {
                                lastEvaluatedKey.map {
                                    "${it.key} : ${
                                        when (it.value.type()) {
                                            AttributeValue.Type.N -> "${it.value.n()} as N"
                                            else -> it.value.s()
                                        }
                                    }"
                                }
                            }
                            val totalScannedCountForAllPages = totalScanned ?: items.size

                            Base64.getUrlEncoder().encodeToString(
                                (
                                    "${extractSelfToken(query.pageToken)}:" +
                                        setPageToken(lastEvaluatedKey, totalScannedCountForAllPages)
                                ).toByteArray(),
                            )
                        }
                    }

                returnTimedItemListResult(
                    baseEtagKey = query.baseEtagKey(),
                    resultHandler = resultHandler,
                    count = count,
                    totalCount = totalScanned ?: items.size,
                    previousPageToken = oldPageToken?.s(),
                    pageToken = query.pageToken,
                    nextPageToken = pagingToken,
                    itemList = itemList,
                    projections = query.projections,
                    preOperationTime = preOperationTime,
                    operationTime = operationTime,
                    postOperationTime = postOperationTime,
                )
            }.doAfterOnError {
                resultHandler.handle(failedFuture(it))
            }.doAfterOnCancel {
                resultHandler.handle(failedFuture("CANCELLED"))
            }.subscribe {
                pages.add(it)
            }.exceptionally { throwable ->
                logger.error {
                    "$throwable : ${throwable.message} : ${throwable.stackTrace.joinToString {
                        "${it.fileName}:${it.lineNumber}\n"
                    }}"
                }

                resultHandler.handle(
                    ServiceException.fail(
                        500,
                        buildString {
                            append(throwable.message)
                            append("\n")
                            append(
                                throwable.stackTrace.joinToString("\n") {
                                    "${it.fileName}:${it.lineNumber}\n"
                                },
                            )
                        },
                        JsonObject(encode(throwable)),
                    ),
                )

                return@exceptionally null
            }
    }

    private fun runIllegalRangedKeyQueryAsScan(
        query: DynamoDBQuery,
        startTime: AtomicLong,
        resultHandler: Promise<ItemListResult<E>>,
    ) {
        val preOperationTime = AtomicLong()
        val operationTime = AtomicLong()
        val postOperationTime = AtomicLong()

        if (logger.isDebugEnabled()) {
            logger.debug { "Running illegal rangedKey query..." }
        }

        val hashScanKey = "#HASH_KEY_VALUE"
        val hashScanValue = ":HASH_VALUE"

        val scanExpression = dbParams.applyParameters(query.filter)
        val conditionString = scanExpression.build().filterExpression()
        val expression = Expression.builder()
        if (query.identifiers.hash != null) expression.expression("$conditionString AND $hashScanKey = $hashScanValue")

        when (query.identifiers.gsi) {
            null -> expression.putExpressionName(hashScanKey, HASH_IDENTIFIER)
            else ->
                expression.putExpressionName(
                    hashScanKey,
                    GSI_KEY_MAP[query.identifiers.gsi]?.getString("hash"),
                )
        }

        expression.putExpressionValue(hashScanValue, fromS(query.identifiers.hash))
        scanExpression.filterExpression(expression.build())

        val pageTokenMap = getTokenMap(query.pageToken)

        scanExpression.consistentRead(query.consistent)
        setProjectionsOnScanExpression(scanExpression, query.projections)

        val timeBefore = System.currentTimeMillis()

        val scan = scanExpression.build()
        preOperationTime.set(System.nanoTime() - startTime.get())
        val indexedMapper =
            query.identifiers.gsi?.let {
                mapper.index(it)
            } ?: query.identifiers.lsi?.let {
                mapper.index(it)
            } ?: mapper

        @Suppress("UNCHECKED_CAST")
        val scanResults =
            when (indexedMapper) {
                is DefaultDynamoDbAsyncTable<*>,
                is DynamoDbAsyncTable<*>,
                -> (indexedMapper as DynamoDbAsyncTable<E>).scan(scan)
                else -> (indexedMapper as DynamoDbAsyncIndex<E>).scan(scan)
            }
        val allResults = mutableListOf<Page<E>>()

        scanResults
            .limit(if (query.autoPaginate) Integer.MAX_VALUE else ceil(query.limit.toDouble() / 100).toInt())
            .doAfterOnComplete {
                operationTime.set(System.nanoTime() - preOperationTime.get())

                val desiredCount =
                    when {
                        query.autoPaginate -> Int.MAX_VALUE
                        else -> query.limit
                    }
                val results = allResults.map { it.items() }.flatten()
                val items = results.subList(0, min(desiredCount, results.size))
                var pageCount = items.size

                if (logger.isDebugEnabled()) {
                    logger.debug { "$pageCount results received in: ${System.currentTimeMillis() - timeBefore} ms" }
                }

                val orderIsAscending = query.orderBy != null && query.orderBy.direction == ASC
                val finalAlternateIndex = query.orderBy?.field ?: PAGINATION_IDENTIFIER
                val comparator =
                    Comparator.comparing<E, String> { e ->
                        db.getFieldAsString(finalAlternateIndex!!, e) ?: "null"
                    }
                val comparing = if (orderIsAscending) comparator else comparator.reversed()

                var allItems = items.sortedWith(comparing)
                var oldPageToken: AttributeValue? = null
                var totalScanned: Int? = null

                if (query.pageToken != null) {
                    val id = pageTokenMap!![IDENTIFIER]?.s()
                    oldPageToken = pageTokenMap.remove("oldPageToken")
                    totalScanned = pageTokenMap.remove("totalScannedCountForAllPages")?.n()?.toInt()

                    val first =
                        allItems
                            .stream()
                            .filter { item ->
                                if (logger.isDebugEnabled()) {
                                    logger.debug { "Id is: " + id + ", Range is: " + item.range(IDENTIFIER_FIELD) }
                                }

                                item.range(IDENTIFIER_FIELD) == id
                            }.findFirst()

                    if (first.isPresent) {
                        allItems = allItems.subList(allItems.indexOf(first.get()) + 1, allItems.size)
                    }
                }

                val itemList =
                    when (query.autoPaginate) {
                        true -> allItems
                        else -> allItems.take(desiredCount)
                    }

                pageCount = allItems.size
                val count = items.size
                val pagingToken =
                    setScanPageToken(
                        pageToken = query.pageToken,
                        pageCount = results.size,
                        desiredCount =
                            when (query.autoPaginate) {
                                true -> Int.MAX_VALUE
                                else -> desiredCount
                            },
                        totalScannedCountForAllPages = totalScanned ?: items.size,
                        itemList = items,
                        gsi = query.identifiers.gsi,
                        lsi = query.identifiers.lsi,
                    )

                returnTimedItemListResult(
                    baseEtagKey = query.baseEtagKey(),
                    resultHandler = resultHandler,
                    count = count,
                    totalCount = totalScanned ?: items.size,
                    previousPageToken = oldPageToken?.s(),
                    pageToken = query.pageToken,
                    nextPageToken = pagingToken,
                    itemList = itemList,
                    projections = query.projections,
                    preOperationTime = preOperationTime,
                    operationTime = operationTime,
                    postOperationTime = postOperationTime,
                )
            }.doAfterOnError {
                resultHandler.handle(failedFuture(it))
            }.doAfterOnCancel {
                resultHandler.handle(failedFuture("CANCELLED"))
            }.subscribe {
                allResults.add(it)
            }.exceptionally { throwable ->
                logger.error {
                    "$throwable : ${throwable.message} : ${throwable.stackTrace.joinToString {
                        "${it.fileName}:${it.lineNumber}\n"
                    }}"
                }

                resultHandler.handle(
                    ServiceException.fail(
                        500,
                        buildString {
                            append(throwable.message)
                            append("\n")
                            append(
                                throwable.stackTrace.joinToString("\n") {
                                    "${it.fileName}:${it.lineNumber}\n"
                                },
                            )
                        },
                        JsonObject(encode(throwable)),
                    ),
                )

                return@exceptionally null
            }
    }

    private fun runRootQuery(
        query: DynamoDBQuery,
        startTime: AtomicLong,
        resultHandler: Promise<ItemListResult<E>>,
    ) {
        when {
            query.identifiers.ranges.isNotEmpty() -> rootMultipleQuery(query, startTime, resultHandler)
            else -> rootRootQuery(query, startTime, resultHandler)
        }
    }

    private fun rootMultipleQuery(
        query: DynamoDBQuery,
        startTime: AtomicLong,
        resultHandler: Promise<ItemListResult<E>>,
    ) {
        val preOperationTime = AtomicLong()
        val operationTime = AtomicLong()
        val postOperationTime = AtomicLong()

        logger.debug { "Running root multiple id query..." }

        val keyPairsList =
            query.identifiers.ranges.map { id ->
                when {
                    hashOnlyModel() -> Key.builder().partitionValue(id).build()
                    else ->
                        keyBuilder(
                            partition = query.identifiers.hash!!,
                            sort = id,
                            sortNumber =
                                (
                                    query.identifiers.gsi?.let {
                                        db.getAlternativeIndexIdentifierReturnType(it)
                                    } ?: db.identifierField?.returnType
                                ) == String::class.java,
                        )
                }
            }

        val keyPairs = HashMap<Class<*>, List<Key>>()
        keyPairs[TYPE] = keyPairsList

        val timeBefore = System.currentTimeMillis()

        preOperationTime.set(System.nanoTime() - startTime.get())

        val indexedMapper =
            query.identifiers.gsi?.let {
                mapper.index(it)
            } ?: query.identifiers.lsi?.let {
                mapper.index(it)
            } ?: mapper
        val itemFutures =
            buildList {
                when (indexedMapper) {
                    is DefaultDynamoDbAsyncTable<*>,
                    is DynamoDbAsyncTable<*>,
                    -> {
                        keyPairs[TYPE]?.chunked(100)?.forEach {
                            val fut = Promise.promise<List<E>>()
                            val batchReader =
                                ReadBatch
                                    .builder(TYPE)
                                    .mappedTableResource(indexedMapper as DynamoDbAsyncTable<E>)
                            it.forEach { key ->
                                batchReader.addGetItem { it.key(key).consistentRead(true).build() }
                            }
                            val list = mutableListOf<E>()
                            val batchRequest = batchReader.build()

                            db.dynamoDbMapper
                                .batchGetItem {
                                    it.addReadBatch(batchRequest)
                                }.doAfterOnError {
                                    fut.fail(it)
                                }.doAfterOnCancel {
                                    fut.fail("CANCELLED")
                                }.doAfterOnComplete {
                                    fut.complete(list)
                                }.subscribe {
                                    val results = it.resultsForTable(indexedMapper)

                                    list.addAll(results)
                                }

                            add(fut.future())
                        }
                    }

                    else ->
                        keyPairs[TYPE]?.map {
                            val fut = Promise.promise<E?>()

                            @Suppress("UNCHECKED_CAST")
                            (indexedMapper as DynamoDbAsyncIndex<E>)
                                .query(QueryConditional.keyEqualTo(it))
                                .subscribe { page ->
                                    fut.complete(page.items().firstOrNull())
                                }.exceptionally { error ->
                                    fut.fail(error)

                                    return@exceptionally null
                                }

                            add(fut.future())
                        } ?: emptyList<Future<*>>()
                }
            }

        Future.all<Any>(itemFutures).andThen { asyncResult ->
            when {
                asyncResult.failed() -> resultHandler.handle(failedFuture(asyncResult.cause()))
                else ->
                    runCatching {
                        operationTime.set(System.nanoTime() - preOperationTime.get())

                        val items =
                            when (indexedMapper) {
                                is DefaultDynamoDbAsyncTable<*>,
                                is DynamoDbAsyncTable<*>,
                                -> itemFutures.map { it.result() as List<E> }.flatten()

                                else -> itemFutures.mapNotNull { it.result() }
                            }
                        var pageCount = items.size
                        val desiredCount =
                            when (query.autoPaginate) {
                                true -> Int.MAX_VALUE
                                else -> query.limit
                            }

                        if (logger.isDebugEnabled()) {
                            logger.debug { "Results received in: " + (System.currentTimeMillis() - timeBefore) + " ms" }
                        }

                        if (logger.isDebugEnabled()) {
                            logger.debug { "Items Returned for collection: $pageCount" }
                        }

                        val allItems =
                            items
                                .map { item -> item as E }
                                .sortedWith(
                                    Comparator.comparing(
                                        if (hashOnlyModel()) {
                                            Function<E, String> { it.hash(HASH_FIELD)!! }
                                        } else {
                                            Function { it.range(IDENTIFIER_FIELD)!! }
                                        },
                                        Comparator.comparingInt { query.identifiers.ranges.indexOf(it) },
                                    ),
                                )

                        var oldPageToken: AttributeValue? = null
                        var totalScanned: Int? = null

                        if (query.pageToken != null) {
                            val pageTokenMap = getTokenMap(query.pageToken)
                            oldPageToken = pageTokenMap?.remove("oldPageToken")
                            totalScanned = pageTokenMap?.remove("totalScannedCountForAllPages")?.n()?.toInt()
                        }

                        val itemList = allItems.take(desiredCount)

                        pageCount = allItems.size
                        val count = itemList.size
                        val pagingToken =
                            setScanPageToken(
                                pageToken = query.pageToken,
                                pageCount = allItems.size,
                                desiredCount = desiredCount,
                                totalScannedCountForAllPages = totalScanned ?: items.size,
                                itemList = itemList,
                                gsi = query.identifiers.gsi,
                                lsi = query.identifiers.lsi,
                            )

                        returnTimedItemListResult(
                            baseEtagKey = query.baseEtagKey(),
                            resultHandler = resultHandler,
                            count = count,
                            totalCount = totalScanned ?: items.size,
                            previousPageToken = oldPageToken?.s(),
                            pageToken = query.pageToken,
                            nextPageToken = pagingToken,
                            itemList = itemList,
                            projections = query.projections,
                            preOperationTime = preOperationTime,
                            operationTime = operationTime,
                            postOperationTime = postOperationTime,
                        )
                    }.onFailure {
                        resultHandler.handle(failedFuture(it))
                    }
            }
        }
    }

    private fun rootRootQuery(
        query: DynamoDBQuery,
        startTime: AtomicLong,
        resultHandler: Promise<ItemListResult<E>>,
    ) {
        val preOperationTime = AtomicLong()
        val operationTime = AtomicLong()
        val postOperationTime = AtomicLong()

        if (logger.isDebugEnabled()) {
            logger.debug { "Running root query..." }
        }

        val scanExpression = dbParams.applyParameters(query.filter)
        val pageTokenMap = getTokenMap(query.pageToken)
        val oldPageToken = pageTokenMap?.remove("oldPageToken")
        val totalScanned = pageTokenMap?.remove("totalScannedCountForAllPages")?.n()?.toInt()

        scanExpression.consistentRead(query.consistent)
        pageTokenMap?.let { scanExpression.exclusiveStartKey(it) }
        setProjectionsOnScanExpression(scanExpression, query.projections)

        val timeBefore = System.currentTimeMillis()

        preOperationTime.set(System.nanoTime() - startTime.get())
        val scan = scanExpression.build()
        val indexedMapper =
            query.identifiers.gsi?.let {
                mapper.index(it)
            } ?: query.identifiers.lsi?.let {
                mapper.index(it)
            } ?: mapper

        @Suppress("UNCHECKED_CAST")
        val scanResults =
            when (indexedMapper) {
                is DefaultDynamoDbAsyncTable<*>,
                is DynamoDbAsyncTable<*>,
                -> (indexedMapper as DynamoDbAsyncTable<E>).scan(scan)
                else -> (indexedMapper as DynamoDbAsyncIndex<E>).scan(scan)
            }
        val allResults = mutableListOf<Page<E>>()

        scanResults
            .limit(if (query.autoPaginate) Integer.MAX_VALUE else ceil(query.limit.toDouble() / 100).toInt())
            .doAfterOnComplete {
                operationTime.set(System.nanoTime() - preOperationTime.get())

                val desiredCount =
                    when {
                        query.autoPaginate -> Int.MAX_VALUE
                        else -> query.limit
                    }
                val results = allResults.map { it.items() }.flatten()
                val items = results.subList(0, min(desiredCount, results.size))

                if (logger.isDebugEnabled()) {
                    logger.debug { "DesiredCount is: $desiredCount" }
                    logger.debug {
                        "${results.size} results received in: ${System.currentTimeMillis() - timeBefore} ms"
                    }
                }

                val count = items.size
                val pagingToken =
                    setScanPageToken(
                        pageToken = query.pageToken,
                        pageCount = results.size,
                        desiredCount =
                            when (query.autoPaginate) {
                                true -> Int.MAX_VALUE
                                else -> desiredCount
                            },
                        totalScannedCountForAllPages = totalScanned ?: results.size,
                        itemList = items,
                        gsi = query.identifiers.gsi,
                        lsi = query.identifiers.lsi,
                    )

                returnTimedItemListResult(
                    baseEtagKey = query.baseEtagKey(),
                    resultHandler = resultHandler,
                    count = count,
                    totalCount = totalScanned ?: results.size,
                    previousPageToken = oldPageToken?.s(),
                    pageToken = query.pageToken,
                    nextPageToken = pagingToken,
                    itemList = items,
                    projections = query.projections,
                    preOperationTime = preOperationTime,
                    operationTime = operationTime,
                    postOperationTime = postOperationTime,
                )
            }.doAfterOnError {
                resultHandler.handle(failedFuture(it))
            }.doAfterOnCancel {
                resultHandler.handle(failedFuture("CANCELLED"))
            }.subscribe {
                allResults.add(it)
            }.exceptionally {
                resultHandler.handle(failedFuture(it))

                return@exceptionally null
            }
    }

    private fun setProjectionsOnScanExpression(
        scanExpression: ScanEnhancedRequest.Builder,
        projections: Set<String>?,
    ) {
        if (!projections.isNullOrEmpty()) {
            scanExpression.attributesToProject(projections)
        }
    }

    private fun setProjectionsOnQueryExpression(
        queryExpression: QueryEnhancedRequest.Builder,
        projections: Set<String>?,
    ) {
        if (!projections.isNullOrEmpty()) {
            queryExpression.attributesToProject(projections)
        }
    }

    private fun setScanPageToken(
        pageToken: String?,
        pageCount: Int,
        desiredCount: Int,
        totalScannedCountForAllPages: Int,
        itemList: List<E>,
        gsi: String?,
        lsi: String?,
    ): String {
        when {
            pageCount > desiredCount -> {
                val lastItem = itemList[if (itemList.isEmpty()) 0 else itemList.size - 1]
                val lastEvaluatedKey = setLastKey(lastItem, gsi, lsi)

                return when {
                    lastEvaluatedKey.isEmpty() -> "END_OF_LIST"
                    else -> {
                        Base64.getUrlEncoder().encodeToString(
                            (
                                "${extractSelfToken(pageToken)}:" +
                                    setPageToken(lastEvaluatedKey, totalScannedCountForAllPages)
                            ).toByteArray(),
                        )
                    }
                }
            }

            else -> return "END_OF_LIST"
        }
    }

    private fun extractSelfToken(pageToken: String?): String {
        if (pageToken == null) return "null"

        return String(Base64.getUrlDecoder().decode(pageToken)).split(":".toRegex(), 2)[1]
    }

    private fun hashOnlyModel(): Boolean = IDENTIFIER == null || IDENTIFIER == ""

    private fun setPageToken(
        lastEvaluatedKey: Map<String, AttributeValue>,
        totalScannedCountForAllPages: Int,
    ): String = dbParams.createNewPageToken(lastEvaluatedKey, totalScannedCountForAllPages)

    private fun getTokenMap(pageToken: String?): MutableMap<String, AttributeValue>? = dbParams.createPageTokenMap(pageToken)

    private fun setLastKey(
        lastKey: E,
        gsi: String?,
        lsi: String?,
    ): Map<String, AttributeValue> {
        val keyMap = HashMap<String, AttributeValue>()
        val hashOnlyModel = hashOnlyModel()

        keyMap[HASH_IDENTIFIER] = fromS(lastKey.hash(HASH_FIELD))
        if (!hashOnlyModel) keyMap[IDENTIFIER!!] = fromS(lastKey.range(IDENTIFIER_FIELD))

        when {
            !hashOnlyModel -> {
                logger.debug { "Fetching remoteIndex value!" }

                val indexValue =
                    when (gsi) {
                        null ->
                            when (lsi) {
                                PAGINATION_INDEX -> PAGINATION_IDENTIFIER
                                else -> lsi
                            }
                        else -> null
                    }

                indexValue?.let { keyMap[indexValue] = db.getIndexValue(indexValue, lastKey) }
            }
        }

        if (gsi != null) {
            val keyObject = GSI_KEY_MAP[gsi]
            val hash = keyObject!!.getString("hash")
            val range = keyObject.getString("range")

            (keyMap as MutableMap<String, AttributeValue>)
                .putIfAbsent(hash, fromS(db.getFieldAsString(hash, lastKey)))

            if (range != null) {
                (keyMap as MutableMap<String, AttributeValue>).putIfAbsent(range, db.getIndexValue(range, lastKey))
            }
        }

        return keyMap
    }

    @Throws(IllegalAccessException::class, InstantiationException::class)
    private fun setFilterExpressionKeyCondition(
        filterExpression: QueryEnhancedRequest.Builder,
        identifier: String,
    ) {
        filterExpression.queryConditional(
            QueryConditional.keyEqualTo(
                Key.builder().partitionValue(identifier).build(),
            ),
        )
        filterExpression.consistentRead(false)
    }
}
