/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.models

import com.fasterxml.jackson.annotation.JsonInclude
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.codegen.annotations.GenIgnore
import java.util.Arrays.stream
import java.util.concurrent.atomic.AtomicLong
import kotlin.Long.Companion.MAX_VALUE

private val logger = KotlinLogging.logger { }

/**
 * This class defines an interface for models that operate on etags.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
interface ETagable {
    var etag: String

    @GenIgnore
    fun generateAndSetEtag(map: MutableMap<String, String>): Map<String, String> {
        val oldTag = etag
        val etagCode = buildNewTag(map)
        val newTag = ModelUtils.returnNewEtag(etagCode.get())

        if (oldTag == newTag) return map
        val etagField =
            kotlin
                .runCatching {
                    this::class.java.superclass.getDeclaredField("etag")
                }.recoverCatching {
                    this::class.java.getDeclaredField("etag")
                }.getOrThrow()
        etagField.isAccessible = true
        etagField.set(this, newTag)
        map[generateEtagKeyIdentifier()] = newTag

        return reGenerateParent(map)
    }

    @GenIgnore
    fun buildNewTag(map: MutableMap<String, String>): AtomicLong {
        val etagCode = AtomicLong(MAX_VALUE)

        stream(javaClass.declaredFields).filter { it.name != "etag" }.forEach { field ->
            field.isAccessible = true

            when {
                Collection::class.java.isAssignableFrom(field.type) ->
                    try {
                        val objectFieldValue = field.get(this)

                        if (objectFieldValue != null) {
                            (objectFieldValue as Collection<*>).forEach { o ->
                                val e = o as ETagable
                                val stringStringMap = e.generateAndSetEtag(map)
                                map.putAll(stringStringMap)

                                val innerEtagCode = e.etag.hashCode().toLong()

                                etagCode.set(etagCode.get() xor innerEtagCode)
                            }
                        }
                    } catch (e: IllegalAccessException) {
                        logger.error(e) { "Cannot access collection for etag!" }
                    }

                else ->
                    try {
                        val value = field.get(this)
                        val innerEtagCode = value?.hashCode()?.toLong() ?: 12345L

                        etagCode.set(etagCode.get() xor innerEtagCode)
                    } catch (e: IllegalAccessException) {
                        logger.error(e) { "Cannot access field for etag!" }
                    }
            }
        }
        return etagCode
    }

    @GenIgnore
    fun reGenerateParent(map: Map<String, String>): Map<String, String> = map

    @GenIgnore
    fun generateEtagKeyIdentifier(): String = javaClass.name + "_etag_" + hashCode()
}
