/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.genghis.tools.repository.repository.cache

import com.fasterxml.jackson.annotation.JsonTypeInfo
import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.utils.ItemList
import com.genghis.tools.repository.utils.ItemListMeta
import com.genghis.tools.repository.utils.PageTokens
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.Promise
import io.vertx.core.Vertx
import io.vertx.core.json.Json
import io.vertx.core.json.Json.encode
import io.vertx.core.json.JsonArray
import io.vertx.core.json.JsonObject
import io.vertx.core.shareddata.LocalMap
import io.vertx.kotlin.coroutines.coAwait
import io.vertx.serviceproxy.ServiceException
import java.util.Arrays
import java.util.concurrent.Callable
import java.util.function.Function
import java.util.function.Supplier
import java.util.stream.Collectors.toList

private val logger = KotlinLogging.logger { }

/**
 * The LocalCacheManagerImpl contains the logic for setting, removing, and replace caches.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
@Suppress("PrivatePropertyName")
class LocalCacheManagerImpl<E>(
    private val TYPE: Class<E>,
    private val vertx: Vertx,
) : CacheManager<E> where E : Model, E : Cacheable {
    private val itemListKeyMap: String = TYPE.simpleName + "/ITEMLIST"
    private val aggregationKeyMap: String = TYPE.simpleName + "/AGGREGATION"

    private val hasTypeJsonField: Boolean = Arrays.stream(TYPE.declaredAnnotations).anyMatch { a -> a is JsonTypeInfo }

    private val objectCache: LocalMap<String, String>?
        get() = vertx.sharedData().getLocalMap("objectCache")

    private val itemListCache: LocalMap<String, String>?
        get() = vertx.sharedData().getLocalMap("itemListCache")

    private val aggregationCache: LocalMap<String, String>?
        get() = vertx.sharedData().getLocalMap("aggregationCache")

    override suspend fun isObjectCacheAvailable(): Boolean {
        val available = objectCache != null

        if (!available) {
            objectCache
        }

        return available
    }

    override suspend fun isItemListCacheAvailable(): Boolean {
        val available = itemListCache != null

        if (!available) {
            itemListCache
        }

        return available
    }

    override suspend fun isAggregationCacheAvailable(): Boolean {
        val available = aggregationCache != null

        if (!available) {
            aggregationCache
        }

        return available
    }

    override suspend fun initializeCache(): Boolean {
        if (cachesCreated) return true

        cachesCreated = true

        return true
    }

    override suspend fun checkObjectCache(
        cacheId: String,
        typeName: String,
    ): E =
        when {
            isObjectCacheAvailable() -> {
                when (val content = objectCache!![cacheId]) {
                    null -> throw ServiceException(404, "Cache result is null!")
                    else -> Json.decodeValue(content, TYPE)
                }
            }

            else -> {
                logger.error { "ObjectCache is null, recreating..." }

                throw ServiceException(404, "Unable to retrieve from cache, cache was null...")
            }
        }

    override suspend fun checkItemListCache(
        cacheId: String,
        projections: Set<String>,
        typeName: String,
    ): ItemList<E> =
        when {
            isItemListCacheAvailable() -> {
                when (val content = itemListCache!![cacheId]) {
                    null -> throw ServiceException(404, "Cache result is null!")
                    else ->
                        runCatching {
                            val jsonObject = JsonObject(content)
                            val jsonArray = jsonObject.getJsonArray("items")
                            val meta = ItemListMeta(jsonObject.getJsonObject("meta"))
                            val pageToken = PageTokens(jsonObject.getJsonObject("paging"))
                            val items =
                                jsonArray
                                    .stream()
                                    .map { json ->
                                        val obj = JsonObject(json.toString())

                                        if (hasTypeJsonField) {
                                            obj.put("@type", TYPE.simpleName)
                                        }

                                        Json.decodeValue(obj.encode(), TYPE)
                                    }.collect(toList())

                            ItemList<E>(
                                items = items,
                                meta = meta,
                                paging = pageToken,
                            )
                        }.getOrElse {
                            logger.error {
                                "$it : ${it.message} : ${it.stackTrace?.joinToString {
                                    "${it.fileName}:${it.lineNumber}\n"
                                }}"
                            }

                            throw ServiceException(404, "Cache result is null...", JsonObject(encode(it)))
                        }
                }
            }

            else -> {
                logger.error { "ItemList Cache is null, recreating..." }

                throw ServiceException(404, "Unable to perform cache fetch, cache was null...")
            }
        }

    override suspend fun checkAggregationCache(
        cacheKey: String,
        typeName: String,
    ): String =
        when {
            isAggregationCacheAvailable() -> {
                when (val content = aggregationCache!![cacheKey]) {
                    null -> throw ServiceException(404, "Cache result is null...")
                    else -> {
                        if (logger.isDebugEnabled()) {
                            logger.debug { "Returning cached content..." }
                        }

                        content
                    }
                }
            }

            else -> throw ServiceException(404, "Cache is null...")
        }

    override suspend fun replaceObjectCache(
        cacheId: String,
        item: E,
        future: Promise<E>,
        projections: Set<String>,
    ) {
        when {
            isObjectCacheAvailable() -> {
                val fullCacheContent = encode(item)
                val jsonRepresentationCache =
                    vertx
                        .executeBlocking(
                            Callable {
                                item.toJsonFormat(projections).encode()
                            },
                        ).coAwait()

                objectCache!!["FULL_CACHE_$cacheId"] = fullCacheContent
                objectCache!![cacheId] = jsonRepresentationCache

                future.complete(item)
            }

            else -> {
                logger.error { "ObjectCache is null, recreating..." }

                future.complete(item)
            }
        }
    }

    override suspend fun replaceCache(
        records: List<E>,
        shortCacheIdSupplier: Function<E, String>,
        cacheIdSupplier: Function<E, String>,
    ) {
        when {
            isObjectCacheAvailable() -> {
                records.forEach { record ->
                    val shortCacheId = shortCacheIdSupplier.apply(record)
                    val cacheId = cacheIdSupplier.apply(record)

                    objectCache!![cacheId] = record.toJsonString()
                    objectCache!![shortCacheId] = record.toJsonString()

                    val secondaryCache = "FULL_CACHE_$cacheId"
                    objectCache!![secondaryCache] = encode(record)
                    objectCache!!["FULL_CACHE_$shortCacheId"] = encode(record)
                }

                purgeSecondaryCaches()
            }

            else -> {
                logger.error { "ObjectCache is null, recreating..." }

                purgeSecondaryCaches()
            }
        }
    }

    override suspend fun replaceItemListCache(
        content: String,
        cacheIdSupplier: Supplier<String>,
    ): Boolean =
        when {
            isItemListCacheAvailable() -> {
                val cacheId = cacheIdSupplier.get()

                itemListCache!![cacheId] = content
                replaceMapValues(itemListKeyMap, cacheId)

                java.lang.Boolean.TRUE
            }

            else -> {
                logger.error { "ItemListCache is null, recreating..." }

                throw ServiceException(500, "Itemlist cache does not exist!")
            }
        }

    override suspend fun replaceAggregationCache(
        content: String,
        cacheIdSupplier: Supplier<String>,
    ): Boolean =
        when {
            isAggregationCacheAvailable() -> {
                val cacheKey = cacheIdSupplier.get()

                aggregationCache!![cacheKey] = content
                replaceMapValues(aggregationKeyMap, cacheKey)

                java.lang.Boolean.TRUE
            }

            else -> {
                logger.error { "AggregationCache is null, recreating..." }

                throw ServiceException(500, "Aggregation cache does not exist!")
            }
        }

    private fun replaceMapValues(
        aggKeyMap: String,
        cacheKey: String,
    ) {
        val map = vertx.sharedData().getLocalMap<String, String>(aggKeyMap)
        var idSet: String? = map[TYPE.simpleName]

        when (idSet) {
            null -> {
                idSet =
                    JsonArray()
                        .add(cacheKey)
                        .encode()

                map[TYPE.simpleName] = idSet
            }

            else -> map.replace(TYPE.simpleName, JsonArray(idSet).add(cacheKey).encode())
        }
    }

    override suspend fun purgeCache(
        records: List<E>,
        cacheIdSupplier: (E) -> String,
    ) {
        when {
            isObjectCacheAvailable() -> {
                records.forEach { record ->
                    val cacheId = cacheIdSupplier(record)
                    val secondaryCache = "FULL_CACHE_$cacheId"

                    objectCache!!.remove(cacheId)
                    objectCache!!.remove(secondaryCache)
                }

                purgeSecondaryCaches()
            }

            else -> {
                logger.error { "ObjectCache is null, recreating..." }

                purgeSecondaryCaches()
            }
        }
    }

    private suspend fun purgeSecondaryCaches() {
        when {
            isItemListCacheAvailable() -> purgeMap(itemListKeyMap, itemListCache)
            else -> logger.error { "ItemListCache is null, recreating..." }
        }

        when {
            isAggregationCacheAvailable() -> purgeMap(aggregationKeyMap, aggregationCache)
            else -> logger.error { "AggregateCache is null, recreating..." }
        }
    }

    private fun purgeMap(
        MAP_KEY: String,
        cache: MutableMap<String, String>?,
    ) {
        try {
            val localMap = vertx.sharedData().getLocalMap<String, String>(MAP_KEY)

            try {
                val cachePartitionKey = TYPE.getDeclaredConstructor().newInstance().cachePartitionKey

                val strings = localMap[cachePartitionKey]

                when {
                    strings != null ->
                        JsonArray(strings)
                            .stream()
                            .map { it.toString() }
                            .forEach { cache!!.remove(it) }

                    else -> localMap[cachePartitionKey] = JsonArray().encode()
                }
            } catch (e: InstantiationException) {
                logger.error(e) { "Unable to build partitionKey" }
            } catch (e: IllegalAccessException) {
                logger.error(e) { "Unable to build partitionKey" }
            }

            if (logger.isDebugEnabled()) {
                logger.debug { "Cache cleared: " + cache!!.size }
            }
        } catch (e: Exception) {
            logger.error(e) { "Error purging cache" }
            logger.error { "Unable to purge cache, nulling..." }

            cache!!.clear()
        }
    }

    companion object {
        private var cachesCreated = false
    }
}
