/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.utils

import com.fasterxml.jackson.annotation.JsonInclude
import com.genghis.tools.repository.models.ModelUtils
import io.vertx.codegen.annotations.DataObject
import io.vertx.codegen.json.annotations.JsonGen
import io.vertx.core.json.JsonArray
import io.vertx.core.json.JsonObject
import io.vertx.core.json.JsonObject.mapFrom

/**
 * This class defines a generic list for items. Used for aggregation.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
@Suppress("unused")
@JsonInclude(JsonInclude.Include.NON_NULL)
@DataObject
@JsonGen
data class GenericItemList(
    val pageToken: PageTokens,
    val count: Int,
    val items: List<JsonObject>,
    val etag: String =
        run {
            val etagCode = longArrayOf(1234567890L)
            items.forEach { item -> etagCode[0] = etagCode[0] xor item.encode().hashCode().toLong() }

            ModelUtils.returnNewEtag(etagCode[0])
        },
) {
    constructor(jsonObject: JsonObject) : this(
        pageToken =
            when (jsonObject.getJsonObject("paging")) {
                null -> PageTokens()
                else -> PageTokens(jsonObject.getJsonObject("paging"))
            },
        count = jsonObject.getInteger("count")!!,
        items = jsonObject.getJsonArray("items").map { it as JsonObject },
        etag = jsonObject.getString("etag"),
    )

    fun toJson(): JsonObject = mapFrom(this)

    @Suppress("UNUSED_PARAMETER")
    fun toJson(projections: Array<String>): JsonObject {
        val jsonObject =
            JsonObject()
                .put("etag", etag)
                .put("paging", pageToken.toJson())
                .put("count", count)

        val jsonItems = JsonArray()

        items.forEach { jsonItems.add(it) }

        jsonObject.put("items", jsonItems)

        return jsonObject
    }

    @JvmOverloads
    fun toJsonString(projections: Array<String> = arrayOf()): String = toJson(projections).encode()
}
