/*
 * MIT License
 *
 * Copyright (c) 2019 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.genghis.tools.version.manager

import com.genghis.tools.version.VersionUtils
import com.genghis.tools.version.models.DiffPair
import com.genghis.tools.version.models.Version
import com.genghis.tools.version.operators.IteratorIdManager
import com.genghis.tools.version.operators.StateApplier
import com.genghis.tools.version.operators.StateExtractor
import io.vertx.core.AbstractVerticle
import io.vertx.core.AsyncResult
import io.vertx.core.Future
import io.vertx.core.Handler
import io.vertx.core.Promise
import io.vertx.core.json.jackson.DatabindCodec
import io.vertx.kotlin.coroutines.awaitResult

class VersionManagerImpl :
    AbstractVerticle(),
    VersionManager {
    private val versionUtils: VersionUtils = VersionUtils()
    private val stateApplier: StateApplier = StateApplier(DatabindCodec.mapper(), versionUtils)
    private val stateExtractor: StateExtractor = StateExtractor(DatabindCodec.mapper(), versionUtils)
    private val iteratorIdManager: IteratorIdManager = IteratorIdManager(versionUtils)

    override fun <T : Any> applyState(
        version: Version,
        obj: T,
        handler: Handler<AsyncResult<T>>,
    ): VersionManagerImpl {
        handler.handle(applyState(version, obj).future())

        return this
    }

    override fun <T : Any> applyState(
        version: Version,
        obj: T,
    ): Promise<T> {
        val fut = Promise.promise<T>()

        stateApplier.applyState(version, obj, fut)

        return fut
    }

    @Throws(IllegalStateException::class)
    override suspend fun <T : Any> applyStateAwait(
        version: Version,
        obj: T,
    ): T = awaitResult { applyState(version, obj, it) }

    override fun <T : Any> applyState(
        versionList: List<Version>,
        obj: T,
        handler: Handler<AsyncResult<T>>,
    ): VersionManagerImpl {
        handler.handle(applyState(versionList, obj).future())

        return this
    }

    override fun <T : Any> applyState(
        versionList: List<Version>,
        obj: T,
    ): Promise<T> {
        val fut = Promise.promise<T>()
        val futList = mutableListOf<Promise<T>>()

        versionList.forEach {
            futList.add(applyState(it, obj))
        }

        Future.all<T>(futList.map { it.future() }).andThen {
            when {
                it.failed() -> fut.fail(it.cause())
                it.succeeded() -> fut.complete(obj)
            }
        }

        return fut
    }

    @Throws(IllegalStateException::class)
    override suspend fun <T : Any> applyStateAwait(
        versionList: List<Version>,
        obj: T,
    ): T = awaitResult { applyState(versionList, obj, it) }

    override fun <T : Any> applyState(
        versionMap: Map<T, List<Version>>,
        objs: List<T>,
        handler: Handler<AsyncResult<List<T>>>,
    ): VersionManagerImpl {
        handler.handle(applyState(versionMap, objs).future())

        return this
    }

    override fun <T : Any> applyState(
        versionMap: Map<T, List<Version>>,
        objs: List<T>,
    ): Promise<List<T>> {
        val fut = Promise.promise<List<T>>()
        val futList = mutableListOf<Promise<T>>()

        objs.forEach {
            futList.add(applyState(versionMap.getValue(it), it))
        }

        Future.all<T>(futList.map { it.future() }).andThen {
            when {
                it.failed() -> fut.fail(it.cause())
                it.succeeded() -> fut.complete(objs)
            }
        }

        return fut
    }

    @Throws(IllegalStateException::class)
    override suspend fun <T : Any> applyStateAwait(
        versionMap: Map<T, List<Version>>,
        objs: List<T>,
    ): List<T> = awaitResult { applyState(versionMap, objs, it) }

    override fun <T : Any> extractVersion(
        pair: DiffPair<T>,
        handler: Handler<AsyncResult<Version>>,
    ): VersionManagerImpl {
        handler.handle(extractVersion(pair).future())

        return this
    }

    override fun <T : Any> extractVersion(pair: DiffPair<T>): Promise<Version> {
        val fut = Promise.promise<Version>()

        stateExtractor.extractVersion(pair, fut)

        return fut
    }

    @Throws(IllegalStateException::class)
    override suspend fun <T : Any> extractVersionAwait(pair: DiffPair<T>): Version = awaitResult { extractVersion(pair, it) }

    override fun <T : Any> extractVersion(
        pairs: List<DiffPair<T>>,
        handler: Handler<AsyncResult<List<Version>>>,
    ): VersionManagerImpl {
        handler.handle(extractVersion(pairs).future())

        return this
    }

    override fun <T : Any> extractVersion(pairs: List<DiffPair<T>>): Promise<List<Version>> {
        val fut = Promise.promise<List<Version>>()
        val futList = mutableListOf<Promise<Version>>()

        pairs.forEach {
            futList.add(extractVersion(it))
        }

        Future.all<T>(futList.map { it.future() }).andThen { result ->
            when {
                result.failed() -> fut.fail(result.cause())
                result.succeeded() -> fut.complete(futList.map { it.future().result() } as List<Version>?)
            }
        }

        return fut
    }

    @Throws(IllegalStateException::class)
    override suspend fun <T : Any> extractVersionAwait(pairs: List<DiffPair<T>>): List<Version> = awaitResult { extractVersion(pairs, it) }

    override fun <T : Any> setIteratorIds(
        obj: T,
        handler: Handler<AsyncResult<T>>,
    ): VersionManagerImpl {
        handler.handle(setIteratorIds(obj).future())

        return this
    }

    override fun <T : Any> setIteratorIds(obj: T): Promise<T> {
        val fut = Promise.promise<T>()

        setIteratorIds(listOf(obj)).future().andThen {
            when {
                it.failed() -> fut.fail(it.cause())
                it.succeeded() -> fut.complete(it.result()[0])
            }
        }

        return fut
    }

    @Throws(IllegalStateException::class)
    override suspend fun <T : Any> setIteratorIdsAwait(obj: T): T = awaitResult { setIteratorIds(obj, it) }

    override fun <T : Any> setIteratorIds(
        objs: List<T>,
        handler: Handler<AsyncResult<List<T>>>,
    ): VersionManagerImpl {
        handler.handle(setIteratorIds(objs).future())

        return this
    }

    override fun <T : Any> setIteratorIds(objs: List<T>): Promise<List<T>> {
        val fut = Promise.promise<List<T>>()

        @Suppress("UNCHECKED_CAST")
        iteratorIdManager.setIteratorIds(objs) {
            when {
                it.failed() -> fut.fail(it.cause())
                it.succeeded() -> fut.complete(it.result() as List<T>)
            }
        }

        return fut
    }

    @Throws(IllegalStateException::class)
    override suspend fun <T : Any> setIteratorIdsAwait(objs: List<T>): List<T> = awaitResult { setIteratorIds(objs, it) }
}
