/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.web

import com.genghis.tools.web.controllers.logger
import com.genghis.tools.web.requestHandlers.RequestLogHandler
import com.genghis.tools.web.responsehandlers.ResponseLogHandler
import com.genghis.tools.web.responsehandlers.ResponseLogHandler.Companion.BODY_CONTENT_TAG
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.Handler
import io.vertx.core.http.HttpMethod
import io.vertx.core.json.JsonObject
import io.vertx.ext.web.Route
import io.vertx.ext.web.RoutingContext
import io.vertx.ext.web.handler.BodyHandler
import io.vertx.ext.web.handler.ResponseContentTypeHandler
import io.vertx.ext.web.handler.ResponseTimeHandler
import io.vertx.kotlin.coroutines.CoroutineRouterSupport
import io.vertx.serviceproxy.ServiceException
import java.io.UnsupportedEncodingException
import java.net.URLDecoder
import java.util.AbstractMap.SimpleImmutableEntry
import java.util.Arrays
import java.util.concurrent.TimeUnit.NANOSECONDS
import java.util.stream.Collectors.groupingBy
import java.util.stream.Collectors.mapping
import java.util.stream.Collectors.toList

@Suppress("unused")
private val logger = KotlinLogging.logger {}

/**
 * This class contains helper methods for routing requests.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
object RoutingHelper {
    private const val DATABASE_PROCESS_TIME = "X-Database-Time-To-Process"

    private val requestLogger = RequestLogHandler()
    private val responseLogger = ResponseLogHandler()

    private val bodyHandler: () -> BodyHandler = {
        BodyHandler
            .create()
            .setMergeFormAttributes(true)
            .setHandleFileUploads(false)
    }
    private val timeOutHandler =
        Handler<RoutingContext> {
            it.vertx().setTimer(9000L) { _ -> if (!it.request().isEnded) it.fail(503) }
            it.next()
        }

    private val responseContentTypeHandler = ResponseContentTypeHandler.create()
    private val responseTimeHandler = ResponseTimeHandler.create()

    @Suppress("unused")
    fun setStatusCode(
        code: Int,
        routingContext: RoutingContext,
        initialProcessTime: Long,
    ) {
        runCatching {
            setDatabaseProcessTime(routingContext, initialProcessTime)
            routingContext.response().statusCode = code
        }.onFailure {
            logger.warn(it) { "Failed setting status code" }
        }
    }

    fun setStatusCodeAndAbort(
        code: Int,
        routingContext: RoutingContext,
        initialProcessTime: Long,
    ) {
        runCatching {
            setDatabaseProcessTime(routingContext, initialProcessTime)
            routingContext.response().statusMessage = routingContext.get(BODY_CONTENT_TAG) ?: ""
            routingContext.fail(code)
        }.onFailure {
            logger.warn(it) { "Failed setting status code and aborting, aborting directly..." }

            routingContext.response().statusCode = 500
            routingContext.fail(code)
        }
    }

    @Suppress("unused")
    fun setStatusCodeAndContinue(
        code: Int,
        routingContext: RoutingContext,
    ) {
        runCatching {
            routingContext.response().statusCode = code
            routingContext.next()
        }.onFailure {
            logger.warn(it) { "Failed setting status code and continuing, continuing directly..." }

            routingContext.next()
        }
    }

    fun setStatusCodeAndContinue(
        code: Int,
        routingContext: RoutingContext,
        initialProcessTime: Long,
    ) {
        runCatching {
            setDatabaseProcessTime(routingContext, initialProcessTime)
            routingContext.response().statusCode = code
            routingContext.next()
        }.onFailure {
            logger.warn(it) { "Failed setting status code with process time and continuing, continuing directly..." }

            routingContext.next()
        }
    }

    private fun setDatabaseProcessTime(
        routingContext: RoutingContext,
        initialTime: Long,
    ) {
        runCatching {
            val processTimeInNano = System.nanoTime() - initialTime

            routingContext
                .response()
                .putHeader(DATABASE_PROCESS_TIME, NANOSECONDS.toMillis(processTimeInNano).toString())
        }.onFailure {
            logger.warn(it) { "Failed setting database processing time" }
        }
    }

    @Suppress("unused")
    fun CoroutineRouterSupport.routeWithAuth(
        routeProducer: () -> Route,
        authHandler: suspend (RoutingContext) -> Unit,
        routeSetter: (() -> Route) -> Unit,
    ) {
        runCatching {
            routeWithAuth(routeProducer, authHandler, {}, routeSetter)
        }.onFailure {
            logger.warn(it) { "Failed setting authed route" }
        }
    }

    @Suppress("SameParameterValue")
    private fun CoroutineRouterSupport.routeWithAuth(
        routeProducer: () -> Route,
        authHandler: suspend (RoutingContext) -> Unit,
        finallyHandler: suspend (RoutingContext) -> Unit = {},
        routeSetter: (() -> Route) -> Unit,
    ) {
        runCatching {
            prependStandards(routeProducer)
            routeProducer().coHandler(requestHandler = authHandler)
            routeSetter(routeProducer)
            appendStandards(routeProducer, finallyHandler)
        }.onFailure {
            logger.warn(it) { "Failed setting authed route" }
        }
    }

    @Suppress("unused")
    fun CoroutineRouterSupport.routeWithBodyHandlerAndAuth(
        routeProducer: () -> Route,
        authHandler: suspend (RoutingContext) -> Unit,
        routeSetter: (() -> Route) -> Unit,
    ) {
        runCatching {
            routeWithBodyHandlerAndAuth(routeProducer, authHandler, {}, routeSetter)
        }.onFailure {
            logger.warn(it) { "Failed setting authed body route" }
        }
    }

    @Suppress("SameParameterValue")
    private fun CoroutineRouterSupport.routeWithBodyHandlerAndAuth(
        routeProducer: () -> Route,
        authHandler: suspend (RoutingContext) -> Unit,
        finallyHandler: suspend (RoutingContext) -> Unit = {},
        routeSetter: (() -> Route) -> Unit,
        bodyHandlerConfigurator: BodyHandler.() -> Unit = {},
    ) {
        runCatching {
            val bodyHandler = bodyHandler()
            bodyHandlerConfigurator(bodyHandler)
            prependStandards(routeProducer)
            routeProducer().handler(bodyHandler)
            routeProducer().coHandler(requestHandler = authHandler)
            routeSetter(routeProducer)
            appendStandards(routeProducer, finallyHandler)
        }.onFailure {
            logger.warn(it) { "Failed setting authed body route" }
        }
    }

    private fun prependStandards(routeProducer: () -> Route) {
        routeProducer().handler(responseTimeHandler)
        routeProducer().handler(timeOutHandler)
        routeProducer().handler(responseContentTypeHandler)
        routeProducer().handler(requestLogger)
    }

    private fun CoroutineRouterSupport.appendStandards(
        routeProducer: () -> Route,
        finallyHandler: suspend (RoutingContext) -> Unit = {},
    ) {
        finallyHandler.let { routeProducer().coHandler(requestHandler = it) }
        routeProducer().handler(responseLogger)
        routeProducer().failureHandler { handleErrors(it) }
    }

    fun CoroutineRouterSupport.routeWithLogger(
        routeProducer: () -> Route,
        routeSetter: (() -> Route) -> Unit,
    ) {
        runCatching {
            routeWithLogger(routeProducer, routeSetter) {}
        }.onFailure {
            logger.warn(it) { "Failed setting logged route" }
        }
    }

    fun CoroutineRouterSupport.routeWithBodyAndLogger(
        routeProducer: () -> Route,
        routeSetter: (() -> Route) -> Unit,
        finallyHandler: suspend (RoutingContext) -> Unit = {},
        bodyHandlerConfigurator: BodyHandler.() -> Unit = {},
    ) {
        runCatching {
            val bodyHandler = bodyHandler()
            bodyHandlerConfigurator(bodyHandler)
            routeProducer().handler(responseTimeHandler)
            routeProducer().handler(timeOutHandler)
            routeProducer().handler(responseContentTypeHandler)
            routeProducer().handler(requestLogger)
            routeProducer().handler(bodyHandler)
            routeSetter(routeProducer)
            finallyHandler.let { routeProducer().coHandler(requestHandler = it) }
            routeProducer().handler(responseLogger)
            routeProducer().failureHandler { handleErrors(it) }
        }.onFailure {
            logger.warn(it) { "Failed setting authed route" }
        }
    }

    fun CoroutineRouterSupport.routeWithLogger(
        routeProducer: () -> Route,
        routeSetter: (() -> Route) -> Unit,
        finallyHandler: suspend (RoutingContext) -> Unit = {},
    ) {
        runCatching {
            routeProducer().handler(responseTimeHandler)
            routeProducer().handler(timeOutHandler)
            routeProducer().handler(responseContentTypeHandler)
            routeProducer().handler(requestLogger)
            routeSetter(routeProducer)
            finallyHandler.let { routeProducer().coHandler(requestHandler = it) }
            routeProducer().handler(responseLogger)
            routeProducer().failureHandler { handleErrors(it) }
        }.onFailure {
            logger.warn(it) { "Failed setting logged route" }
        }
    }

    private fun handleErrors(routingContext: RoutingContext) {
        runCatching {
            val throwable = routingContext.failure()
            val statusCode =
                when {
                    throwable is ServiceException -> throwable.failureCode()
                    else -> routingContext.statusCode()
                }

            routingContext.response().statusCode = if (statusCode != -1) statusCode else 500

            responseLogger.handle(routingContext)
        }.onFailure {
            logger.warn(it) { "Failure in http error processing" }

            routingContext.response().statusCode = 500
            routingContext.end()
        }
    }

    fun denyQuery(routingContext: RoutingContext): Boolean =
        runCatching {
            val query = routingContext.request().query()

            return when {
                query != null && !routingContext.request().method().equals(HttpMethod.GET) -> true
                query != null ->
                    invalidQuery(
                        query,
                        routingContext,
                    )

                else -> false
            }
        }.getOrElse {
            logger.warn(it) { "Failure in deny query, denying by default" }

            true
        }

    private fun invalidQuery(
        query: String,
        routingContext: RoutingContext,
    ): Boolean =
        runCatching {
            val queryMap = splitQuery(query)

            return when {
                unqueryAble(queryMap) -> {
                    routingContext.put(
                        BODY_CONTENT_TAG,
                        JsonObject()
                            .put("query_error", "No query accepted for this route"),
                    )
                    routingContext.fail(400)

                    true
                }

                queryMap.isEmpty() -> {
                    routingContext.put(
                        BODY_CONTENT_TAG,
                        JsonObject()
                            .put("query_error", "Cannot parse this query string, are you sure it is in UTF-8?"),
                    )
                    routingContext.fail(400)

                    true
                }

                else -> false
            }
        }.getOrElse {
            logger.warn(it) { "Failure in invalid query, invalid by default" }

            true
        }

    private fun unqueryAble(queryMap: MutableMap<String, List<String>>) =
        queryMap.size > 2 ||
            (queryMap.size == 1 || queryMap.size == 2) &&
            (queryMap["projection"] == null && queryMap["globalHashIndex"] == null)

    fun splitQuery(query: String): MutableMap<String, List<String>> {
        val decoded: String

        try {
            decoded = URLDecoder.decode(query, "UTF-8")
        } catch (e: UnsupportedEncodingException) {
            logger.error(e) { "Failed URL Decoding" }

            return mutableMapOf()
        }

        return Arrays
            .stream(decoded.split("&".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray())
            .map { splitQueryParameter(it) }
            .collect(
                groupingBy(
                    { it.key },
                    { mutableMapOf() },
                    mapping({ it.value }, toList()),
                ),
            )
    }

    private fun splitQueryParameter(it: String): SimpleImmutableEntry<String, String> {
        val idx = it.indexOf("=")
        val key = if (idx > 0) it.substring(0, idx) else it
        val value = if (idx > 0 && it.length > idx + 1) it.substring(idx + 1) else null

        return SimpleImmutableEntry<String, String>(key, value)
    }
}
