/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

@file:OptIn(DelicateCoroutinesApi::class)

package com.genghis.tools.web.controllers

import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.models.Model.Companion.buildValidationErrorObject
import com.genghis.tools.repository.models.ModelUtils
import com.genghis.tools.repository.repository.etag.ETagManager
import com.genghis.tools.repository.utils.ItemList
import com.genghis.tools.repository.utils.Query
import com.genghis.tools.repository.utils.QueryBuilder
import com.genghis.tools.web.RoutingHelper.denyQuery
import com.genghis.tools.web.RoutingHelper.setStatusCodeAndAbort
import com.genghis.tools.web.RoutingHelper.setStatusCodeAndContinue
import com.genghis.tools.web.requestHandlers.RequestLogHandler.Companion.REQUEST_PROCESS_TIME_TAG
import com.genghis.tools.web.requestHandlers.RequestLogHandler.Companion.addLogMessageToRequestLog
import com.genghis.tools.web.responsehandlers.ResponseLogHandler.Companion.BODY_CONTENT_TAG
import io.vertx.core.Vertx
import io.vertx.core.http.HttpHeaders.CONTENT_TYPE
import io.vertx.core.json.Json.encode
import io.vertx.core.json.JsonArray
import io.vertx.core.json.JsonObject
import io.vertx.ext.web.RoutingContext
import kotlinx.coroutines.DelicateCoroutinesApi
import org.apache.commons.lang3.ArrayUtils
import java.lang.reflect.Field
import java.lang.reflect.Method
import java.time.Instant

/**
 * This interface defines the RestController. It defines a chain of operations for CRUD and Index operations. Overriding
 * functions must remember to call the next element in the chain.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
@Suppress("unused")
interface RestController<E : Model, Q : Query, QB : QueryBuilder> {
    val vertx: Vertx
    val etagManager: ETagManager<*>?

    suspend fun show(routingContext: RoutingContext) {
        runCatching {
            preShow(routingContext)
        }.onFailure { throwable ->
            runCatching {
                addLogMessageToRequestLog(routingContext, "Error in Show!", throwable)
            }.onFailure {
                logger.warn(it) { "Unable to add error log for show exception" }
            }

            if (!routingContext.failed()) {
                routingContext.fail(throwable)
            }
        }.getOrNull()
    }

    suspend fun preShow(routingContext: RoutingContext) {
        performShow(routingContext)
    }

    suspend fun performShow(routingContext: RoutingContext)

    suspend fun postShow(
        routingContext: RoutingContext,
        item: E,
        projections: Set<String>?,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val requestEtag = routingContext.request().getHeader("If-None-Match")

        when {
            requestEtag != null && item is ETagable && requestEtag == item.etag ->
                unChangedShow(routingContext)

            else -> {
                routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
                routingContext.put(
                    BODY_CONTENT_TAG,
                    if (projections != null) item.toJsonString(projections) else item.toJsonString(),
                )

                setStatusCodeAndContinue(200, routingContext, initialNanoTime)
            }
        }
    }

    suspend fun unChangedShow(routingContext: RoutingContext) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        setStatusCodeAndContinue(304, routingContext, initialNanoTime)
    }

    suspend fun notFoundShow(routingContext: RoutingContext) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        setStatusCodeAndAbort(404, routingContext, initialNanoTime)
    }

    suspend fun failedShow(
        routingContext: RoutingContext,
        debugInformation: JsonObject?,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        routingContext.put(BODY_CONTENT_TAG, debugInformation?.encode())
        setStatusCodeAndAbort(500, routingContext, initialNanoTime)
    }

    suspend fun index(routingContext: RoutingContext) {
        runCatching {
            preIndex(routingContext, null)
        }.onFailure { throwable ->
            runCatching {
                addLogMessageToRequestLog(routingContext, "Error in Index!", throwable)
            }.onFailure {
                logger.warn(it) { "Unable to add error log for index exception" }
            }

            if (!routingContext.failed()) {
                routingContext.fail(throwable)
            }
        }.getOrNull()
    }

    suspend fun index(
        routingContext: RoutingContext,
        customQuery: String,
    ) = runCatching {
        preIndex(routingContext, customQuery)
    }.onFailure { throwable ->
        runCatching {
            addLogMessageToRequestLog(routingContext, "Error in Index!", throwable)
        }.onFailure {
            logger.warn(it) { "Unable to add error log for index exception" }
        }

        if (!routingContext.failed()) {
            routingContext.fail(throwable)
        }
    }

    suspend fun preIndex(
        routingContext: RoutingContext,
        customQuery: String?,
    ) {
        prepareQuery(routingContext, customQuery)
    }

    suspend fun prepareQuery(
        routingContext: RoutingContext,
        customQuery: String?,
    )

    suspend fun preProcessQuery(
        routingContext: RoutingContext,
        queryMap: MutableMap<String, List<String>>,
    ) {
        processQuery(routingContext, queryMap)
    }

    suspend fun processQuery(
        routingContext: RoutingContext,
        queryMap: MutableMap<String, List<String>>,
    )

    suspend fun postProcessQuery(
        routingContext: RoutingContext,
        query: Q,
    ) {
        postPrepareQuery(routingContext, query)
    }

    suspend fun postPrepareQuery(
        routingContext: RoutingContext,
        query: Q,
    ) {
        createIdObjectForIndex(routingContext, query)
    }

    suspend fun createIdObjectForIndex(
        routingContext: RoutingContext,
        query: Q,
    )

    suspend fun performIndex(
        routingContext: RoutingContext,
        query: Q,
    )

    suspend fun proceedWithPagedIndex(
        routingContext: RoutingContext,
        query: Q,
    )

    suspend fun proceedWithAggregationIndex(
        routingContext: RoutingContext,
        query: Q,
    )

    suspend fun postIndex(
        routingContext: RoutingContext,
        items: ItemList<E>,
        projections: Set<String>?,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val requestEtag = routingContext.request().getHeader("If-None-Match")

        when {
            requestEtag != null && requestEtag == items.meta.etag -> unChangedIndex(routingContext)
            else -> {
                val content = items.toJsonString(projections ?: emptySet())

                routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
                routingContext.put(BODY_CONTENT_TAG, content)

                setStatusCodeAndContinue(200, routingContext, initialNanoTime)
            }
        }
    }

    suspend fun postAggregation(
        routingContext: RoutingContext,
        content: String,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val requestEtag = routingContext.request().getHeader("If-None-Match")

        when {
            requestEtag != null && requestEtag == ModelUtils.returnNewEtag(content.hashCode().toLong()) ->
                unChangedIndex(routingContext)

            else -> {
                routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
                routingContext.put(BODY_CONTENT_TAG, content)

                setStatusCodeAndContinue(200, routingContext, initialNanoTime)
            }
        }
    }

    suspend fun unChangedIndex(routingContext: RoutingContext) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        setStatusCodeAndContinue(304, routingContext, initialNanoTime)
    }

    suspend fun failedIndex(
        routingContext: RoutingContext,
        debugInformation: JsonObject?,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        routingContext.put(BODY_CONTENT_TAG, debugInformation?.encode())
        setStatusCodeAndAbort(500, routingContext, initialNanoTime)
    }

    suspend fun create(routingContext: RoutingContext) {
        runCatching {
            preCreate(routingContext)
        }.onFailure { throwable ->
            runCatching {
                addLogMessageToRequestLog(routingContext, "Error in Create!", throwable)
            }.onFailure {
                logger.warn(it) { "Unable to add error log for create exception" }
            }

            if (!routingContext.failed()) {
                routingContext.fail(throwable)
            }
        }.getOrNull()
    }

    suspend fun preCreate(routingContext: RoutingContext) {
        if (denyQuery(routingContext)) return

        parseBodyForCreate(routingContext)
    }

    suspend fun parseBodyForCreate(routingContext: RoutingContext)

    suspend fun preVerifyNotExists(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        verifyNotExists(newRecords, routingContext)
    }

    suspend fun verifyNotExists(
        newRecords: List<E>,
        routingContext: RoutingContext,
    )

    suspend fun postVerifyNotExists(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        preSetIdentifiers(newRecords, routingContext)
    }

    suspend fun preSetIdentifiers(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        setIdentifiers(newRecords, routingContext)
    }

    suspend fun setIdentifiers(
        newRecords: List<E>,
        routingContext: RoutingContext,
    )

    suspend fun preSanitizeForCreate(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        performSanitizeForCreate(newRecords, routingContext)
    }

    suspend fun performSanitizeForCreate(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        newRecords.forEach { it.sanitize() }

        postSanitizeForCreate(newRecords, routingContext)
    }

    suspend fun postSanitizeForCreate(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        preValidateForCreate(newRecords, routingContext)
    }

    suspend fun preValidateForCreate(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        performValidateForCreate(newRecords, routingContext)
    }

    suspend fun performValidateForCreate(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val errors = newRecords.map { it.validateCreate() }.flatten()

        when {
            errors.isNotEmpty() -> {
                routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
                routingContext.put(BODY_CONTENT_TAG, encode(buildValidationErrorObject(errors)))
                setStatusCodeAndAbort(422, routingContext, initialNanoTime)
            }

            else -> postValidateForCreate(newRecords, routingContext)
        }
    }

    suspend fun postValidateForCreate(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        performCreate(newRecords, routingContext)
    }

    suspend fun performCreate(
        newRecords: List<E>,
        routingContext: RoutingContext,
    )

    suspend fun postCreate(
        createdRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val content =
            when {
                createdRecords.size == 1 && !routingContext.get("MULTI_BODY", false) ->
                    createdRecords.first().toJsonString()

                else -> encode(JsonArray(createdRecords.map { it.toJsonFormat() }))
            }

        routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
        routingContext.put(BODY_CONTENT_TAG, content)
        setStatusCodeAndContinue(201, routingContext, initialNanoTime)
    }

    suspend fun failedCreate(
        routingContext: RoutingContext,
        userFeedBack: JsonObject?,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        if (userFeedBack != null) {
            routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
            routingContext.put(BODY_CONTENT_TAG, userFeedBack.encode())
        }

        setStatusCodeAndContinue(500, routingContext, initialNanoTime)
    }

    suspend fun update(routingContext: RoutingContext) {
        runCatching {
            preUpdate(routingContext)
        }.onFailure { throwable ->
            runCatching {
                addLogMessageToRequestLog(routingContext, "Error in Update!", throwable)
            }.onFailure {
                logger.warn(it) { "Unable to add error log for update exception" }
            }

            if (!routingContext.failed()) {
                routingContext.fail(throwable)
            }
        }.getOrNull()
    }

    suspend fun preUpdate(routingContext: RoutingContext) {
        if (denyQuery(routingContext)) return

        parseBodyForUpdate(routingContext)
    }

    suspend fun parseBodyForUpdate(routingContext: RoutingContext)

    suspend fun preVerifyExistsForUpdate(
        updateRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        verifyExistsForUpdate(updateRecords, routingContext)
    }

    suspend fun verifyExistsForUpdate(
        updateRecords: List<E>,
        routingContext: RoutingContext,
    )

    suspend fun postVerifyExistsForUpdate(
        updateMap: Map<E, E>,
        routingContext: RoutingContext,
    ) {
        preSanitizeForUpdate(updateMap, routingContext)
    }

    suspend fun preSanitizeForUpdate(
        updateMap: Map<E, E>,
        routingContext: RoutingContext,
    ) {
        performSanitizeForUpdate(updateMap, routingContext)
    }

    suspend fun performSanitizeForUpdate(
        updateMap: Map<E, E>,
        routingContext: RoutingContext,
    ) {
        val updateFuncMap =
            updateMap
                .map {
                    val setNewValues: suspend E.(E) -> Unit = { newModel ->
                        setModifiables(newModel)
                        sanitize()
                    }

                    it.value to setNewValues
                }.toMap()

        postSanitizeForUpdate(updateFuncMap, routingContext)
    }

    suspend fun postSanitizeForUpdate(
        updateFuncMap: Map<E, suspend E.(E) -> Unit>,
        routingContext: RoutingContext,
    ) {
        preValidateForUpdate(updateFuncMap, routingContext)
    }

    suspend fun preValidateForUpdate(
        updateFuncMap: Map<E, suspend E.(E) -> Unit>,
        routingContext: RoutingContext,
    ) {
        performValidateForUpdate(updateFuncMap, routingContext)
    }

    suspend fun performValidateForUpdate(
        updateFuncMap: Map<E, suspend E.(E) -> Unit>,
        routingContext: RoutingContext,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val updateTime = Instant.now().toEpochMilli()

        val errors =
            updateFuncMap
                .map {
                    it.key.updatedAt = updateTime
                    it.key.validateUpdate()
                }.flatten()

        when {
            errors.isNotEmpty() -> {
                routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
                routingContext.put(BODY_CONTENT_TAG, encode(buildValidationErrorObject(errors)))
                setStatusCodeAndAbort(422, routingContext, initialNanoTime)
            }

            else -> postValidateForUpdate(updateFuncMap, routingContext)
        }
    }

    suspend fun postValidateForUpdate(
        updateFuncMap: Map<E, suspend E.(E) -> Unit>,
        routingContext: RoutingContext,
    ) {
        performUpdate(updateFuncMap, routingContext)
    }

    suspend fun performUpdate(
        updateFuncMap: Map<E, suspend E.(E) -> Unit>,
        routingContext: RoutingContext,
    )

    suspend fun postUpdate(
        updatedRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val content =
            when {
                updatedRecords.size == 1 && !routingContext.get("MULTI_BODY", false) ->
                    updatedRecords.first().toJsonString()
                else -> encode(JsonArray(updatedRecords.map { it.toJsonFormat() }))
            }

        routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
        routingContext.put(BODY_CONTENT_TAG, content)
        setStatusCodeAndContinue(200, routingContext, initialNanoTime)
    }

    suspend fun failedUpdate(
        routingContext: RoutingContext,
        userFeedBack: JsonObject?,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        if (userFeedBack != null) {
            routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
            routingContext.put(BODY_CONTENT_TAG, userFeedBack.encode())
        }

        setStatusCodeAndContinue(500, routingContext, initialNanoTime)
    }

    suspend fun destroy(routingContext: RoutingContext) {
        runCatching {
            preDestroy(routingContext)
        }.onFailure { throwable ->
            runCatching {
                addLogMessageToRequestLog(routingContext, "Error in Destroy!", throwable)
            }.onFailure {
                logger.warn(it) { "Unable to add error log for destroy exception" }
            }

            if (!routingContext.failed()) {
                routingContext.fail(throwable)
            }
        }.getOrNull()
    }

    suspend fun preDestroy(routingContext: RoutingContext) {
        if (denyQuery(routingContext)) return

        verifyExistsForDestroy(routingContext)
    }

    suspend fun verifyExistsForDestroy(routingContext: RoutingContext)

    suspend fun postVerifyExistsForDestroy(
        recordsForDestroy: List<E>,
        routingContext: RoutingContext,
    ) {
        performDestroy(recordsForDestroy, routingContext)
    }

    suspend fun performDestroy(
        recordsForDestroy: List<E>,
        routingContext: RoutingContext,
    )

    suspend fun postDestroy(
        recordsForDestroy: List<E>,
        routingContext: RoutingContext,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        setStatusCodeAndContinue(204, routingContext, initialNanoTime)
    }

    suspend fun failedDestroy(
        routingContext: RoutingContext,
        userFeedBack: JsonObject?,
    ) {
        val initialNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        if (userFeedBack != null) {
            routingContext.response().putHeader(CONTENT_TYPE, "application/json; charset=utf-8")
            routingContext.put(BODY_CONTENT_TAG, userFeedBack.encode())
        }

        setStatusCodeAndContinue(500, routingContext, initialNanoTime)
    }

    fun getAllFieldsOnType(klazz: Class<*>): Array<Field> {
        val fields = klazz.declaredFields

        return if (klazz.superclass != null && klazz.superclass != Any::class.java) {
            ArrayUtils.addAll(fields, *getAllFieldsOnType(klazz.superclass))
        } else {
            fields
        }
    }

    fun getAllMethodsOnType(klazz: Class<*>): Array<Method> {
        val methods = klazz.declaredMethods

        return if (klazz.superclass != null && klazz.superclass != Any::class.java) {
            ArrayUtils.addAll(methods, *getAllMethodsOnType(klazz.superclass))
        } else {
            methods
        }
    }
}
