/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.web.responsehandlers

import com.genghis.tools.web.controllers.logger
import com.genghis.tools.web.requestHandlers.RequestLogHandler.Companion.REQUEST_ID_TAG
import com.genghis.tools.web.requestHandlers.RequestLogHandler.Companion.REQUEST_LOG_TAG
import com.genghis.tools.web.requestHandlers.RequestLogHandler.Companion.REQUEST_PROCESS_TIME_TAG
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.Handler
import io.vertx.core.http.HttpHeaders.CACHE_CONTROL
import io.vertx.core.json.DecodeException
import io.vertx.core.json.Json
import io.vertx.core.json.JsonObject
import io.vertx.ext.web.RoutingContext
import java.util.concurrent.TimeUnit

@Suppress("unused")
private val logger = KotlinLogging.logger { }

/**
 * This interface defines the ResponseLogHandler. It starts the logging process, to be concluded by the
 * responseloghandler.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
class ResponseLogHandler : Handler<RoutingContext> {
    override fun handle(routingContext: RoutingContext) {
        runCatching {
            val uniqueToken =
                runCatching {
                    routingContext.get<String>(REQUEST_ID_TAG)
                }.getOrNull()
            val statusCode = routingContext.response().statusCode
            val body =
                runCatching {
                    routingContext.get<Any>(BODY_CONTENT_TAG)
                }.getOrNull()
            val debug =
                runCatching {
                    routingContext.get<Any>(DEBUG_INFORMATION_OBJECT)
                }.getOrNull()
            val totalProcessTime =
                runCatching {
                    routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
                }.getOrNull()?.let {
                    val processTimeInNano = System.nanoTime() - it

                    TimeUnit.NANOSECONDS.toMillis(processTimeInNano)
                }

            routingContext.response().putHeader(CACHE_CONTROL.toString(), "max-age=0, must-revalidate")

            uniqueToken?.let {
                routingContext.response().putHeader("X-genghis-Debug", it)
            }
            totalProcessTime?.let {
                routingContext.response().putHeader("X-Internal-Time-To-Process", it.toString())
            }

            if (statusCode >= 400 || statusCode == 202) {
                addCorsHeaders(routingContext)
            }

            val sb = buildLogs(routingContext, statusCode, uniqueToken, body, debug)

            outputLog(statusCode, sb)

            when (body) {
                null -> routingContext.response().end()
                else -> routingContext.response().end((body as? String)?.toString() ?: Json.encode(body))
            }
        }.onFailure {
            logger.warn(it) { "Failed response logging" }

            routingContext.response().end()
        }.getOrNull()
    }

    private fun addCorsHeaders(routingContext: RoutingContext) {
        routingContext.response().putHeader("Access-Control-Allow-Origin", "*")
        routingContext.response().putHeader("Access-Control-Allow-Credentials", "false")
        routingContext.response().putHeader(
            "Access-Control-Allow-Methods",
            "POST, GET, PUT, DELETE, OPTIONS",
        )
        routingContext.response().putHeader(
            "Access-Control-Allow-Headers",
            "DNT,Authorization,X-Real-IP,X-Forwarded-For,Keep-Alive,User-Agent," +
                "X-Requested-With,If-None-Match,Cache-Control,Content-Type",
        )
    }

    companion object {
        const val BODY_CONTENT_TAG = "bodyContent"
        const val DEBUG_INFORMATION_OBJECT = "debugInfo"

        fun buildLogs(
            routingContext: RoutingContext,
            statusCode: Int,
            uniqueToken: String?,
            body: Any?,
            debug: Any?,
        ): StringBuffer {
            val stringBuilder = routingContext.get<StringBuffer>(REQUEST_LOG_TAG)
            val sb = stringBuilder ?: StringBuffer()

            sb
                .append("\n--- ")
                .append("Logging Frame End: ")
                .append(uniqueToken)
                .append(" ---\n")
            sb
                .append("\n--- ")
                .append("Request Frame : ")
                .append(statusCode)
                .append(" ---\n")
            sb.append("\nHeaders:\n")

            routingContext.request().headers().forEach(appendRequestHeaders(sb))

            var bodyObject: JsonObject? = null
            var requestBody: String? = null

            try {
                requestBody = routingContext.body().asString()

                if (routingContext.body() != null &&
                    routingContext
                        .body()
                        .buffer()
                        ?.bytes
                        ?.isNotEmpty() == true
                ) {
                    try {
                        bodyObject = JsonObject(requestBody)
                    } catch (ignored: DecodeException) {
                    }
                }
            } catch (e: Exception) {
                logger.debug(e) { "${"Parse exception!"}" }
            }

            sb.append("\nRequest Body:\n").append(if (bodyObject == null) requestBody else bodyObject.encodePrettily())
            sb
                .append("\n--- ")
                .append("End Request: ")
                .append(uniqueToken)
                .append(" ---")
            sb
                .append("\n--- ")
                .append("Debug Info: ")
                .append(uniqueToken)
                .append(" ---")
            sb
                .append("\n")
                .append(
                    if (debug == null) {
                        null
                    } else {
                        (debug as? String)?.toString() ?: Json.encodePrettily(debug)
                    },
                ).append("\n")
            sb
                .append("\n--- ")
                .append("End Debug Info: ")
                .append(uniqueToken)
                .append(" ---")
            sb
                .append("\n--- ")
                .append("Response: ")
                .append(uniqueToken)
                .append(" ---")
            sb.append("\nHeaders:\n")

            routingContext.response().headers().forEach(appendResponseHeaders(sb))

            sb.append("\nResponse Body:\n").append(bodyToAppend(body))
            sb
                .append("\n--- ")
                .append("Request Frame : ")
                .append(statusCode)
                .append(" ---\n")
            sb
                .append("\n--- ")
                .append("End Request Logging: ")
                .append(uniqueToken)
                .append(" ---\n")

            return sb
        }

        private fun appendResponseHeaders(sb: StringBuffer): (MutableMap.MutableEntry<String, String>) -> Unit =
            {
                sb
                    .append(it.key)
                    .append(" : ")
                    .append(it.value)
                    .append("\n")
            }

        private fun bodyToAppend(body: Any?): String? =
            if (body == null) {
                null
            } else {
                (body as? String)?.toString() ?: Json.encodePrettily(body)
            }

        private fun appendRequestHeaders(sb: StringBuffer): (MutableMap.MutableEntry<String, String>) -> Unit =
            {
                when {
                    filteredHeader(it) ->
                        sb
                            .append(it.key)
                            .append(" : ")
                            .append("[FILTERED]")
                            .append("\n")
                    else ->
                        sb
                            .append(it.key)
                            .append(" : ")
                            .append(it.value)
                            .append("\n")
                }
            }

        private fun filteredHeader(it: MutableMap.MutableEntry<String, String>) =
            it.key.equals("Authorization", ignoreCase = true) ||
                it.key.equals("X-Forwarded-For", ignoreCase = true)

        private fun outputLog(
            statusCode: Int,
            sb: StringBuffer,
        ) {
            when {
                statusCode in 200..399 ->
                    if (logger.isDebugEnabled()) {
                        logger.info { sb.toString() }
                    }

                statusCode in 400..499 -> logger.warn { sb.toString() }
                statusCode >= 500 -> logger.error { sb.toString() }
                else ->
                    if (logger.isDebugEnabled()) {
                        logger.debug { sb.toString() }
                    }
            }
        }
    }
}
