/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.genghis.tools.web.controllers

import com.genghis.tools.repository.dynamodb.DynamoDBRepository.Companion.PAGINATION_INDEX
import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.ModelUtils
import com.genghis.tools.repository.models.ValidationError
import com.genghis.tools.repository.repository.Repository
import com.genghis.tools.repository.utils.AggregateFunction
import com.genghis.tools.repository.utils.AggregateFunctions
import com.genghis.tools.repository.utils.AggregateFunctions.AVG
import com.genghis.tools.repository.utils.AggregateFunctions.COUNT
import com.genghis.tools.repository.utils.AggregateFunctions.SUM
import com.genghis.tools.repository.utils.CrossModelAggregateFunction
import com.genghis.tools.repository.utils.CrossModelGroupingConfiguration
import com.genghis.tools.repository.utils.CrossTableProjection
import com.genghis.tools.repository.utils.DynamoDBQuery
import com.genghis.tools.repository.utils.DynamoDBQuery.DynamoDBQuerybuilder
import com.genghis.tools.repository.utils.FilterPack
import com.genghis.tools.repository.utils.FilterPackField
import com.genghis.tools.repository.utils.FilterPackModel
import com.genghis.tools.repository.utils.FilterParameter
import com.genghis.tools.repository.utils.GroupingConfiguration
import com.genghis.tools.repository.utils.HashAndRange
import com.genghis.tools.repository.utils.hashAndRange
import com.genghis.tools.web.RoutingHelper.setStatusCodeAndAbort
import com.genghis.tools.web.RoutingHelper.setStatusCodeAndContinue
import com.genghis.tools.web.RoutingHelper.splitQuery
import com.genghis.tools.web.requestHandlers.RequestLogHandler.Companion.addLogMessageToRequestLog
import com.genghis.tools.web.responsehandlers.ResponseLogHandler.Companion.BODY_CONTENT_TAG
import com.google.common.util.concurrent.AtomicDouble
import io.vertx.core.Future
import io.vertx.core.Handler
import io.vertx.core.Promise
import io.vertx.core.http.HttpHeaders.ETAG
import io.vertx.core.http.HttpHeaders.IF_NONE_MATCH
import io.vertx.core.json.DecodeException
import io.vertx.core.json.EncodeException
import io.vertx.core.json.Json
import io.vertx.core.json.JsonArray
import io.vertx.core.json.JsonObject
import io.vertx.ext.web.RoutingContext
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import java.util.AbstractMap.SimpleEntry
import java.util.Arrays
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.CopyOnWriteArrayList
import java.util.function.BiConsumer
import java.util.function.BiFunction
import java.util.function.Function
import java.util.stream.Collectors.groupingBy
import java.util.stream.Collectors.toList
import java.util.stream.Collectors.toMap
import java.util.stream.Collectors.toSet
import kotlin.collections.Map.Entry

/**
 * This class defines a Handler implementation that handles aggregation queries on multiple models.
 *
 * @author Anders Mikkelsen
 * @version 13/11/17
 */
class CrossModelAggregationController(
    private val repositoryProvider: (Class<*>) -> (Repository<*, DynamoDBQuery, DynamoDBQuerybuilder>?),
    models: Array<Class<*>>,
) : Handler<RoutingContext> {
    private val modelMap: Map<String, Class<*>>

    init {
        this.modelMap = buildModelMap(models)
    }

    private fun buildModelMap(models: Array<Class<*>>): Map<String, Class<*>> =
        Arrays
            .stream(models)
            .map { SimpleEntry(buildCollectionName(it.simpleName), it) }
            .collect(toMap({ it.key }) { it.value })

    private fun buildCollectionName(typeName: String): String {
        val c = typeName.toCharArray()
        c[0] = c[0] + 32

        return String(c) + "s"
    }

    @OptIn(DelicateCoroutinesApi::class)
    override fun handle(routingContext: RoutingContext) {
        runCatching {
            val initialProcessNanoTime = System.nanoTime()
            val request = routingContext.request()
            val query = request.query()

            val aggregationPack =
                verifyRequest(
                    routingContext,
                    query,
                    initialProcessNanoTime,
                )

            when (aggregationPack?.aggregate) {
                null -> {
                    routingContext.put(
                        BODY_CONTENT_TAG,
                        JsonObject()
                            .put("request_error", "aggregate function cannot be null!"),
                    )

                    setStatusCodeAndContinue(400, routingContext, initialProcessNanoTime)
                }

                else ->
                    GlobalScope.launch {
                        performAggregation(
                            routingContext,
                            aggregationPack.aggregate,
                            aggregationPack.projection,
                            initialProcessNanoTime,
                        )
                    }
            }
        }.onFailure {
            logger.warn(it) { "Uncaught Aggregation error!" }
        }
    }

    private suspend fun performAggregation(
        routingContext: RoutingContext,
        aggregateFunction: CrossModelAggregateFunction,
        projection: Map<Class<*>, Set<String>>,
        initialNanoTime: Long,
    ) {
        val request = routingContext.request()
        val query = request.query()
        val requestEtag = request.getHeader(IF_NONE_MATCH)
        val identifier = hashAndRange {}
        val function = aggregateFunction.function
        val projections = getProjections(routingContext)

        when (function) {
            AVG, SUM, COUNT ->
                doValueAggregation(
                    routingContext,
                    query,
                    requestEtag,
                    identifier,
                    aggregateFunction,
                    projection,
                    function,
                    projections,
                    initialNanoTime,
                )

            else -> {
                val supportErrorObject =
                    JsonObject().put(
                        "function_support_error",
                        "Function " + function?.name + " is not yet supported...",
                    )
                sendQueryErrorResponse(supportErrorObject, routingContext, initialNanoTime)
            }
        }
    }

    private suspend fun doValueAggregation(
        routingContext: RoutingContext,
        query: String,
        requestEtag: String,
        identifier: HashAndRange,
        aggregate: CrossModelAggregateFunction,
        projection: Map<Class<*>, Set<String>>,
        aggregateFunction: AggregateFunctions,
        projections: Set<String>?,
        initialNanoTime: Long,
    ) {
        val groupingList = CopyOnWriteArrayList<Map<String, Double>>()
        val totalValue = AtomicDouble()
        val aggFutures = CopyOnWriteArrayList<Promise<*>>()

        if (logger.isDebugEnabled()) {
            addLogMessageToRequestLog(routingContext, "ProjectionMap: " + Json.encodePrettily(projection))
        }

        when (val valueResultHandler = getResultHandler(aggregate)) {
            null -> {
                addLogMessageToRequestLog(routingContext, "ResultHandler is null!")

                setStatusCodeAndAbort(500, routingContext, initialNanoTime)
            }

            else -> {
                val queryMap = splitQuery(query)
                val filterPack = convertToFilterPack(routingContext, queryMap[FILTER_KEY])

                projection.keys.forEach { clazz ->
                    val groupingConfigurations = getGroupingConfigurations(aggregate, clazz, true)
                    val longSorter = buildLongSorter(aggregate, groupingConfigurations)
                    val doubleSorter = buildDoubleSorter(aggregate, groupingConfigurations)

                    val aggregationResultHandler =
                        getAggregationResultHandler(longSorter, doubleSorter, aggregateFunction)

                    val aggregator: suspend (String?) -> Unit = { field ->
                        val fut = Promise.promise<Boolean>()
                        val temp =
                            AggregateFunction.builder {
                                withAggregateFunction(aggregateFunction)
                                withField(if (aggregateFunction != COUNT) field else null)
                                withGroupBy(groupingConfigurations)
                            }

                        when (val repo = repositoryProvider(clazz)) {
                            null -> {
                                addLogMessageToRequestLog(routingContext, clazz.simpleName + " is not valid!")

                                val future = Promise.promise<Boolean>()
                                aggFutures.add(future)

                                future.fail(IllegalArgumentException(clazz.simpleName + " is not valid!"))
                            }

                            else -> {
                                valueAggregation(
                                    routingContext,
                                    clazz,
                                    repo,
                                    fut,
                                    aggregationResultHandler,
                                    identifier,
                                    requestEtag,
                                    if (aggregateFunction == COUNT) null else projections,
                                    groupingList,
                                    totalValue,
                                    aggregate,
                                    temp,
                                    filterPack,
                                )

                                aggFutures.add(fut)
                            }
                        }
                    }

                    when (aggregationResultHandler) {
                        null -> {
                            addLogMessageToRequestLog(routingContext, "AggResultHandler is null!")

                            val fut = Promise.promise<Boolean>()
                            aggFutures.add(fut)
                            fut.tryFail(IllegalArgumentException("AggResultHandler cannot be null!"))
                        }

                        else ->
                            when {
                                projection[clazz]?.isEmpty()!! -> aggregator(null)
                                else -> projection[clazz]?.forEach { aggregator(it) }
                            }
                    }
                }

                Future.all<Any>(aggFutures.map { it.future() }).andThen { res ->
                    when {
                        res.failed() -> {
                            addLogMessageToRequestLog(routingContext, "Unknown aggregation error!", res.cause())

                            routingContext.put(
                                BODY_CONTENT_TAG,
                                JsonObject()
                                    .put("unknown_error", "Something went horribly wrong..."),
                            )
                            routingContext.response().statusCode = 500
                            routingContext.fail(res.cause())
                        }

                        else ->
                            valueResultHandler.accept(
                                ValueAggregationResultPack(aggregate, routingContext, initialNanoTime, totalValue),
                                groupingList,
                            )
                    }
                }
            }
        }
    }

    private fun getModelName(klazz: Class<*>): String = klazz.simpleName.lowercase() + "s"

    private fun buildLongSorter(
        function: CrossModelAggregateFunction,
        groupingConfigurations: List<GroupingConfiguration>,
    ): Comparator<JsonObject> {
        val lowerCaseFunctionName = function.function?.name?.lowercase()
        if (groupingConfigurations.size > 3) throw IllegalArgumentException("GroupBy size of three is max!")
        val levelOne = if (groupingConfigurations.isNotEmpty()) groupingConfigurations[0] else null
        val levelTwo = if (groupingConfigurations.size > 1) groupingConfigurations[1] else null
        val levelThree = if (groupingConfigurations.size > 2) groupingConfigurations[2] else null
        var finalConfig: GroupingConfiguration? = null

        when {
            levelThree != null -> finalConfig = levelThree
            levelTwo != null -> finalConfig = levelTwo
            levelOne != null -> finalConfig = levelOne
        }

        return if (finalConfig == null || finalConfig.groupingSortOrder == "asc") {
            Comparator.comparingLong { item -> item.getLong(lowerCaseFunctionName) }
        } else {
            Comparator.comparingLong<JsonObject> { item -> item.getLong(lowerCaseFunctionName) }.reversed()
        }
    }

    private fun buildDoubleSorter(
        function: CrossModelAggregateFunction,
        groupingConfigurations: List<GroupingConfiguration>,
    ): Comparator<JsonObject> {
        val lowerCaseFunctionName = function.function?.name?.lowercase()
        if (groupingConfigurations.size > 3) throw IllegalArgumentException("GroupBy size of three is max!")
        val levelOne = if (groupingConfigurations.isNotEmpty()) groupingConfigurations[0] else null
        val levelTwo = if (groupingConfigurations.size > 1) groupingConfigurations[1] else null
        val levelThree = if (groupingConfigurations.size > 2) groupingConfigurations[2] else null
        var finalConfig: GroupingConfiguration? = null

        when {
            levelThree != null -> finalConfig = levelThree
            levelTwo != null -> finalConfig = levelTwo
            levelOne != null -> finalConfig = levelOne
        }

        return if (finalConfig == null || finalConfig.groupingSortOrder == "asc") {
            Comparator.comparingDouble { item -> item.getDouble(lowerCaseFunctionName) }
        } else {
            Comparator.comparingDouble<JsonObject> { item -> item.getDouble(lowerCaseFunctionName) }.reversed()
        }
    }

    private inner class AggregationResultPack(
        val aggregate: CrossModelAggregateFunction,
        val result: JsonObject,
        val value: AtomicDouble,
    )

    private fun getAggregationResultHandler(
        longSorter: Comparator<JsonObject>,
        doubleSorter: Comparator<JsonObject>,
        aggregateFunction: AggregateFunctions,
    ): BiConsumer<AggregationResultPack, MutableList<Map<String, Double>>>? {
        val keyMapper =
            BiFunction<AggregationResultPack, JsonObject, String> { resultPack, item ->
                val aggregate = resultPack.aggregate

                when {
                    aggregate.hasGrouping() && aggregate.groupBy!![0].hasGroupRanging() -> item.encode()
                    else -> item.getString("groupByKey")
                }
            }

        when (aggregateFunction) {
            AVG -> return BiConsumer { resultPack, groupingList ->
                when {
                    resultPack.aggregate.hasGrouping() -> {
                        val array = resultPack.result.getJsonArray("results")

                        logger.debug { "Results before sort is: " + array.encodePrettily() }
                        logger.debug { "Aggregate: " + Json.encodePrettily(aggregateFunction) }

                        val collect =
                            array
                                .stream()
                                .map { itemAsString -> JsonObject(itemAsString.toString()) }
                                .sorted(doubleSorter)
                                .collect(
                                    toMap<JsonObject, String, Double, LinkedHashMap<String, Double>>(
                                        { keyMapper.apply(resultPack, it) },
                                        { it.getDouble(aggregateFunction.name.lowercase()) },
                                        { u, _ -> throw IllegalStateException(String.format("Duplicate key %s", u)) },
                                        { LinkedHashMap() },
                                    ),
                                )

                        groupingList.add(collect)
                    }

                    else ->
                        resultPack.value.addAndGet(
                            resultPack.result.getDouble(aggregateFunction.name.lowercase())!!,
                        )
                }
            }

            SUM, COUNT -> return BiConsumer { resultPack, groupingList ->
                when {
                    resultPack.aggregate.hasGrouping() -> {
                        val array = resultPack.result.getJsonArray("results")

                        logger.debug { "Results before sort is: " + array.encodePrettily() }
                        logger.debug { "Aggregate: " + Json.encodePrettily(aggregateFunction) }

                        val collect =
                            array
                                .stream()
                                .map { itemAsString -> JsonObject(itemAsString.toString()) }
                                .sorted(longSorter)
                                .collect(
                                    toMap<JsonObject, String, Double, LinkedHashMap<String, Double>>(
                                        { keyMapper.apply(resultPack, it) },
                                        { it.getLong(aggregateFunction.name.lowercase())!!.toDouble() },
                                        { u, _ -> throw IllegalStateException(String.format("Duplicate key %s", u)) },
                                        { LinkedHashMap() },
                                    ),
                                )

                        groupingList.add(collect)
                    }

                    else ->
                        resultPack.value.addAndGet(
                            resultPack.result.getLong(aggregateFunction.name.lowercase())!!.toDouble(),
                        )
                }
            }

            else -> return null
        }
    }

    private inner class ValueAggregationResultPack(
        val aggregate: CrossModelAggregateFunction,
        val routingContext: RoutingContext,
        val initTime: Long,
        val value: AtomicDouble,
    )

    private fun getResultHandler(
        aggregateFunction: CrossModelAggregateFunction,
    ): BiConsumer<ValueAggregationResultPack, List<Map<String, Double>>>? {
        val asc =
            aggregateFunction.hasGrouping() &&
                aggregateFunction.groupBy!![0].groupingSortOrder!!.equals(
                    "asc",
                    ignoreCase = true,
                )
        val comparator =
            buildComparator<Long>(
                aggregateFunction,
                if (aggregateFunction.hasGrouping()) {
                    aggregateFunction.groupBy!![0]
                } else {
                    null
                },
            )

        var valueMapper =
            Function<List<Map<String, Double>>, JsonArray> { groupingList ->
                val results = JsonArray()
                val countMap = LinkedHashMap<String, Long>()
                groupingList.forEach { map ->
                    map.forEach { (k, v) ->
                        when {
                            aggregateFunction.hasGrouping() && aggregateFunction.groupBy!![0].hasGroupRanging() -> {
                                val groupingObject = JsonObject(k)
                                val key =
                                    JsonObject()
                                        .put("floor", groupingObject.getLong("floor"))
                                        .put("ceil", groupingObject.getLong("ceil"))
                                        .encode()

                                when {
                                    countMap.containsKey(key) -> countMap[key] = countMap[key]!! + v.toLong()
                                    else -> countMap[key] = v.toLong()
                                }
                            }

                            else ->
                                when {
                                    countMap.containsKey(k) -> countMap[k] = countMap[k]!! + v.toLong()
                                    else -> countMap[k] = v.toLong()
                                }
                        }
                    }
                }

                countMap.entries
                    .sortedWith(
                        createValueComparator(
                            asc,
                            when {
                                aggregateFunction.hasGrouping() -> aggregateFunction.groupBy!![0]
                                else -> null
                            },
                        ),
                    ).take(
                        when {
                            aggregateFunction.hasGrouping() -> aggregateFunction.groupBy!![0].groupingListLimit
                            else -> 10
                        },
                    ).forEach { x -> comparator.accept(results, x) }

                results
            }

        when (aggregateFunction.function) {
            AVG -> {
                val doubleComparator =
                    buildComparator<Double>(
                        aggregateFunction,
                        if (aggregateFunction.hasGrouping()) {
                            aggregateFunction.groupBy!![0]
                        } else {
                            null
                        },
                    )

                valueMapper =
                    Function { groupingList ->
                        val results = JsonArray()
                        val countMap = LinkedHashMap<String, Double>()
                        groupingList.forEach { map ->
                            map.forEach { (k, v) ->
                                when {
                                    aggregateFunction.hasGrouping() && aggregateFunction.groupBy!![0].hasGroupRanging() -> {
                                        val groupingObject = JsonObject(k)
                                        val key =
                                            JsonObject()
                                                .put("floor", groupingObject.getLong("floor"))
                                                .put("ceil", groupingObject.getLong("ceil"))
                                                .encode()

                                        when {
                                            countMap.containsKey(key) -> countMap[key] = countMap[key]!! + v
                                            else -> countMap[key] = v
                                        }
                                    }

                                    else ->
                                        when {
                                            countMap.containsKey(k) -> countMap[k] = countMap[k]!! + v
                                            else -> countMap[k] = v
                                        }
                                }
                            }
                        }

                        countMap.entries
                            .stream()
                            .sorted(
                                createValueComparator(
                                    asc,
                                    if (aggregateFunction.hasGrouping()) {
                                        aggregateFunction.groupBy!![0]
                                    } else {
                                        null
                                    },
                                ),
                            ).limit(
                                (
                                    if (aggregateFunction.hasGrouping()) {
                                        aggregateFunction.groupBy!![0].groupingListLimit
                                    } else {
                                        10
                                    }
                                ).toLong(),
                            ).forEachOrdered { x -> doubleComparator.accept(results, x) }

                        results
                    }

                val finalValueMapper = valueMapper

                return BiConsumer { resultPack, groupingList ->
                    val routingContext = resultPack.routingContext
                    val initialNanoTime = resultPack.initTime

                    when {
                        resultPack.aggregate.hasGrouping() -> {
                            val result = finalValueMapper.apply(groupingList)
                            val newEtag = ModelUtils.returnNewEtag(result.encode().hashCode().toLong())
                            val results = JsonObject().put("count", result.size())

                            if (resultPack.aggregate.hasGrouping() && resultPack.aggregate.groupBy!![0].hasGroupRanging()) {
                                results.put(
                                    "rangeGrouping",
                                    JsonObject()
                                        .put("unit", aggregateFunction.groupBy!![0].groupByUnit)
                                        .put("range", aggregateFunction.groupBy!![0].groupByRange),
                                )
                            }

                            results.put("results", result)
                            val content = results.encode()

                            checkEtagAndReturn(content, newEtag, routingContext, initialNanoTime)
                        }

                        else -> {
                            val functionName = aggregateFunction.function!!.name.lowercase()
                            val result =
                                JsonObject().put(
                                    functionName,
                                    if (aggregateFunction.function == AVG) {
                                        resultPack.value
                                    } else {
                                        resultPack.value.toLong()
                                    },
                                )
                            val content = result.encode()
                            val newEtag = ModelUtils.returnNewEtag(content.hashCode().toLong())

                            checkEtagAndReturn(content, newEtag, routingContext, initialNanoTime)
                        }
                    }
                }
            }

            SUM, COUNT -> {
                val finalValueMapper = valueMapper

                return BiConsumer { resultPack, groupingList ->
                    val routingContext = resultPack.routingContext
                    val initialNanoTime = resultPack.initTime

                    when {
                        resultPack.aggregate.hasGrouping() -> {
                            val result = finalValueMapper.apply(groupingList)
                            val newEtag = ModelUtils.returnNewEtag(result.encode().hashCode().toLong())
                            val results = JsonObject().put("count", result.size())
                            if (resultPack.aggregate.hasGrouping() && resultPack.aggregate.groupBy!![0].hasGroupRanging()) {
                                results.put(
                                    "rangeGrouping",
                                    JsonObject()
                                        .put("unit", aggregateFunction.groupBy!![0].groupByUnit)
                                        .put("range", aggregateFunction.groupBy!![0].groupByRange),
                                )
                            }
                            results.put("results", result)
                            val content = results.encode()
                            checkEtagAndReturn(content, newEtag, routingContext, initialNanoTime)
                        }

                        else -> {
                            val functionName = aggregateFunction.function!!.name.lowercase()
                            val result =
                                JsonObject().put(
                                    functionName,
                                    if (aggregateFunction.function == AVG) {
                                        resultPack.value
                                    } else {
                                        resultPack.value.toLong()
                                    },
                                )
                            val content = result.encode()
                            val newEtag = ModelUtils.returnNewEtag(content.hashCode().toLong())
                            checkEtagAndReturn(content, newEtag, routingContext, initialNanoTime)
                        }
                    }
                }
            }

            else -> return null
        }
    }

    private fun <T : Comparable<T>> createValueComparator(
        asc: Boolean,
        groupingConfiguration: CrossModelGroupingConfiguration?,
    ): Comparator<Entry<String, T>> =
        when {
            groupingConfiguration != null && groupingConfiguration.hasGroupRanging() ->
                if (asc) KeyComparator() else KeyComparator<String, T>().reversed()

            else ->
                if (asc) ValueThenKeyComparator() else ValueThenKeyComparator<String, T>().reversed()
        }

    private fun <T> buildComparator(
        aggregateFunction: CrossModelAggregateFunction,
        groupingConfiguration: CrossModelGroupingConfiguration?,
    ): BiConsumer<JsonArray, Entry<String, T>> =
        when {
            groupingConfiguration != null && groupingConfiguration.hasGroupRanging() ->
                BiConsumer { results, x ->
                    results.add(
                        JsonObject(x.key)
                            .put(aggregateFunction.function?.name?.lowercase(), x.value),
                    )
                }

            else ->
                BiConsumer { results, x ->
                    results.add(
                        JsonObject().put(x.key, x.value),
                    )
                }
        }

    private inner class KeyComparator<K : Comparable<K>, V : Comparable<V>> : Comparator<Entry<K, V>> {
        override fun compare(
            a: Entry<K, V>,
            b: Entry<K, V>,
        ): Int {
            val keyObjectA = JsonObject(a.key.toString())
            val keyObjectB = JsonObject(b.key.toString())

            return keyObjectA.getLong("ceil")!!.compareTo(keyObjectB.getLong("ceil")!!)
        }
    }

    private inner class ValueThenKeyComparator<K : Comparable<K>, V : Comparable<V>> : Comparator<Entry<K, V>> {
        override fun compare(
            a: Entry<K, V>,
            b: Entry<K, V>,
        ): Int {
            val cmp1 = a.value.compareTo(b.value)

            return when {
                cmp1 != 0 -> cmp1
                else -> a.key.compareTo(b.key)
            }
        }
    }

    private fun checkEtagAndReturn(
        content: String,
        newEtag: String?,
        routingContext: RoutingContext,
        initialNanoTime: Long,
    ) {
        val requestEtag = routingContext.request().getHeader(IF_NONE_MATCH)

        when {
            newEtag != null && requestEtag != null && requestEtag.equals(newEtag, ignoreCase = true) ->
                setStatusCodeAndContinue(304, routingContext, initialNanoTime)

            else -> {
                if (newEtag != null) routingContext.response().putHeader(ETAG, newEtag)
                routingContext.put(BODY_CONTENT_TAG, content)
                setStatusCodeAndContinue(200, routingContext, initialNanoTime)
            }
        }
    }

    private suspend fun valueAggregation(
        routingContext: RoutingContext,
        modelType: Class<*>,
        repo: Repository<*, DynamoDBQuery, DynamoDBQuerybuilder>,
        fut: Promise<Boolean>,
        aggregationResultHandler: BiConsumer<AggregationResultPack, MutableList<Map<String, Double>>>?,
        identifier: HashAndRange,
        requestEtag: String,
        projections: Set<String>?,
        groupingList: List<Map<String, Double>>,
        totalCount: AtomicDouble,
        aggregate: CrossModelAggregateFunction,
        temp: AggregateFunction,
        filterPack: FilterPack?,
    ) {
        val params = convertToFilterList(modelType, filterPack)
        val query =
            DynamoDBQuery.builder {
                hashAndRange(identifier)
                lsi(PAGINATION_INDEX)
                routingContext.request().getParam(PAGING_TOKEN_KEY)?.let { pageToken(it) }
                etag(requestEtag)
                filter(params)
                aggregateFunction(temp)
                projections(projections ?: emptySet())
            }

        if (logger.isDebugEnabled()) {
            addLogMessageToRequestLog(routingContext, "PROJECTIONS: $projections")
        }

        runCatching {
            repo.aggregation(query)
        }.onSuccess {
            if (logger.isDebugEnabled()) {
                addLogMessageToRequestLog(routingContext, "Countres: $it")
            }

            val valueResultObject = JsonObject(it)

            if (logger.isDebugEnabled()) {
                addLogMessageToRequestLog(
                    routingContext,
                    modelType.simpleName + "_agg: " +
                        Json.encodePrettily(valueResultObject),
                )
            }

            val pack = AggregationResultPack(aggregate, valueResultObject, totalCount)
            aggregationResultHandler!!.accept(pack, groupingList.toMutableList())

            fut.complete(java.lang.Boolean.TRUE)
        }.onFailure {
            addLogMessageToRequestLog(routingContext, "Unable to fetch res for " + modelType.simpleName + "!", it)

            fut.fail(it)
        }
    }

    private fun convertToFilterPack(
        routingContext: RoutingContext,
        filter: List<String>?,
    ): FilterPack? {
        if (filter == null) return null

        try {
            return Json.decodeValue(filter[0], FilterPack::class.java)
        } catch (e: DecodeException) {
            addLogMessageToRequestLog(routingContext, "Cannot decode FilterPack...", e)
        }

        return null
    }

    private fun convertToFilterList(
        modelType: Class<*>,
        filterPack: FilterPack?,
    ): Map<String, MutableList<FilterParameter>> {
        if (filterPack == null) return ConcurrentHashMap()

        val params = ConcurrentHashMap<String, MutableList<FilterParameter>>()

        val first =
            filterPack.models!!
                .stream()
                .filter { model -> model.model!!.equals("${modelType.simpleName}s", ignoreCase = true) }
                .findFirst()

        if (first.isPresent) {
            val parameters = ArrayList<FilterParameter>()

            val groupedFields = getGroupedParametersForFields(first.get())
            groupedFields.keys.forEach {
                groupedFields[it]?.forEach { fpf ->
                    fpf.parameters!!.forEach { fieldFilter ->
                        fieldFilter.field = it
                        parameters.add(fieldFilter)
                    }
                }

                params[it] = parameters
            }
        }

        return params
    }

    private fun getGroupedParametersForFields(pm: FilterPackModel): Map<String, List<FilterPackField>> =
        pm.fields!!.stream().collect(
            groupingBy {
                it.field
            },
        )

    private fun verifyRequest(
        routingContext: RoutingContext,
        query: String?,
        initialProcessTime: Long,
    ): AggregationPack? {
        when (query) {
            null -> noQueryError(routingContext, initialProcessTime)
            else -> {
                val queryMap = splitQuery(query)

                if (verifyQuery(queryMap, routingContext, initialProcessTime)) {
                    try {
                        val aggregateFunction =
                            Json.decodeValue(
                                queryMap[AGGREGATE_KEY]?.get(0),
                                CrossModelAggregateFunction::class.java,
                            )
                        var crossTableProjection =
                            Json.decodeValue(
                                queryMap[PROJECTION_KEY]?.get(0),
                                CrossTableProjection::class.java,
                            )
                        crossTableProjection =
                            CrossTableProjection(
                                crossTableProjection.models,
                                ArrayList(modelMap.keys),
                                crossTableProjection.fields,
                            )
                        val pageToken = routingContext.request().getParam(PAGING_TOKEN_KEY)

                        when {
                            aggregateFunction.field != null -> {
                                val aggregationQueryErrorObject =
                                    JsonObject().put(
                                        "aggregate_field_error",
                                        "Field must be null in aggregate, use fields in projection instead",
                                    )

                                sendQueryErrorResponse(aggregationQueryErrorObject, routingContext, initialProcessTime)
                            }

                            else -> {
                                val errors = crossTableProjection.validate(aggregateFunction.function!!)

                                when {
                                    errors.isEmpty() -> {
                                        val projectionMap = buildProjectionMap(crossTableProjection)
                                        val aggregationErrors = verifyAggregation(aggregateFunction, projectionMap)

                                        when {
                                            aggregationErrors.isEmpty() ->
                                                return AggregationPack(aggregateFunction, projectionMap, pageToken)

                                            else ->
                                                sendQueryErrorResponse(
                                                    buildJsonErrorObject(
                                                        "Aggregate",
                                                        aggregationErrors,
                                                    ),
                                                    routingContext,
                                                    initialProcessTime,
                                                )
                                        }
                                    }

                                    else ->
                                        sendQueryErrorResponse(
                                            buildValidationErrorObject("Projection", errors),
                                            routingContext,
                                            initialProcessTime,
                                        )
                                }
                            }
                        }
                    } catch (e: DecodeException) {
                        val aggregationQueryErrorObject =
                            JsonObject()
                                .put("json_parse_error", "Unable to parse json in query, are you sure it is URL encoded?")

                        sendQueryErrorResponse(aggregationQueryErrorObject, routingContext, initialProcessTime)
                    }
                }
            }
        }

        return null
    }

    private fun buildProjectionMap(crossTableProjection: CrossTableProjection): Map<Class<*>, Set<String>> =
        crossTableProjection.models!!
            .stream()
            .map { SimpleEntry<Class<*>, Set<String>>(modelMap[it], getFieldsForCollection(crossTableProjection, it)) }
            .collect(toMap({ it.key }) { it.value })

    private fun getFieldsForCollection(
        crossTableProjection: CrossTableProjection,
        collection: String,
    ): Set<String> =
        when (crossTableProjection.fields) {
            null -> HashSet()
            else ->
                crossTableProjection.fields!!
                    .stream()
                    .map<String> { field ->
                        val fieldSplit = field.split("\\.".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()

                        if (fieldSplit[0].equals(collection, ignoreCase = true)) fieldSplit[1] else null
                    }.collect(toSet())
        }

    private fun verifyAggregation(
        aggregateFunction: CrossModelAggregateFunction,
        crossTableProjection: Map<Class<*>, Set<String>>,
    ): List<JsonObject> {
        val function = aggregateFunction.function
        val errors = ArrayList<JsonObject>()

        when {
            aggregateFunction.groupBy!!.size > 1 ->
                errors.add(
                    JsonObject().put("grouping_error", "Max grouping for cross model is 1!"),
                )

            else ->
                crossTableProjection.forEach { (klazz, fieldSet) ->
                    fieldSet.forEach {
                        val groupingConfigurations = getGroupingConfigurations(aggregateFunction, klazz)
                        val temp =
                            AggregateFunction.builder {
                                withAggregateFunction(function!!)
                                withField(it)
                                withGroupBy(groupingConfigurations)
                            }

                        @Suppress("UNCHECKED_CAST")
                        temp.validateFieldForFunction(klazz as Class<ETagable>)
                        if (!temp.validationError.isEmpty) errors.add(temp.validationError)
                    }
                }
        }

        return errors
    }

    private fun getGroupingConfigurations(
        aggregateFunction: CrossModelAggregateFunction,
        klazz: Class<*>,
        fullList: Boolean = false,
    ): List<GroupingConfiguration> {
        val groupBy = aggregateFunction.groupBy ?: return ArrayList()
        val modelName = getModelName(klazz)

        return groupBy
            .stream()
            .map { cmgf ->
                val innerGroupBy = cmgf.groupBy
                if (innerGroupBy!!.size == 1) innerGroupBy[0]

                innerGroupBy
                    .stream()
                    .filter { gb -> gb.startsWith(modelName) }
                    .findFirst()
                    .map { s ->
                        s.split("\\.".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()[1]
                    }.orElse(null)
            }.findFirst()
            .map { s ->
                aggregateFunction.groupBy!!
                    .stream()
                    .map {
                        GroupingConfiguration
                            .builder()
                            .withGroupBy(s)
                            .withGroupByUnit(it.groupByUnit)
                            .withGroupByRange(it.groupByRange)
                            .withGroupingSortOrder(it.groupingSortOrder)
                            .withGroupingListLimit(it.groupingListLimit)
                            .withFullList(fullList)
                            .build()
                    }.collect(toList())
            }.orElseGet { ArrayList() }
    }

    @Suppress("SameParameterValue")
    private fun buildValidationErrorObject(
        projection: String,
        errors: List<ValidationError>,
    ): JsonObject {
        val validationArray = JsonArray()
        errors.stream().map { it.toJson() }.forEach { validationArray.add(it) }

        return JsonObject()
            .put("validation_error", "$projection is invalid...")
            .put("errors", validationArray)
    }

    @Suppress("SameParameterValue")
    private fun buildJsonErrorObject(
        projection: String,
        errors: List<JsonObject>,
    ): JsonObject {
        val validationArray = JsonArray()
        errors.forEach { validationArray.add(it) }

        return JsonObject()
            .put("validation_error", "$projection is invalid...")
            .put("errors", validationArray)
    }

    private fun verifyQuery(
        queryMap: Map<String, List<String>>?,
        routingContext: RoutingContext,
        initialProcessTime: Long,
    ): Boolean {
        when {
            queryMap == null -> {
                noAggregateError(routingContext, initialProcessTime)

                return false
            }

            queryMap[AGGREGATE_KEY] == null -> {
                noAggregateError(routingContext, initialProcessTime)

                return false
            }

            queryMap[PROJECTION_KEY] == null -> {
                noProjectionError(routingContext, initialProcessTime)

                return false
            }

            queryMap[PAGING_TOKEN_KEY] != null &&
                queryMap[PAGING_TOKEN_KEY]?.get(0).equals(END_OF_PAGING_KEY, ignoreCase = true) -> {
                noPageError(routingContext, initialProcessTime)

                return false
            }

            else -> return true
        }
    }

    private fun noAggregateError(
        routingContext: RoutingContext,
        initialProcessTime: Long,
    ) {
        val aggregationQueryErrorObject =
            JsonObject()
                .put("aggregate_query_error", "$AGGREGATE_KEY query param is required!")

        sendQueryErrorResponse(aggregationQueryErrorObject, routingContext, initialProcessTime)
    }

    private fun noProjectionError(
        routingContext: RoutingContext,
        initialProcessTime: Long,
    ) {
        val aggregationQueryErrorObject =
            JsonObject()
                .put("projection_query_error", "$PROJECTION_KEY query param is required!")

        sendQueryErrorResponse(aggregationQueryErrorObject, routingContext, initialProcessTime)
    }

    private fun noQueryError(
        routingContext: RoutingContext,
        initialProcessTime: Long,
    ) {
        val aggregationQueryErrorObject =
            JsonObject()
                .put("aggregation_error", "Query cannot be null for this endpoint!")

        sendQueryErrorResponse(aggregationQueryErrorObject, routingContext, initialProcessTime)
    }

    private fun noPageError(
        routingContext: RoutingContext,
        initialProcessTime: Long,
    ) {
        val aggregationQueryErrorObject =
            JsonObject()
                .put(
                    "paging_error",
                    "You cannot page for the " + END_OF_PAGING_KEY + ", " +
                        "this message means you have reached the end of the results requested.",
                )

        sendQueryErrorResponse(aggregationQueryErrorObject, routingContext, initialProcessTime)
    }

    private fun sendQueryErrorResponse(
        aggregationQueryErrorObject: JsonObject,
        routingContext: RoutingContext,
        initialProcessTimeNano: Long,
    ) {
        routingContext.put(BODY_CONTENT_TAG, aggregationQueryErrorObject)
        setStatusCodeAndContinue(400, routingContext, initialProcessTimeNano)
    }

    private fun getProjections(routingContext: RoutingContext): Set<String>? {
        val projectionJson = routingContext.request().getParam(PROJECTION_KEY)
        var projections: Set<String>? = null

        if (projectionJson != null) {
            try {
                val projection = JsonObject(projectionJson)
                val array = projection.getJsonArray(PROJECTION_FIELDS_KEY, null)

                if (array != null) {
                    projections =
                        array
                            .stream()
                            .map { o ->
                                o
                                    .toString()
                                    .split("\\.".toRegex())
                                    .dropLastWhile { it.isEmpty() }
                                    .toTypedArray()[1]
                            }.collect(toList())
                            .filterNotNull()
                            .toSet()

                    if (logger.isDebugEnabled()) {
                        addLogMessageToRequestLog(routingContext, "Projection ready!")
                    }
                }
            } catch (e: DecodeException) {
                addLogMessageToRequestLog(routingContext, "Unable to parse projections: $e")

                projections = null
            } catch (e: EncodeException) {
                addLogMessageToRequestLog(routingContext, "Unable to parse projections: $e")
                projections = null
            }
        }

        return projections
    }

    @Suppress("unused")
    private inner class AggregationPack(
        val aggregate: CrossModelAggregateFunction,
        val projection: Map<Class<*>, Set<String>>,
        val pageToken: String?,
    )

    companion object {
        private const val PROJECTION_KEY = "projection"
        private const val PROJECTION_FIELDS_KEY = "fields"
        private const val FILTER_KEY = "filter"
        private const val AGGREGATE_KEY = "aggregate"

        private const val PAGING_TOKEN_KEY = "paging"
        private const val END_OF_PAGING_KEY = "END_OF_LIST"
    }
}
