/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.web.controllers

import com.genghis.tools.repository.dynamodb.DynamoDBRepository
import com.genghis.tools.repository.dynamodb.DynamoDBRepository.Companion.PAGINATION_INDEX
import com.genghis.tools.repository.dynamodb.hash
import com.genghis.tools.repository.dynamodb.range
import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.DynamoDBModel
import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.models.ModelUtils
import com.genghis.tools.repository.repository.etag.ETagManager
import com.genghis.tools.repository.repository.etag.InMemoryETagManagerImpl
import com.genghis.tools.repository.repository.etag.RedisETagManagerImpl
import com.genghis.tools.repository.repository.redis.RedisUtils.getRedisClient
import com.genghis.tools.repository.repository.results.ItemListResult
import com.genghis.tools.repository.repository.results.ItemResult
import com.genghis.tools.repository.utils.AggregateFunction
import com.genghis.tools.repository.utils.AggregateFunctions
import com.genghis.tools.repository.utils.AggregateFunctions.MAX
import com.genghis.tools.repository.utils.AggregateFunctions.MIN
import com.genghis.tools.repository.utils.DynamoDBQuery
import com.genghis.tools.repository.utils.DynamoDBQuery.DynamoDBQuerybuilder
import com.genghis.tools.repository.utils.FilterParameter
import com.genghis.tools.repository.utils.HashAndRange
import com.genghis.tools.repository.utils.OrderByParameter
import com.genghis.tools.repository.utils.hashAndRange
import com.genghis.tools.web.RoutingHelper.denyQuery
import com.genghis.tools.web.RoutingHelper.setStatusCodeAndAbort
import com.genghis.tools.web.RoutingHelper.splitQuery
import com.genghis.tools.web.requestHandlers.RequestLogHandler.Companion.REQUEST_PROCESS_TIME_TAG
import com.genghis.tools.web.requestHandlers.RequestLogHandler.Companion.addLogMessageToRequestLog
import com.genghis.tools.web.responsehandlers.ResponseLogHandler.Companion.BODY_CONTENT_TAG
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.codegen.annotations.Nullable
import io.vertx.core.Vertx
import io.vertx.core.http.HttpHeaders.CONTENT_TYPE
import io.vertx.core.http.HttpHeaders.ETAG
import io.vertx.core.http.HttpHeaders.IF_NONE_MATCH
import io.vertx.core.json.DecodeException
import io.vertx.core.json.EncodeException
import io.vertx.core.json.Json.decodeValue
import io.vertx.core.json.Json.encode
import io.vertx.core.json.Json.encodePrettily
import io.vertx.core.json.JsonArray
import io.vertx.core.json.JsonObject
import io.vertx.ext.web.RoutingContext
import io.vertx.kotlin.core.json.jsonObjectOf
import io.vertx.kotlin.coroutines.dispatcher
import kotlinx.coroutines.runBlocking
import software.amazon.awssdk.enhanced.dynamodb.mapper.annotations.DynamoDbSecondaryPartitionKey
import software.amazon.awssdk.enhanced.dynamodb.mapper.annotations.DynamoDbSecondarySortKey
import software.amazon.awssdk.utils.Either
import software.amazon.awssdk.utils.Either.left
import software.amazon.awssdk.utils.Either.right
import java.lang.reflect.Field
import java.lang.reflect.Method
import java.util.Arrays
import java.util.concurrent.ConcurrentHashMap
import java.util.stream.Collectors.toList
import kotlin.collections.component1
import kotlin.collections.component2

/**
 * This interface defines the default RestControllerImpl. It prepares queries and builds responses. Standard model
 * operations need not override anything to use this controller. Overriding functions must remember to call the next
 * element in the chain.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */

internal val logger = KotlinLogging.logger { }

open class DynamoDbRestController<E>(
    final override val vertx: Vertx,
    private val type: Class<E>,
    private val appConfig: JsonObject,
    private val repo: DynamoDBRepository<E>,
    private val idSupplier: (RoutingContext) -> HashAndRange = defaultSupplier,
    etagManager: ETagManager<E>?,
) : RestController<E, DynamoDBQuery, DynamoDBQuerybuilder>
    where E : ETagable, E : Model, E : DynamoDBModel, E : Cacheable {
    private val collection: String
    private val fields: Array<Field>
    private val methods: Array<Method>

    final override var etagManager: ETagManager<E>? = null
        protected set

    init {
        this.collection = buildCollectionName(type.name)
        fields = this.getAllFieldsOnType(type)
        methods = this.getAllMethodsOnType(type)
        val etagManagerRepo = repo.etagManager

        when {
            etagManager != null -> this.etagManager = etagManager
            etagManagerRepo != null -> this.etagManager = etagManagerRepo
            appConfig.getString("redis_host") != null ->
                this.etagManager =
                    RedisETagManagerImpl(type, runBlocking(vertx.dispatcher()) { getRedisClient(vertx, appConfig) })

            else -> this.etagManager = InMemoryETagManagerImpl(vertx, type)
        }
    }

    private fun buildCollectionName(typeName: String): String {
        val c = typeName.toCharArray()
        c[0] = c[0] + 32

        return String(c) + "s"
    }

    override suspend fun performShow(routingContext: RoutingContext) {
        if (denyQuery(routingContext)) return

        val initialProcessNanoTime = System.nanoTime()
        val id = getAndVerifyId(routingContext)

        when {
            id.isEmpty() -> setStatusCodeAndAbort(400, routingContext, initialProcessNanoTime)
            else -> processValidRead(routingContext, id)
        }
    }

    private suspend fun processValidRead(
        routingContext: RoutingContext,
        id: HashAndRange,
    ) {
        val gsi = routingContext.request().getParam(GLOBAL_HASH_INDEX)
        val projectionJson = routingContext.request().getParam(PROJECTION_KEY)
        val projections: Set<String>? = extractProjection(projectionJson, routingContext)

        if (logger.isDebugEnabled()) {
            addLogMessageToRequestLog(routingContext, "Show projection: $projections")
        }

        val query =
            DynamoDBQuery.builder {
                hashAndRange(id)
                gsi?.let { gsi(it) }
                projections?.let { projections(it) }
            }

        when {
            isProjectedRead(projections) -> proceedWithProjectedRead(routingContext, query, etagManager)
            else -> proceedWithRead(routingContext, query)
        }
    }

    private fun isProjectedRead(projections: Set<String>?) = !projections.isNullOrEmpty()

    private suspend fun proceedWithProjectedRead(
        routingContext: RoutingContext,
        query: DynamoDBQuery,
        etagManager: ETagManager<E>?,
    ) {
        val etag = routingContext.request().getHeader(IF_NONE_MATCH)

        if (logger.isDebugEnabled()) {
            addLogMessageToRequestLog(routingContext, "Etag is: $etag")
        }

        when {
            etag != null && etagManager != null -> {
                val hash = query.identifiers.hashCode()
                val etagKeyBase = "${type.simpleName}_$hash/projections"
                val key = "${type.simpleName}_$hash/projections${query.projections.hashCode()}"

                if (logger.isDebugEnabled()) {
                    addLogMessageToRequestLog(routingContext, "Checking etag for show...")
                }

                runCatching {
                    etagManager.checkItemEtag(
                        etagKeyBase,
                        key,
                        etag,
                    )
                }.onSuccess {
                    unChangedIndex(routingContext)
                }.onFailure {
                    proceedWithRead(routingContext, query)
                }
            }

            else -> proceedWithRead(routingContext, query)
        }
    }

    private fun extractProjection(
        projectionJson: String?,
        routingContext: RoutingContext,
    ): Set<String>? {
        if (projectionJson != null) {
            try {
                val projection = JsonObject(projectionJson)
                val array = projection.getJsonArray(PROJECTION_FIELDS_KEY, null)

                if (array != null) {
                    val projections =
                        array
                            .stream()
                            .map { it.toString() }
                            .collect(toList())
                            .filterNotNull()
                            .toSet()

                    if (logger.isDebugEnabled()) {
                        addLogMessageToRequestLog(routingContext, "Projection ready!")
                    }

                    return projections
                }
            } catch (e: DecodeException) {
                addLogMessageToRequestLog(routingContext, "Unable to parse projections: ", e)
            } catch (e: EncodeException) {
                addLogMessageToRequestLog(routingContext, "Unable to parse projections: ", e)
            }
        }

        return null
    }

    private suspend fun proceedWithRead(
        routingContext: RoutingContext,
        query: DynamoDBQuery,
    ) = runCatching {
        repo.show(query)
    }.onFailure {
        when (it) {
            is NoSuchElementException -> notFoundShow(routingContext)
            else -> {
                logger.warn(it) { "Failed Repository read" }

                failedShow(
                    routingContext,
                    JsonObject().put("error", "Service Unavailable..."),
                )
            }
        }
    }.onSuccess {
        val etag = routingContext.request().getHeader(IF_NONE_MATCH)
        val item = it.item

        setReadResponseHeaders(routingContext, item, it)

        when {
            etag != null && item.etag.equals(etag, ignoreCase = true) -> unChangedShow(routingContext)
            else -> postShow(routingContext, item, query.projections)
        }
    }

    private fun setReadResponseHeaders(
        routingContext: RoutingContext,
        item: E,
        itemResult: ItemResult<E>,
    ) {
        routingContext.response().putHeader(ETAG, item.etag)
        routingContext.response().putHeader("X-Cache", if (itemResult.isCacheHit) "HIT" else "MISS")
        routingContext.response().putHeader(
            "X-Repository-Pre-Operation-Nanos",
            "" + itemResult.preOperationProcessingTime,
        )
        routingContext.response().putHeader(
            "X-Repository-Operation-Nanos",
            "" + itemResult.operationProcessingTime,
        )
        routingContext.response().putHeader(
            "X-Repository-Post-Operation-Nanos",
            "" + itemResult.postOperationProcessingTime,
        )
    }

    override suspend fun prepareQuery(
        routingContext: RoutingContext,
        customQuery: String?,
    ) {
        val initialProcessNanoTime = System.nanoTime()
        routingContext.put(CONTROLLER_START_TIME, initialProcessNanoTime)
        val query = customQuery ?: routingContext.request().query()

        when {
            query == null || query.isEmpty() -> preProcessQuery(routingContext, ConcurrentHashMap())
            else -> {
                val queryMap = splitQuery(query)

                preProcessQuery(routingContext, queryMap)
            }
        }
    }

    override suspend fun processQuery(
        routingContext: RoutingContext,
        queryMap: MutableMap<String, List<String>>,
    ) {
        runCatching {
            val parameters =
                buildParameters(
                    routingContext,
                    queryMap,
                    fields,
                    methods,
                )

            when {
                parameters.left().isPresent -> postProcessQuery(routingContext, parameters.left().get())
                else -> {
                    val errorObject = JsonObject()
                    errorObject.put("request_errors", parameters.right().get())

                    routingContext.response().statusCode = 400
                    routingContext.put(BODY_CONTENT_TAG, errorObject)
                    routingContext.next()
                }
            }
        }.onFailure {
            logger.warn(it) { "Failed Repository params" }
        }
    }

    override suspend fun createIdObjectForIndex(
        routingContext: RoutingContext,
        query: DynamoDBQuery,
    ) {
        performIndex(routingContext, query)
    }

    override suspend fun performIndex(
        routingContext: RoutingContext,
        query: DynamoDBQuery,
    ) {
        logger.debug { "Running index" }

        val initialProcessNanoTime = routingContext.get<Long>(CONTROLLER_START_TIME)
        val request = routingContext.request()
        val pageToken: String? = request.getParam(PAGING_TOKEN_KEY)

        if (logger.isDebugEnabled()) {
            addLogMessageToRequestLog(routingContext, "Started index!")
        }

        when {
            isEndOfPaging(pageToken) -> returnEndOfPaging(routingContext, initialProcessNanoTime)
            else -> performPaging(routingContext, query, etagManager)
        }
    }

    private suspend fun performPaging(
        routingContext: RoutingContext,
        query: DynamoDBQuery,
        etagManager: ETagManager<E>?,
    ) {
        when {
            query.aggregate != null -> proceedWithAggregationIndex(routingContext, query)

            else -> {
                val hash = query.identifiers.hashCode()
                val etagItemListHashKey = type.simpleName + "_" + hash + "_" + "itemListEtags"
                val etagKey = query.baseEtagKey()

                if (logger.isDebugEnabled()) {
                    logger.debug { "EtagKey is: $etagKey" }

                    addLogMessageToRequestLog(routingContext, "Querypack ok, fetching etag for $etagKey")
                }

                when {
                    etagManager != null &&
                        query.etag != null &&
                        query.identifiers.ranges.size > 1 ->
                        runCatching {
                            etagManager.checkItemListEtag(
                                etagItemListHashKey,
                                etagKey,
                                query.etag!!,
                            )
                        }.onSuccess {
                            when (it) {
                                false -> proceedWithPagedIndex(routingContext, query)
                                else -> unChangedIndex(routingContext)
                            }
                        }.onFailure {
                            proceedWithPagedIndex(routingContext, query)
                        }

                    else -> proceedWithPagedIndex(routingContext, query)
                }
            }
        }
    }

    private fun returnEndOfPaging(
        routingContext: RoutingContext,
        initialProcessNanoTime: Long,
    ) {
        routingContext.put(
            BODY_CONTENT_TAG,
            encode(
                jsonObjectOf(
                    "error" to "You cannot page for the $END_OF_PAGING_KEY, " +
                        "this message means you have reached the end of the results requested.",
                ),
            ),
        )

        setStatusCodeAndAbort(400, routingContext, initialProcessNanoTime)
    }

    private fun isEndOfPaging(pageToken: String?) = pageToken != null && pageToken.equals(END_OF_PAGING_KEY, ignoreCase = true)

    override suspend fun proceedWithPagedIndex(
        routingContext: RoutingContext,
        query: DynamoDBQuery,
    ) {
        runCatching {
            repo.index(query)
        }.onSuccess {
            val items = it.itemList

            addIndexResponseHeaders(routingContext, it)

            routingContext.response().putHeader(ETAG, items.meta.etag)

            if (logger.isDebugEnabled()) {
                addLogMessageToRequestLog(
                    routingContext,
                    "Projections for output is: ${query.projections}",
                )
            }

            postIndex(routingContext, items, query.projections)
        }.onFailure {
            logger.warn(it) { "Failed Repository Index" }

            returnFailedIndex(routingContext, it)
        }
    }

    private fun addIndexResponseHeaders(
        routingContext: RoutingContext,
        itemsResult: ItemListResult<E>,
    ) {
        routingContext.response().putHeader(
            "X-Cache",
            when {
                itemsResult.isCacheHit -> "HIT"
                else -> "MISS"
            },
        )
        routingContext.response().putHeader(
            "X-Repository-Pre-Operation-Nanos",
            "" + itemsResult.preOperationProcessingTime,
        )
        routingContext.response().putHeader(
            "X-Repository-Operation-Nanos",
            "" + itemsResult.operationProcessingTime,
        )
        routingContext.response().putHeader(
            "X-Repository-Post-Operation-Nanos",
            "" + itemsResult.postOperationProcessingTime,
        )
    }

    private suspend fun returnFailedIndex(
        routingContext: RoutingContext,
        it: Throwable,
    ) {
        addLogMessageToRequestLog(
            routingContext,
            "FAILED",
            it,
        )

        failedIndex(routingContext, JsonObject().put("error", "Service unavailable..."))
    }

    override suspend fun proceedWithAggregationIndex(
        routingContext: RoutingContext,
        query: DynamoDBQuery,
    ) {
        if (logger.isDebugEnabled()) {
            addLogMessageToRequestLog(routingContext, "Started aggregation request")
        }

        val function = query.aggregate
        val baseEtagKey = query.baseEtagKey()
        val hashCode = function?.groupBy.hashCode()
        val etag = routingContext.request().getHeader(IF_NONE_MATCH)

        val etagKey =
            when (function?.function) {
                MIN -> "${baseEtagKey}_${function.field}_MIN$hashCode"
                MAX -> "${baseEtagKey}_${function.field}_MAX$hashCode"
                AggregateFunctions.AVG -> "${baseEtagKey}_${function.field}_AVG$hashCode"
                AggregateFunctions.SUM -> "${baseEtagKey}_${function.field}_SUM$hashCode"
                AggregateFunctions.COUNT -> "${baseEtagKey}_COUNT$hashCode"
                else -> throw IllegalArgumentException("Illegal Function!")
            }

        when {
            etag != null && etagManager != null -> {
                val hash = query.identifiers.hashCode()
                val etagItemListHashKey = "${type.simpleName}_${hash}_itemListEtags"

                runCatching {
                    etagManager!!.checkAggregationEtag(
                        etagItemListHashKey,
                        etagKey,
                        etag,
                    )
                }.onSuccess {
                    when (it) {
                        true -> unChangedIndex(routingContext)
                        else -> doAggregation(routingContext, query)
                    }
                }.onFailure {
                    doAggregation(routingContext, query)
                }
            }

            else -> doAggregation(routingContext, query)
        }
    }

    private suspend fun doAggregation(
        routingContext: RoutingContext,
        query: DynamoDBQuery,
    ) {
        runCatching {
            repo.aggregation(query)
        }.onSuccess {
            val newEtag = ModelUtils.returnNewEtag(it.hashCode().toLong())

            routingContext.response().putHeader(ETAG, newEtag)

            postAggregation(routingContext, it)
        }.onFailure {
            logger.warn(it) { "Failed Repository Aggregation" }

            addLogMessageToRequestLog(routingContext, "FAILED AGGREGATION, NULL")

            failedIndex(routingContext, JsonObject().put("error", "Aggregation Index failed..."))
        }
    }

    override suspend fun setIdentifiers(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        val hashAndRange = getAndVerifyId(routingContext)

        newRecords.forEach {
            it.setIdentifiers(
                jsonObjectOf(
                    "hash" to hashAndRange.hash,
                ),
            )
        }

        preSanitizeForCreate(newRecords, routingContext)
    }

    override suspend fun parseBodyForCreate(routingContext: RoutingContext) {
        val initialProcessNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        when {
            routingContext
                .body()
                .buffer()
                .bytes
                .isEmpty() ->
                try {
                    preVerifyNotExists(listOf(type.getDeclaredConstructor().newInstance()), routingContext)
                } catch (e: InstantiationException) {
                    logger.warn(e) { "Error parsing body" }

                    addLogMessageToRequestLog(routingContext, "Unable to create empty body!", e)

                    setStatusCodeAndAbort(500, routingContext, initialProcessNanoTime)
                } catch (e: IllegalAccessException) {
                    logger.warn(e) { "Error parsing body" }

                    addLogMessageToRequestLog(routingContext, "Unable to create empty body!", e)
                    setStatusCodeAndAbort(500, routingContext, initialProcessNanoTime)
                }
            else ->
                try {
                    val json = routingContext.body().asString()
                    val newRecords =
                        runCatching {
                            val values =
                                JsonArray(json).map { decodeValue(it.toString(), type) }

                            routingContext.put("MULTI_BODY", true)

                            values
                        }.recover {
                            val values =
                                listOf(decodeValue(json, type))

                            routingContext.put("MULTI_BODY", false)

                            values
                        }.getOrThrow()

                    preVerifyNotExists(newRecords, routingContext)
                } catch (e: DecodeException) {
                    logger.warn(e) { "Error parsing body" }

                    addLogMessageToRequestLog(routingContext, "Unable to parse body!", e)

                    setStatusCodeAndAbort(500, routingContext, initialProcessNanoTime)
                }
        }
    }

    @Suppress("UNCHECKED_CAST")
    override suspend fun verifyNotExists(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        val initialProcessNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val id = getAndVerifyId(routingContext)

        runCatching {
            when {
                id.isEmpty() ->
                    postVerifyNotExists(
                        newRecords = newRecords.map { it.setInitialValues(it) as E },
                        routingContext = routingContext,
                    )
                else ->
                    runCatching {
                        val items =
                            repo.index {
                                autoPaginate(true)
                                hashAndRange {
                                    hash = if (repo.hasRangeKey()) id.hash else null
                                    ranges =
                                        when {
                                            repo.hasRangeKey() ->
                                                newRecords.map { it.range(repo.identifierField) }.toSet()
                                            else ->
                                                newRecords.map { it.hash(repo.hashField) }.toSet()
                                        }
                                }
                            }

                        when {
                            items.itemList.meta.totalCount != 0 ->
                                setStatusCodeAndAbort(
                                    code = 409,
                                    routingContext = routingContext,
                                    initialProcessTime = initialProcessNanoTime,
                                )
                            else ->
                                postVerifyNotExists(
                                    newRecords = newRecords.map { it.setInitialValues(it) as E },
                                    routingContext = routingContext,
                                )
                        }
                    }.getOrThrow()
            }
        }.onFailure {
            when (it) {
                is InstantiationException -> {
                    logger.warn(it) { "Error parsing item" }
                    addLogMessageToRequestLog(routingContext, "Could not create item!", it)

                    setStatusCodeAndAbort(500, routingContext, initialProcessNanoTime)
                }

                is IllegalAccessException -> {
                    logger.warn(it) { "Error parsing item" }

                    addLogMessageToRequestLog(routingContext, "Could not create item!", it)

                    setStatusCodeAndAbort(500, routingContext, initialProcessNanoTime)
                }

                else -> {
                    logger.warn(it) { "Error verifying not exists item" }

                    addLogMessageToRequestLog(routingContext, "Could not verify non-existant item!", it)

                    setStatusCodeAndAbort(500, routingContext, initialProcessNanoTime)
                }
            }
        }
    }

    override suspend fun performCreate(
        newRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        runCatching {
            repo.batchCreate(newRecords)
        }.onSuccess {
            routingContext
                .response()
                .putHeader(CONTENT_TYPE, "application/json; charset=utf-8")

            postCreate(it.map { it.item }, routingContext)
        }.onFailure {
            logger.warn(it) { "Error during create" }

            addLogMessageToRequestLog(routingContext, "Could not create item!", it)

            val errorObject =
                JsonObject()
                    .put("create_error", "Unable to create record...")

            failedCreate(routingContext, errorObject)
        }
    }

    override suspend fun parseBodyForUpdate(routingContext: RoutingContext) {
        val initialProcessNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)

        when (val json = routingContext.body().asString()) {
            null -> setStatusCodeAndAbort(422, routingContext, initialProcessNanoTime)
            else ->
                try {
                    val updateRecords =
                        runCatching {
                            val values =
                                JsonArray(json).map { decodeValue(it.toString(), type) }

                            routingContext.put("MULTI_BODY", true)

                            values
                        }.recover {
                            val values =
                                listOf(decodeValue(json, type))

                            routingContext.put("MULTI_BODY", false)

                            values
                        }.getOrThrow()

                    preVerifyExistsForUpdate(updateRecords, routingContext)
                } catch (e: DecodeException) {
                    addLogMessageToRequestLog(routingContext, "Unable to parse body!", e)

                    setStatusCodeAndAbort(500, routingContext, initialProcessNanoTime)
                }
        }
    }

    override suspend fun verifyExistsForUpdate(
        updateRecords: List<E>,
        routingContext: RoutingContext,
    ) {
        val initialProcessNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val id = getAndVerifyId(routingContext)

        when {
            id.isEmpty() && repo.hasRangeKey() -> setStatusCodeAndAbort(400, routingContext, initialProcessNanoTime)
            else ->
                try {
                    runCatching {
                        val items =
                            repo.index {
                                autoPaginate(true)
                                hashAndRange {
                                    hash = if (repo.hasRangeKey()) id.hash else null
                                    ranges =
                                        when {
                                            repo.hasRangeKey() ->
                                                updateRecords.map { it.range(repo.identifierField) }.toSet()
                                            else ->
                                                updateRecords.map { it.hash(repo.hashField) }.toSet()
                                        }
                                }
                            }
                        val itemsMap =
                            items.itemList.items.associate {
                                when {
                                    repo.hasRangeKey() -> it.range(repo.identifierField) to it
                                    else -> it.range(repo.hashField) to it
                                }
                            }
                        val updateRecordsMap =
                            updateRecords.associate {
                                when {
                                    repo.hasRangeKey() -> it.range(repo.identifierField) to it
                                    else -> it.range(repo.hashField) to it
                                }
                            }

                        when {
                            updateRecords.any {
                                !itemsMap.contains(
                                    when {
                                        repo.hasRangeKey() -> it.range(repo.identifierField)
                                        else -> it.range(repo.hashField)
                                    },
                                )
                            } -> setStatusCodeAndAbort(404, routingContext, initialProcessNanoTime)

                            else -> {
                                val updateMap = itemsMap.map { it.value to updateRecordsMap[it.key]!! }.toMap()

                                preSanitizeForUpdate(
                                    updateMap = updateMap,
                                    routingContext = routingContext,
                                )
                            }
                        }
                    }.getOrThrow()
                } catch (e: DecodeException) {
                    logger.warn(e) { "Error parsing body" }

                    addLogMessageToRequestLog(routingContext, "Unable to parse body!", e)

                    setStatusCodeAndAbort(500, routingContext, initialProcessNanoTime)
                }
        }
    }

    override suspend fun performUpdate(
        updateFuncMap: Map<E, suspend E.(E) -> Unit>,
        routingContext: RoutingContext,
    ) {
        runCatching {
            repo.batchUpdate(updateFuncMap)
        }.onSuccess {
            val finalRecords =
                it.map {
                    it.item
                }

            routingContext
                .response()
                .putHeader(CONTENT_TYPE, "application/json; charset=utf-8")

            postUpdate(finalRecords, routingContext)
        }.onFailure {
            failedUpdate(routingContext, JsonObject().put("error", "Unable to update record..."))
        }
    }

    override suspend fun verifyExistsForDestroy(routingContext: RoutingContext) {
        val initialProcessNanoTime = routingContext.get<Long>(REQUEST_PROCESS_TIME_TAG)
        val id = getAndVerifyId(routingContext)

        when {
            id.isEmpty() && repo.hasRangeKey() -> setStatusCodeAndAbort(400, routingContext, initialProcessNanoTime)
            else ->
                try {
                    runCatching {
                        val items =
                            repo
                                .index {
                                    autoPaginate(true)
                                    hashAndRange {
                                        hash = id.hash
                                        ranges = id.ranges
                                    }
                                }.itemList.items
                        val idsMap =
                            items
                                .map {
                                    when {
                                        repo.hasRangeKey() -> it.range(repo.identifierField)
                                        else -> it.range(repo.hashField)
                                    }
                                }.toSet()

                        when {
                            id.ranges.all {
                                idsMap.contains(it)
                            } -> postVerifyExistsForDestroy(items, routingContext)

                            else -> setStatusCodeAndAbort(404, routingContext, initialProcessNanoTime)
                        }
                    }.getOrThrow()
                } catch (e: DecodeException) {
                    logger.warn(e) { "Error parsing body" }

                    addLogMessageToRequestLog(routingContext, "Unable to parse body!", e)

                    setStatusCodeAndAbort(500, routingContext, initialProcessNanoTime)
                }
        }
    }

    override suspend fun performDestroy(
        recordsForDestroy: List<E>,
        routingContext: RoutingContext,
    ) {
        val id = getAndVerifyId(routingContext)

        runCatching {
            repo.batchDelete(
                hash =
                    when {
                        repo.hasRangeKey() -> id.hash
                        else -> null
                    },
                ranges =
                    recordsForDestroy
                        .mapNotNull {
                            when {
                                repo.hasRangeKey() -> it.range(repo.identifierField)
                                else -> it.hash(repo.hashField)
                            }
                        }.takeIf { it.isNotEmpty() },
            )
        }.onSuccess {
            val finalRecords = it.map { it.item }

            postDestroy(finalRecords, routingContext)
        }.onFailure {
            failedDestroy(routingContext, JsonObject().put("error", "Unable to destroy record!"))
        }
    }

    private fun getAndVerifyId(routingContext: RoutingContext): HashAndRange = idSupplier(routingContext)

    private fun buildParameters(
        routingContext: RoutingContext,
        queryMap: MutableMap<String, List<String>>,
        fields: Array<Field>,
        methods: Array<Method>,
    ): Either<DynamoDBQuery, JsonObject> =
        runCatching {
            val id = getAndVerifyId(routingContext)
            val request = routingContext.request()
            val body = routingContext.body()?.asJsonObject()
            val errors = jsonObjectOf()

            val query =
                DynamoDBQuery.builder {
                    hashAndRange(id)
                    request.getHeader(IF_NONE_MATCH)?.let { etag(it) }
                    var gsi: String? = null

                    ((body?.fieldNames() ?: emptySet()) + queryMap.keys).forEach { key ->
                        logger.debug { "Parsing $key" }

                        when {
                            key.equals(LIMIT_KEY, ignoreCase = true) ||
                                key.equals(AUTO_PAGINATE_KEY, ignoreCase = true) ||
                                key.equals(MULTIPLE_IDS_KEY, ignoreCase = true) ||
                                key.equals(ORDER_BY_KEY, ignoreCase = true) ||
                                key.equals(PROJECTION_KEY, ignoreCase = true) ||
                                key.equals(PAGING_TOKEN_KEY, ignoreCase = true) ||
                                key.equals(GLOBAL_HASH_INDEX, ignoreCase = true) ||
                                key.equals(AGGREGATE_KEY, ignoreCase = true) ||
                                repo.hasField(fields, key) -> {
                                val value =
                                    body?.getValue(key)?.let { valueItem ->
                                        runCatching {
                                            JsonObject(valueItem.toString()).encode()
                                        }.getOrElse { valueItem.toString() }
                                    } ?: queryMap[key]?.first()

                                when {
                                    key.equals(PAGING_TOKEN_KEY, ignoreCase = true) -> value?.let { pageToken(it) }
                                    key.equals(AGGREGATE_KEY, ignoreCase = true) ->
                                        aggregate(
                                            value = value,
                                            queryMap = queryMap,
                                            errors = errors,
                                            routingContext = routingContext,
                                        )
                                    projection(key) -> projections(routingContext)

                                    else ->
                                        when {
                                            value == null ->
                                                errors.put(
                                                    "field_null",
                                                    "Value in '$key' cannot be null!",
                                                )

                                            key.equals(GLOBAL_HASH_INDEX, ignoreCase = true) ->
                                                runCatching {
                                                    Arrays.stream(methods).forEach { method ->
                                                        val indexHashkey =
                                                            method.getDeclaredAnnotation(
                                                                DynamoDbSecondaryPartitionKey::class.java,
                                                            )

                                                        if (indexHashkey != null) {
                                                            indexHashkey.indexNames.firstOrNull { it == value }?.let {
                                                                gsi(it)
                                                                gsi = it
                                                            }
                                                        }
                                                    }
                                                }.onFailure {
                                                    logger.warn(it) { "Error parsing GSI" }

                                                    errors.put(
                                                        "GSI_parse_error",
                                                        "Could not parse GSI: $value",
                                                    )
                                                }

                                            key.equals(ORDER_BY_KEY, ignoreCase = true) ->
                                                orderBy(
                                                    value = value,
                                                    methods = methods,
                                                    body = body,
                                                    queryMap = queryMap,
                                                    errors = errors,
                                                )

                                            key.equals(LIMIT_KEY, ignoreCase = true) -> limit(value, errors, key)
                                            key.equals(AUTO_PAGINATE_KEY, ignoreCase = true) ->
                                                autoPaginate(
                                                    value.toBooleanStrictOrNull() ?: false,
                                                )
                                            else -> filterParameter(key, gsi, errors, value)
                                        }
                                }
                            }

                            else ->
                                errors.put(
                                    "${key}_field_error",
                                    "This field does not exist on the selected resource.",
                                )
                        }
                    }

                    if (lsi == null && !repo.paginationIdentifier.isNullOrBlank()) {
                        lsi(PAGINATION_INDEX)
                    }
                }

            logger.debug { "Built params" }

            when {
                errors.isEmpty -> left<DynamoDBQuery, JsonObject>(query)
                else -> right(errors)
            }
        }.getOrElse {
            right(
                jsonObjectOf(
                    "processing_error" to "Could not parse parameters",
                    "error_message" to it.message,
                ),
            )
        }

    private fun DynamoDBQuerybuilder.projections(routingContext: RoutingContext) {
        runCatching {
            routingContext.request().getParam(PROJECTION_KEY)?.let { projectionQuery ->
                val projection = JsonObject(projectionQuery)

                projection
                    .getJsonArray(PROJECTION_FIELDS_KEY, null)
                    ?.stream()
                    ?.map { it.toString() }
                    ?.collect(toList())
                    ?.filterNotNull()
                    ?.toSet()
            }
        }.onFailure {
            logger.error(it) { "Unable to parse projections" }
        }.onSuccess {
            it?.let {
                projections(it)

                if (logger.isDebugEnabled()) {
                    addLogMessageToRequestLog(routingContext, "Index projections: $it")
                }
            }
        }
    }

    private fun DynamoDBQuerybuilder.aggregate(
        value: String?,
        queryMap: MutableMap<String, List<String>>,
        errors: JsonObject,
        routingContext: RoutingContext,
    ) {
        when {
            value != null && queryMap[ORDER_BY_KEY] != null -> {
                runCatching {
                    val aggregateFunction = decodeValue(value, AggregateFunction::class.java)

                    if (!(aggregateFunction!!.function == MIN || aggregateFunction.function == MAX)) {
                        errors.put(
                            "aggregate_error",
                            "AVG, SUM and COUNT cannot be performed in conjunction with ordering...",
                        )
                    }

                    aggregateFunction
                }.onFailure {
                    addLogMessageToRequestLog(routingContext, "Unable to parse projections", it)

                    errors.put("aggregate_query_error", "Unable to parse json...")
                }.getOrNull()
            }

            value != null -> {
                runCatching {
                    decodeValue(value, AggregateFunction::class.java)
                }.onFailure {
                    addLogMessageToRequestLog(routingContext, "Unable to parse projections", it)

                    errors.put("aggregate_query_error", "Unable to parse json...")
                }.getOrNull()
            }

            else -> null
        }?.let { aggregateFunction(it) }
    }

    private fun DynamoDBQuerybuilder.orderBy(
        value: String?,
        methods: Array<Method>,
        body: @Nullable JsonObject?,
        queryMap: MutableMap<String, List<String>>,
        errors: JsonObject,
    ) {
        runCatching {
            JsonObject(value)
        }.getOrNull()?.let { orderByParam ->
            runCatching {
                val orderByParameter = decodeValue(value, OrderByParameter::class.java)

                when {
                    orderByParameter.isValid -> {
                        parseOrderBy(
                            methods,
                            orderByParameter,
                            body,
                            queryMap,
                        )?.let { jsonObject ->
                            jsonObject.map.forEach {
                                errors.put(it.key, it.value)
                            }
                        } ?: orderBy(orderByParameter)
                    }

                    else ->
                        errors.put(
                            "orderBy_parameter_error",
                            "Field cannot be null!",
                        )
                }
            }.onFailure {
                logger.warn(it) { "Error parsing Order By" }

                errors.put(
                    "orderBy_parse_error",
                    "Could not parse: $orderByParam",
                )
            }
        } ?: errors.put("orderBy", "Unparseable: $value")
    }

    private fun DynamoDBQuerybuilder.limit(
        value: String?,
        errors: JsonObject,
        key: String?,
    ) {
        runCatching {
            logger.debug { "Parsing limit.." }

            val valueAsLimit = runCatching { Integer.parseInt(value) }.getOrNull() ?: -999

            logger.debug { "Limit is: $valueAsLimit" }

            when {
                valueAsLimit == -999 ->
                    errors.put(
                        "${key}_invalid_error",
                        "Limit could not be parsed into whole positive Integer!",
                    )

                valueAsLimit < 1 ->
                    errors.put(
                        "${key}_negative_error",
                        "Limit must be a whole positive Integer!",
                    )

                valueAsLimit > 1000 ->
                    errors.put(
                        "${key}_exceed_max_error",
                        "Maximum limit is 1000!",
                    )

                else -> {
                    limit(valueAsLimit)

                    logger.debug { "Limit value ok: $valueAsLimit" }
                }
            }
        }.onFailure {
            errors.put("${key}_error", "Limit must be a whole positive Integer!")
        }
    }

    private fun DynamoDBQuerybuilder.filterParameter(
        key: String,
        gsi: String?,
        errors: JsonObject,
        value: String?,
    ) = runCatching {
        when {
            key == repo.hashIdentifier && gsi == null ->
                errors.put(
                    "${key}_invalid_filter_value",
                    "Hash cannot be filtered upon when not using a GSI",
                )

            gsi == key ->
                errors.put(
                    "${key}_invalid_filter_value",
                    "GSI Hash cannot be filtered upon when using a GSI",
                )

            else ->
                value?.let { paramValue ->
                    val result = repo.parseParam(paramValue, key)

                    handleParsedParam(result, errors, key)
                }
        }
    }.recoverCatching { throwable ->
        logger.debug(throwable) {
            "Could not parse filterParams as a JsonObject, attempting array..."
        }

        runCatching {
            val (filters, filterErrors) =
                JsonArray(value)
                    .map { jsonObjectAsString ->
                        repo.parseParam(jsonObjectAsString.toString(), key)
                    }.partition {
                        it.left().isPresent
                    }

            when {
                filterErrors.isEmpty() ->
                    filter(
                        mapOf(
                            key to filters.map { it.left().get() },
                        ),
                    )

                else ->
                    filterErrors
                        .map {
                            it.right().get()
                        }.forEach { jsonObject ->
                            jsonObject.map.forEach {
                                errors.put(it.key, it.value)
                            }
                        }
            }
        }.onFailure {
            logger.error(it) { "Unable to parse json as array: $value" }

            errors.put("${key}_value_json_error", "Unable to parse this json...")
        }
    }.getOrThrow()

    private fun DynamoDBQuerybuilder.handleParsedParam(
        result: Either<FilterParameter, JsonObject>,
        errors: JsonObject,
        key: String,
    ) {
        val param = result.left()
        val error = result.right()

        when {
            error.isPresent ->
                error.get().map.forEach {
                    errors.put(it.key, it.value)
                }

            else -> {
                logger.debug { encodePrettily(param) }

                filter(
                    mapOf(
                        key to listOf(param.get()),
                    ),
                )
            }
        }
    }

    private fun projection(key: String?) =
        key.equals(PROJECTION_KEY, ignoreCase = true) ||
            key.equals(MULTIPLE_IDS_KEY, ignoreCase = true)

    private fun parseOrderBy(
        methods: Array<Method>,
        orderByParameter: OrderByParameter,
        body: JsonObject?,
        queryMap: Map<String, List<String>>,
    ): JsonObject? {
        var indexName: String? = null

        Arrays.stream(methods).forEach { method ->
            val indexRangeKey =
                method.getDeclaredAnnotation(
                    DynamoDbSecondarySortKey::class.java,
                )

            if (indexRangeKey != null &&
                stripGet(method.name) == orderByParameter.field
            ) {
                val keys = body?.fieldNames() ?: queryMap.keys
                val gsiValue =
                    body
                        ?.getValue(GLOBAL_HASH_INDEX)
                        ?.let { valueItem ->
                            runCatching {
                                JsonObject(valueItem.toString()).encode()
                            }.getOrElse { valueItem.toString() }
                        } ?: queryMap[GLOBAL_HASH_INDEX]?.firstOrNull()

                when {
                    keys.contains(GLOBAL_HASH_INDEX) &&
                        indexRangeKey.indexNames.any { it == gsiValue } -> {
                        indexRangeKey.indexNames
                            .firstOrNull { it == gsiValue }
                            ?.let {
                                indexName = it
                            }
                    }

                    else -> {
                        indexName = indexRangeKey.indexNames.firstOrNull()
                    }
                }
            }
        }

        return when {
            indexName.isNullOrBlank() -> {
                jsonObjectOf(
                    "orderBy_parameter_error" to "This is not a valid remoteIndex!",
                )
            }
            else -> null
        }
    }

    private fun stripGet(string: String): String {
        val newString = string.replace("get", "")
        val c = newString.toCharArray()
        c[0] = c[0] + 32

        return String(c)
    }

    companion object {
        const val GLOBAL_HASH_INDEX = "globalHashIndex"
        const val PROJECTION_KEY = "projection"
        const val PROJECTION_FIELDS_KEY = "fields"
        const val ORDER_BY_KEY = "orderBy"
        const val AGGREGATE_KEY = "aggregate"
        const val LIMIT_KEY = "limit"
        const val AUTO_PAGINATE_KEY = "autoPaginate"

        const val MULTIPLE_IDS_KEY = "ids"
        const val PAGING_TOKEN_KEY = "pageToken"
        const val END_OF_PAGING_KEY = "END_OF_LIST"

        const val CONTROLLER_START_TIME = "controllerStartTimeTag"

        inline fun <reified T> create(
            vertx: Vertx = Vertx.currentContext().owner(),
            appConfig: JsonObject,
            repo: DynamoDBRepository<T>,
        ) where T : DynamoDBModel, T : Model, T : Cacheable, T : ETagable =
            DynamoDbRestController(
                vertx = vertx,
                type = T::class.java,
                appConfig = appConfig,
                repo = repo,
                idSupplier = defaultSupplier,
                etagManager = repo.etagManager,
            )

        val defaultSupplier: (RoutingContext) -> HashAndRange = { context ->
            runCatching {
                val body = context.body()
                val idsQuery = context.request().params()["ids"]

                body
                    ?.asJsonObject()
                    ?.getJsonArray("ids")
                    ?.map {
                        it.toString()
                    }?.toSet() ?: idsQuery?.let { s ->
                    val queryValues = JsonArray(s).list
                    val setValues = queryValues.map { it.toString() }.toSet()

                    setValues
                }
            }.getOrNull()?.let {
                hashAndRange {
                    hash = context.pathParam("hash")
                    ranges = it
                }
            } ?: hashAndRange {
                hash = context.pathParam("hash")
                range = context.pathParam("range")
            }
        }
    }
}
