@file:Suppress("unused")

package no.solibo.digipost

import no.digipost.api.client.DigipostClient
import no.digipost.api.client.DigipostClientConfig
import no.digipost.api.client.SenderId
import no.digipost.api.client.representations.Document
import no.digipost.api.client.representations.EmailNotification
import no.digipost.api.client.representations.FileType
import no.digipost.api.client.representations.FileType.PDF
import no.digipost.api.client.representations.Identification
import no.digipost.api.client.representations.IdentificationResult
import no.digipost.api.client.representations.ListedTime
import no.digipost.api.client.representations.Message
import no.digipost.api.client.representations.MessageDelivery
import no.digipost.api.client.representations.MessageRecipient
import no.digipost.api.client.representations.NameAndAddress
import no.digipost.api.client.representations.NorwegianAddress
import no.digipost.api.client.representations.OrganisationNumber
import no.digipost.api.client.representations.PersonalIdentificationNumber
import no.digipost.api.client.representations.PrintDetails
import no.digipost.api.client.representations.PrintDetails.NondeliverableHandling.RETURN_TO_SENDER
import no.digipost.api.client.representations.PrintDetails.PrintColors.COLORS
import no.digipost.api.client.representations.PrintRecipient
import no.digipost.api.client.representations.inbox.Inbox
import no.digipost.api.client.security.Signer
import no.digipost.api.datatypes.types.invoice.Invoice
import java.io.InputStream
import java.math.BigDecimal
import java.net.URI
import java.security.PrivateKey
import java.time.ZonedDateTime
import java.time.ZonedDateTime.now
import java.util.UUID

data class DigiPostConfig(
    val host: String,
    val sender: Long,
    val key: PrivateKey,
)

data class DigiPostLetter(
    val subject: String,
    val content: InputStream,
    val fileType: FileType = PDF,
)

data class DigiPostAddress(
    val fullName: String,
    val adresseLineOne: String,
    val adresseLineTwo: String? = null,
    val postalArea: String,
    val city: String,
) {
    fun toDigipost(): NameAndAddress =
        NameAndAddress(
            fullName,
            adresseLineOne,
            adresseLineTwo,
            postalArea,
            city,
        )
}

data class DigiPostInvoiceInfo(
    val link: String? = null,
    val dueDate: ZonedDateTime,
    val sum: BigDecimal,
    val account: String,
    val kid: String,
)

data class DigiPostEmailInfo(
    val email: String,
    val subject: String,
    val text: String,
    val at: ZonedDateTime = now(),
)

class DigiPost(
    config: DigiPostConfig,
) {
    private val senderId = SenderId.of(config.sender)
    private val signer = Signer.using(config.key)
    private val client =
        DigipostClient(
            DigipostClientConfig.newConfiguration().digipostApiUri(URI.create(config.host)).build(),
            senderId.asBrokerId(),
            signer,
        )
    private val defaultSoliboReturnAddress =
        PrintRecipient(
            "Solibo AS",
            NorwegianAddress(
                "Dronning Eufemias gate 16",
                "0191",
                "Oslo",
            ),
        )

    fun search(pnr: String): IdentificationResult =
        client.identifyRecipient(
            Identification(PersonalIdentificationNumber(pnr)),
        )

    fun searchOrg(orgNr: String): IdentificationResult =
        client.identifyRecipient(
            Identification(OrganisationNumber(orgNr)),
        )

    fun sendAdHocLetter(
        address: DigiPostAddress,
        emailInfo: DigiPostEmailInfo? = null,
        primary: DigiPostLetter,
        fallbackPrint: Boolean = false,
        vararg attachments: DigiPostLetter,
    ): MessageDelivery {
        val print = buildPrintJob(address)

        return sendLetter(primary, attachments, emailInfo) {
            when {
                fallbackPrint -> {
                    recipient(address.toDigipost())
                    printDetails(print)
                }
                else -> recipient(address.toDigipost())
            }
        }
    }

    fun sendAdHocLetterToPersonnummer(
        address: DigiPostAddress,
        personnummer: String,
        emailInfo: DigiPostEmailInfo? = null,
        primary: DigiPostLetter,
        fallbackPrint: Boolean = false,
        vararg attachments: DigiPostLetter,
    ): MessageDelivery {
        val pin = PersonalIdentificationNumber(personnummer)
        val print = buildPrintJob(address)

        return sendLetter(primary, attachments, emailInfo) {
            when {
                fallbackPrint -> recipient(MessageRecipient(pin, print))
                else -> recipient(pin)
            }
        }
    }

    fun sendAdHocLetterToOrgnr(
        address: DigiPostAddress,
        orgnr: String,
        emailInfo: DigiPostEmailInfo? = null,
        primary: DigiPostLetter,
        fallbackPrint: Boolean = false,
        vararg attachments: DigiPostLetter,
    ): MessageDelivery {
        val pin = OrganisationNumber(orgnr)
        val print = buildPrintJob(address)

        return sendLetter(primary, attachments, emailInfo) {
            when {
                fallbackPrint -> recipient(MessageRecipient(pin, print))
                else -> recipient(pin)
            }
        }
    }

    fun sendGjenpartToPersonnummer(
        address: DigiPostAddress,
        personnummer: String,
        emailInfo: DigiPostEmailInfo? = null,
        primary: DigiPostLetter,
        fallbackPrint: Boolean = false,
        vararg attachments: DigiPostLetter,
    ): MessageDelivery {
        val pin = PersonalIdentificationNumber(personnummer)
        val print = buildPrintJob(address)

        return sendLetter(primary, attachments, emailInfo) {
            when {
                fallbackPrint -> recipient(MessageRecipient(pin, print))
                else -> recipient(pin)
            }
        }
    }

    fun sendInvoiceToPersonnummer(
        address: DigiPostAddress,
        personnummer: String,
        invoiceInfo: DigiPostInvoiceInfo,
        emailInfo: DigiPostEmailInfo? = null,
        primary: DigiPostLetter,
        fallbackPrint: Boolean = false,
        vararg attachments: DigiPostLetter,
    ): MessageDelivery {
        val pin = PersonalIdentificationNumber(personnummer)
        val print = buildPrintJob(address)

        return sendInvoice(primary, attachments, invoiceInfo, emailInfo) {
            when {
                fallbackPrint -> recipient(MessageRecipient(pin, print))
                else -> recipient(pin)
            }
        }
    }

    fun sendGjenpartToOrgnr(
        address: DigiPostAddress,
        orgnr: String,
        emailInfo: DigiPostEmailInfo? = null,
        primary: DigiPostLetter,
        fallbackPrint: Boolean = false,
        vararg attachments: DigiPostLetter,
    ): MessageDelivery {
        val pin = OrganisationNumber(orgnr)
        val print = buildPrintJob(address)

        return sendLetter(primary, attachments, emailInfo) {
            when {
                fallbackPrint -> recipient(MessageRecipient(pin, print))
                else -> recipient(pin)
            }
        }
    }

    fun sendInvoiceToOrgnr(
        address: DigiPostAddress,
        orgnr: String,
        invoiceInfo: DigiPostInvoiceInfo,
        emailInfo: DigiPostEmailInfo? = null,
        primary: DigiPostLetter,
        fallbackPrint: Boolean = false,
        vararg attachments: DigiPostLetter,
    ): MessageDelivery {
        val pin = OrganisationNumber(orgnr)
        val print = buildPrintJob(address)

        return sendInvoice(primary, attachments, invoiceInfo, emailInfo) {
            when {
                fallbackPrint -> recipient(MessageRecipient(pin, print))
                else -> recipient(pin)
            }
        }
    }

    fun readInbox(): Inbox = client.getInbox(senderId)

    private fun sendLetter(
        primary: DigiPostLetter,
        attachments: Array<out DigiPostLetter>,
        emailInfo: DigiPostEmailInfo? = null,
        recipientSetter: Message.MessageBuilder.() -> Unit,
    ): MessageDelivery {
        val uuid = UUID.randomUUID()
        val attachmentsUUIDs = attachments.associate { it.subject to UUID.randomUUID() }
        val primaryDocument =
            Document(
                uuid,
                primary.subject,
                primary.fileType,
                null,
                null,
                emailInfo?.let {
                    EmailNotification(
                        emailInfo.email,
                        emailInfo.subject,
                        emailInfo.text,
                        listOf(ListedTime(emailInfo.at)),
                    )
                },
                null,
                null,
                null,
                null,
            )
        val attachmentDocuments =
            attachments.associate {
                Document(attachmentsUUIDs[it.subject], it.subject, it.fileType) to it.content
            }

        return send(uuid, primaryDocument, recipientSetter, attachmentDocuments, primary)
    }

    private fun sendInvoice(
        primary: DigiPostLetter,
        attachments: Array<out DigiPostLetter>,
        invoiceInfo: DigiPostInvoiceInfo,
        emailInfo: DigiPostEmailInfo? = null,
        recipientSetter: Message.MessageBuilder.() -> Unit,
    ): MessageDelivery {
        val uuid = UUID.randomUUID()
        val attachmentsUUIDs = attachments.associate { it.subject to UUID.randomUUID() }
        val invoiceDataType =
            Invoice(
                null,
                invoiceInfo.dueDate,
                invoiceInfo.sum,
                invoiceInfo.account,
                invoiceInfo.kid,
            )
        val primaryDocument =
            Document(
                uuid,
                primary.subject,
                primary.fileType,
                null,
                null,
                emailInfo?.let {
                    EmailNotification(
                        emailInfo.email,
                        emailInfo.subject,
                        emailInfo.text,
                        listOf(ListedTime(emailInfo.at)),
                    )
                },
                null,
                null,
                null,
                invoiceDataType,
            )
        val attachmentDocuments =
            attachments.associate {
                Document(attachmentsUUIDs[it.subject], it.subject, it.fileType) to it.content
            }

        return send(uuid, primaryDocument, recipientSetter, attachmentDocuments, primary)
    }

    private fun send(
        uuid: UUID,
        primaryDocument: Document,
        recipientSetter: Message.MessageBuilder.() -> Unit,
        attachmentDocuments: Map<Document, InputStream>,
        primary: DigiPostLetter,
    ): MessageDelivery {
        val messageBuilder = Message.newMessage(uuid, primaryDocument)
        recipientSetter(messageBuilder)
        if (attachmentDocuments.isNotEmpty()) messageBuilder.attachments(attachmentDocuments.keys)
        val message = messageBuilder.build()

        val clientRequest =
            client
                .createMessage(message)
                .addContent(primaryDocument, primary.content)
        attachmentDocuments.forEach { (k, v) -> clientRequest.addContent(k, v) }

        return clientRequest.send()
    }

    private fun buildPrintJob(address: DigiPostAddress) =
        PrintDetails(
            PrintRecipient(
                address.fullName,
                NorwegianAddress(
                    address.adresseLineOne,
                    address.adresseLineTwo,
                    address.postalArea,
                    address.city,
                ),
            ),
            defaultSoliboReturnAddress,
            COLORS,
            RETURN_TO_SENDER,
        )
}
