package no.tornado.inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * <p>Even if you opt to eagerly load a bean, it will still be wrapped in a BeanProxy. This
 * makes it possible to apply interceptor to eagerly loaded beans as well.</p>
 *
 * </p>Only beanProviderMethods that don't return an interface will result in beans not being proxied.</p>
 *
 * <p>The actual bean is not created before the first method call to it.</p>
 *
 * <p>If an Interceptor was supplied on the beanProviderMethod, the method invocation is intercepted.</p>
 */
public class BeanProxy implements InvocationHandler {
    private static Logger logger = LoggerFactory.getLogger(BeanProxy.class);
    private BeanInfo beanInfo;
    private Object delegate;
    private Interceptor interceptor;
    
    public BeanProxy(BeanInfo beanInfo) {
        this.beanInfo = beanInfo;
    }

    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        if (delegate == null) {
            logger.info("Lazily instantiating bean " + beanInfo.getBeanName());
            delegate = beanInfo.instantiate();
            Intercept intercept = beanInfo.getBeanCreationMethod().getAnnotation(Intercept.class);
            if (intercept != null)
                interceptor = (Interceptor) ApplicationContext.getBean(intercept.value());
        }

        if (interceptor != null)
            return interceptor.intercept(delegate, delegate.getClass().getMethod(method.getName(), method.getParameterTypes()), args);

        try {
          return delegate.getClass().getMethod(method.getName(), method.getParameterTypes()).invoke(delegate, args);
        } catch (InvocationTargetException ite) {
            if (ite.getTargetException() != null)
                throw ite.getTargetException();
            else
                throw ite;
        }
    }

    public Object getDelegate() {
        return delegate;
    }
}
