package no.tornado.inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

/**
 * <p>The BeanInfo object holds references to beans in the ApplicationContext. The BeanInfo
 * is responsible for proxying objects, or instantiating them when needed if the
 * beanProviderMethod does not return an interface. (Only interfaces can be proxied
 * with the default Java proxy).</p>
 *
 * <p>You normally never need to bother with the BeanInfo object directly.</p>
 */
public class BeanInfo {
    private static Logger logger = LoggerFactory.getLogger(BeanInfo.class);
    private Method beanCreationMethod;
    private Object provider;
    private Object bean;

    BeanInfo() {
    }

    public BeanInfo(Method beanCreationMethod, Object provider) {
        this.beanCreationMethod = beanCreationMethod;
        this.provider = provider;
    }

    public boolean isEager() {
        return beanCreationMethod.getAnnotation(Eager.class) != null;
    }

    public Object getBean() {
        if (bean == null) {
            if (beanCreationMethod.getReturnType().isInterface()) {
                bean = Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(), new Class[]{beanCreationMethod.getReturnType()}, new BeanProxy(this));
            } else {
                logger.info("Eagerly creating no-interface bean " + getBeanName());
                try {
                    bean = instantiate();
                } catch (InvocationTargetException ex) {
                    throw new BeanCreationException(this, ex.getCause() != null ? ex.getCause() : ex);
                } catch (IllegalAccessException ex) {
                    throw new BeanCreationException(this, ex);
                }
            }
        }
        
        return bean;
    }

    public Object instantiate() throws IllegalAccessException, InvocationTargetException {
        if (!beanCreationMethod.isAccessible())
            beanCreationMethod.setAccessible(true);
        
        Object delegate = beanCreationMethod.invoke(provider);
        ApplicationContext.postProcessBean(this, delegate);
        return delegate;
    }

    public boolean equals(Object obj) {
        if (obj instanceof BeanInfo) {
            return ((BeanInfo) obj).getBeanName().equals(getBeanName());
        } else if (obj instanceof String)
            return obj.equals(getBeanName());
        return false;
    }

    public String getBeanName() {
        return beanCreationMethod.getName();
    }

    public Object getProvider() {
        return provider;
    }

    public void setProvider(Object provider) {
        this.provider = provider;
    }

    public Method getBeanCreationMethod() {
        return beanCreationMethod;
    }

    public void setBeanCreationMethod(Method beanCreationMethod) {
        this.beanCreationMethod = beanCreationMethod;
    }
}
