package no.tornado.inject;

import org.apache.commons.beanutils.BeanUtilsBean2;
import org.apache.commons.beanutils.PropertyUtilsBean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>Even if you opt to eagerly load a bean, it will still be wrapped in a BeanProxy. This
 * makes it possible to apply interceptor to eagerly loaded beans as well.</p>
 * <p/>
 * </p>Only beanProviderMethods that don't return an interface will result in beans not being proxied.</p>
 * <p/>
 * <p>The actual bean is not created before the first method call to it.</p>
 * <p/>
 * <p>If an MethodInterceptor was supplied on the beanProviderMethod, the method invocation is intercepted.</p>
 */
public class BeanProxy implements InvocationHandler {
    private static Logger logger = LoggerFactory.getLogger(BeanProxy.class);
    private static final PropertyUtilsBean propertyUtils = BeanUtilsBean2.getInstance().getPropertyUtils();
    private BeanInfo beanInfo;
    private Object delegate;
    private MethodInterceptor methodInterceptor;

    public BeanProxy(BeanInfo beanInfo) {
        this.beanInfo = beanInfo;
    }

    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        if (delegate == null) {
            logger.info("Lazily instantiating bean " + beanInfo.getBeanName());
            delegate = beanInfo.instantiate();
            Intercept intercept = beanInfo.getBeanCreationMethod().getAnnotation(Intercept.class);
            if (intercept != null)
                methodInterceptor = (MethodInterceptor) ApplicationContext.getBean(intercept.value());

        }

        method = getDelegateMethod(method);
        
        CacheStore cacheStore = null;
        String cacheKey = null;
        Cache cache = method.getAnnotation(Cache.class);
        if (cache != null) {
            cacheStore = (CacheStore) ApplicationContext.getBean(cache.value());
            cacheKey = cacheStore.getKeyGenerator().generate(beanInfo.getBeanName(), method, args);
            Object cachedResult = cacheStore.lookup(cacheKey);
            if (cachedResult != null)
                return cachedResult;
        }

        try {
            Object result;

            if (methodInterceptor != null)
                result = methodInterceptor.intercept(delegate, method, args);
            else
                result = method.invoke(delegate, args);

            Map<String, Serializable> cacheIndexes = getCacheIndexes(method, args);

            InvalidateCaches invalidateCaches = method.getAnnotation(InvalidateCaches.class);
            if (invalidateCaches != null) {
                for (String cacheName : invalidateCaches.value()) {
                    CacheStore cs = (CacheStore) ApplicationContext.getBean(cacheName);
                    if (cs == null) {
                        logger.error(beanInfo.getBeanName() + " is configured to invalidate non-existent cache " + cacheName);
                        continue;
                    }
                    if (cs.getCacheInterceptor() != null)
                        cs.getCacheInterceptor().beforeInvalidate(beanInfo, method, args, cacheIndexes);
                    cs.invalidate(cacheIndexes);
                }
            }

            if (cacheKey != null && (result == null || result instanceof Serializable)) {
                if (cacheStore.getCacheInterceptor() != null)
                    cacheStore.getCacheInterceptor().beforeStore(beanInfo, method, args, cacheIndexes, cacheKey);

                Long expires = cache.expiresMinutes() == 0 ? null : (new Date().getTime() + cache.expiresMinutes() * 60000);
                cacheStore.store(cacheKey, (Serializable) result, cacheIndexes, expires);
            }

            return result;

        } catch (InvocationTargetException ite) {
            if (ite.getTargetException() != null)
                throw ite.getTargetException();
            else
                throw ite;
        }

    }

    private Method getDelegateMethod(Method method) throws NoSuchMethodException {
        return delegate.getClass().getMethod(method.getName(), method.getParameterTypes());
    }

    private Map<String, Serializable> getCacheIndexes(Method method, Object[] args) {
        Map<String, Serializable> invalidateIndexes = new HashMap<>();
        for (int i = 0; i < method.getParameterAnnotations().length; i++) {
            Annotation[] annotationsForParameter = method.getParameterAnnotations()[i];
            for (Annotation a : annotationsForParameter) {
                if (a.annotationType().equals(CacheIndex.class)) {
                    CacheIndex cacheIndex = (CacheIndex) a;
                    invalidateIndexes.put(cacheIndex.key(), getArgValue(args[i], cacheIndex.property()));
                }
            }
        }
        return invalidateIndexes;
    }

    private Serializable getArgValue(Object object, String property) {
        if (object == null)
            return null;

        if ("".equals(property))
            return (Serializable) object;

        try {
            return (Serializable) propertyUtils.getNestedProperty(object, property);
        } catch (Exception e) {
            logger.warn("Failed to extract argValue for " + object + "." + property, e);
            return null;
        }
    }

    public Object getDelegate() {
        return delegate;
    }
}
