package no.tornado.inject;

import org.apache.commons.collections.map.LRUMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.*;
import java.util.*;

public class CacheStore implements Serializable, BeanNameAware {
    private static Log logger = LogFactory.getLog(CacheStore.class);

    public static final Integer DEFAULT_CACHE_MINUTES = 60;
    public static final Integer DEFAULT_MAX_OBJECTS = 8192;

    private String beanName;
    private Integer cacheMinutes = DEFAULT_CACHE_MINUTES;
    private Integer maxObjects = DEFAULT_MAX_OBJECTS;
    private CacheKeyGenerator keyGenerator;
    private CacheInterceptor cacheInterceptor;

    private Map<String/*indexKey*/, Map<Serializable/*indexValue*/, Map<String/*cacheKey*/, Void>>> indexes;
    private Map<String, CacheEntry> store;

    public CacheStore() {
        this(DEFAULT_MAX_OBJECTS);
    }

    @SuppressWarnings("unchecked")
    public CacheStore(Integer maxObjects) {
        store = Collections.synchronizedMap(new IndexedLRUMap(maxObjects));
        indexes = new HashMap<>();
        keyGenerator = new DefaultCacheKeyGenerator();
    }

    public String toString() {
        StringBuilder s = new StringBuilder(beanName + "\n");
        s.append("Cache timeout: " + cacheMinutes + " minutes\n");
        s.append("Max objects: " + maxObjects + "\n");
        s.append("Current objects: " + store.size() + "\n");
        s.append("Current size: " + calculateObjectSize() + " bytes\n");
        s.append("Registered indexes: " + indexes.keySet());
        return s.toString();
    }

    private long calculateObjectSize() {
        long size = 0l;
        for (CacheEntry entry : store.values())
            if (entry.data != null)
                size += entry.data.length;
        return size;
    }

    public void store(String cacheKey, Serializable object, Map<String, Serializable> indexEntries, Long expires) throws IOException {
        CacheEntry cacheEntry = new CacheEntry();
        cacheEntry.setKey(cacheKey);
        cacheEntry.setIndexEntries(indexEntries);
        cacheEntry.setCachedObject(object);
        cacheEntry.setExpires(expires);

        store.put(cacheKey, cacheEntry);

        for (Map.Entry<String, Serializable> indexEntry : indexEntries.entrySet()) {

            Map<Serializable, Map<String, Void>> keysForThisIndex = indexes.get(indexEntry.getKey());
            if (keysForThisIndex == null) {
                keysForThisIndex = new HashMap<>();
                indexes.put(indexEntry.getKey(), keysForThisIndex);
            }
            
            Map<String, Void> valuesForThisIndex = keysForThisIndex.get(indexEntry.getValue());
            if (valuesForThisIndex == null) {
                valuesForThisIndex = new HashMap<>();
                keysForThisIndex.put(indexEntry.getValue(), valuesForThisIndex);
            }
            
            valuesForThisIndex.put(cacheKey, null);
        }
    }
    
    public Object lookup(String key) throws ClassNotFoundException, IOException {
        CacheEntry entry = store.get(key);
        if (entry != null) {
            logger.debug(beanName + " hit: " + key);
            return entry.getCachedObject();
        }
        
        logger.debug(beanName + " miss: " + key);
        return null;
    }

    public boolean invalidate(Map<String, Serializable> invalidateIndexes) {
        if (invalidateIndexes.isEmpty()) {
            logger.debug("Invalidating " + beanName);
            synchronized (store) {
                store.clear();
                indexes.clear();
                return true;
            }
        }
        
        logger.debug("Invalidating " + beanName + " entries with indexes " + invalidateIndexes);

        for (Map.Entry<String, Serializable> indexEntry : invalidateIndexes.entrySet()) {
            Map<Serializable, Map<String, Void>> keysForThisIndex = indexes.get(indexEntry.getKey());
            if (keysForThisIndex != null) {
                Map<String, Void> valuesForThisIndex = keysForThisIndex.get(indexEntry.getValue());
                if (valuesForThisIndex != null) {
                    for (String cacheKey : valuesForThisIndex.keySet())
                        invalidate(cacheKey);
                }
                return true;
            }
        }
        return false;
    }

    public void flush() {
        store.clear();
    }

    private void invalidate(String cacheKey) {
        CacheEntry cacheEntry = store.remove(cacheKey);
        if (cacheEntry != null) {
            logger.debug("Invalidating " + beanName + " cache key " + cacheKey);
            invalidateIndexes(cacheKey, cacheEntry);
        } else {
            logger.debug("Tried to invalidate non-existent " + beanName + " cache key " + cacheKey);
        }
    }

    private void invalidateIndexes(String cacheKey, CacheEntry cacheEntry) {
        // Remove index entries pointing to this cache key
        @SuppressWarnings("unchecked") Set<Map.Entry<String, Serializable>> indexSet = cacheEntry.getIndexEntries().entrySet();
        for (Map.Entry<String, Serializable> indexEntry : indexSet) {
            Map<Serializable, Map<String, Void>> indexesForKey = indexes.get(indexEntry.getKey());
            if (indexesForKey != null) {
                indexesForKey.remove(indexEntry.getValue());
                logger.debug("Removed index " + indexEntry.getKey() + "=" + indexEntry.getValue() + " for cache key " + cacheKey);
            }
        }
    }

    public Integer getCacheMinutes() {
        return cacheMinutes;
    }

    public void setCacheMinutes(Integer cacheMinutes) {
        this.cacheMinutes = cacheMinutes;
    }

    public Integer getMaxObjects() {
        return maxObjects;
    }

    public void setMaxObjects(Integer maxObjects) {
        this.maxObjects = maxObjects;
    }

    public CacheKeyGenerator getKeyGenerator() {
        return keyGenerator;
    }

    public void setKeyGenerator(CacheKeyGenerator keyGenerator) {
        this.keyGenerator = keyGenerator;
    }

    public CacheInterceptor getCacheInterceptor() {
        return cacheInterceptor;
    }

    public void setCacheInterceptor(CacheInterceptor cacheInterceptor) {
        this.cacheInterceptor = cacheInterceptor;
    }

    public String getBeanName() {
        return beanName;
    }

    public void setBeanName(String beanName) {
        this.beanName = beanName;
    }

    private class CacheEntry implements Serializable {
        private String key;
        private Map<String/*indexKey*/, Serializable/*indexValue*/> indexEntries = new HashMap<>();
        private byte[] data;
        private Long touched;
        private Long expires;

        public String toString() {
            return key;
        }
        
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            CacheEntry that = (CacheEntry) o;

            if (!key.equals(that.key)) return false;

            return true;
        }

        public int hashCode() {
            return key.hashCode();
        }

        public String getKey() {
            return key;
        }

        public void setKey(String key) {
            this.key = key;
        }

        public Map<String, Serializable> getIndexEntries() {
            return indexEntries;
        }

        public void setIndexEntries(Map<String, Serializable> indexEntries) {
            this.indexEntries = indexEntries;
        }

        public void setCachedObject(Object cachedObject) throws IOException {
            touched = new Date().getTime();
            if (cachedObject == null) {
                data = null;
            } else {
                try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
                     ObjectOutputStream out = new ObjectOutputStream(baos)) {

                    out.writeObject(cachedObject);
                    data = baos.toByteArray();
                }
            }
        }

        public Object getCachedObject() throws IOException, ClassNotFoundException {
            touched = new Date().getTime();
            if (data == null)
                return null;

            try (ByteArrayInputStream bais = new ByteArrayInputStream(data);
                 ObjectInputStream in = new ObjectInputStream(bais)) {
                return in.readObject();
            }
        }

        public Long getTouched() {
            return touched;
        }

        public Long getExpires() {
            return expires;
        }

        public void setExpires(Long expires) {
            this.expires = expires;
        }
    }

    public void invalidateExpired() {
        List<String> expiredKeys = new ArrayList<>();
        Long now = new Date().getTime();

        for (CacheEntry entry : store.values()) {
            if ((now - entry.getTouched()) > (cacheMinutes * 60000))
                expiredKeys.add(entry.getKey());
            else if (entry.getExpires() != null && entry.expires > now)
                expiredKeys.add(entry.getKey());
        }

        for (String key : expiredKeys)
            invalidate(key);
    }

    private class IndexedLRUMap extends LRUMap {
        public IndexedLRUMap(Integer maxObjects) {
            super(maxObjects);
        }

        protected boolean removeLRU(LinkEntry entry) {
            invalidateIndexes((String)entry.getKey(), (CacheEntry)entry.getValue());
            return true;
        }
    }
}