package no.tornado.inject.module;

import no.tornado.inject.ApplicationContext;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.List;

import static no.tornado.inject.module.Module.STATE.*;

public class Module {
    public enum STATE {INSTALLED, STARTING, RUNNING, STOPPING}

    public enum CLASSLOADER_STATE {OK, DIRTY}

    public static final String MODULE_NAME = "Module-Name";
    public static final String MODULE_DESCRIPTION = "Module-Description";
    public static final String MODULE_CONTEXT = "Module-Context";
    public static final String MODULE_VERSION = "Module-Version";
    public static final String EXPORT_PACKAGE = "Export-Package";
    public static final String EXPORT_PACKAGE_DELIMITER = "\\s*,\\s*";

    private Integer id;
    private String name;
    private String description;
    private String contextClass;
    private Object _context;
    private String version;
    private List<String> exportPackages;
    private ModuleClassLoader _classLoader;
    private ModuleClassLoaderFactory classLoaderFactory;
    private STATE state = STATE.INSTALLED;
    private CLASSLOADER_STATE classloaderState = CLASSLOADER_STATE.OK;
    private File jarFile;
    private Date startTime;

    public Module() {
    }

    public Module(String name, String description, String contextClass, String version, List<String> exportPackages, ModuleClassLoaderFactory classLoaderFactory) {
        this.name = name;
        this.description = description;
        this.contextClass = contextClass;
        this.version = version;
        this.exportPackages = exportPackages;
        this.classLoaderFactory = classLoaderFactory;
    }

    public String toString() {
        return name + " (" + description + ")" + " version " + version;
    }

    public STATE getState() {
        return state;
    }

    public void setState(STATE state) {
        this.state = state;
    }

    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Module module = (Module) o;

        if (jarFile != null ? !jarFile.equals(module.jarFile) : module.jarFile != null) return false;
        if (!name.equals(module.name)) return false;
        if (!version.equals(module.version)) return false;

        return true;
    }

    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + version.hashCode();
        result = 31 * result + (jarFile != null ? jarFile.hashCode() : 0);
        return result;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public List<String> getExportPackages() {
        return exportPackages;
    }

    public void setExportPackages(List<String> exportPackages) {
        this.exportPackages = exportPackages;
    }

    public ModuleClassLoader getClassLoader() {
        if (_classLoader == null)
            _classLoader = classLoaderFactory.createClassLoader();
        return _classLoader;
    }

    public void stop() throws Exception {
        if (state == RUNNING || state == STARTING) {
            state = STOPPING;
            try {
                invokeOnApplicationContext("shutdown");
                invalidateClassLoader();
                ModuleSystem.signoff(this);
            } finally {
                state = INSTALLED;
                startTime = null;
            }
        }
    }

    public void invalidateClassLoader() {
        _classLoader = null;
        _context = null;
    }

    public boolean start() throws Exception {
        if (state != RUNNING && state != STARTING) {
            state = STOPPING;
            try {
                if (getContext() != null)
                    invokeOnApplicationContext("registerBeans", getContext());
                return true;
            } finally {
                state = RUNNING;
                classloaderState = CLASSLOADER_STATE.OK;
                startTime = new Date();
            }
        }
        return false;
    }

    public String getUpTime() {
        if (startTime == null)
            return "DOWN";

        long uptime = (new Date().getTime() - startTime.getTime()) / 1000;
        int days = (int) uptime / (60 * 60 * 24);
        int minutes, hours;

        String retval = "";
        if (days != 0) {
            retval += days + " " + ((days > 1) ? "days" : "day") + ", ";
        }

        minutes = (int) uptime / 60;
        hours = minutes / 60;
        hours %= 24;
        minutes %= 60;

        if (hours != 0) {
            retval += hours + ":" + minutes;
        } else if (minutes != 0) {
            retval += minutes + " min";
        } else {
            retval += uptime + " sec";
        }

        return retval;
    }

    public static String formatUptime(double uptime) {
        String retval = "";

        int days = (int) uptime / (60 * 60 * 24);
        int minutes, hours;

        if (days != 0) {
            retval += days + " " + ((days > 1) ? "days" : "day") + ", ";
        }

        minutes = (int) uptime / 60;
        hours = minutes / 60;
        hours %= 24;
        minutes %= 60;

        if (hours != 0) {
            retval += hours + ":" + minutes;
        } else {
            retval += minutes + " min";
        }

        return retval;
    }


    private Object getContext() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
        if (_context != null)
            return _context;

        if (contextClass != null) {
            try {
                _context = getClassLoader().loadClass(contextClass).newInstance();
            } catch (ClassNotFoundException ex) {
                throw new RuntimeException("Module '" + getName() + "' defined context class " + contextClass + ", but it was not found in the module classpath");
            }
            return _context;
        }

        return null;
    }

    public Object invokeOnApplicationContext(String methodName, Object... args) throws Exception {
        try {
            Class appContextClass = getClassLoader().loadClass(ApplicationContext.class.getName());
            Method method;
            if (args == null || args.length == 0) {
                method = appContextClass.getDeclaredMethod(methodName);
            } else {
                Class[] parameterTypes = new Class[args.length];
                if (methodName.equals("registerBeans"))
                    parameterTypes[0] = Object.class;
                else {
                    for (int i = 0; i < args.length; i++) {
                        Object arg = args[i];
                        parameterTypes[i] = arg == null ? Object.class : arg.getClass();
                    }
                }
                method = appContextClass.getDeclaredMethod(methodName, parameterTypes);
            }
            if (method == null)
                throw new IllegalArgumentException("Tried to execute non-existent method on ApplicationContext.class: " + methodName + ". Check args/type!");
            return method.invoke(null, args);
        } catch (InvocationTargetException ex) {
            if (ex.getCause() instanceof Exception)
                throw (Exception) ex.getCause();
            ex.getCause().printStackTrace();
            return null;
        } catch (ClassNotFoundException | NoSuchMethodException | IllegalAccessException ignored) {
            ignored.printStackTrace();
            return null;
        }
    }

    public CLASSLOADER_STATE getClassloaderState() {
        return classloaderState;
    }

    public void setClassloaderState(CLASSLOADER_STATE classloaderState) {
        this.classloaderState = classloaderState;
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(String version) {
        this.version = version;
    }

    public ModuleClassLoaderFactory getClassLoaderFactory() {
        return classLoaderFactory;
    }

    public void setClassLoaderFactory(ModuleClassLoaderFactory classLoaderFactory) {
        this.classLoaderFactory = classLoaderFactory;
    }

    public String getContextClass() {
        return contextClass;
    }

    public void setContextClass(String contextClass) {
        this.contextClass = contextClass;
    }

    public Integer getId() {
        return id;
    }

    public void setId(Integer id) {
        this.id = id;
    }

    public File getJarFile() {
        return jarFile;
    }

    public void setJarFile(File jarFile) {
        this.jarFile = jarFile;
    }

}
