package no.tornado.inject.module;

import no.tornado.inject.Inject;
import no.tornado.inject.webconsole.WebConsole;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathFactory;
import java.io.File;
import java.io.FileFilter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import static no.tornado.inject.module.ServiceAvailabilityEvent.EVENT_TYPE.SERVICE_ADDED;
import static no.tornado.inject.module.ServiceAvailabilityEvent.EVENT_TYPE.SERVICE_REMOVED;

public class ModuleSystem {
    private static Log logger = LogFactory.getLog(ModuleSystem.class);
    private static Map<Integer, Module> modules = new HashMap<>();
    private static Map<String, Integer> moduleNameMap = new HashMap<>();
    private static Map<String, List<Integer>> exportedPackages = new HashMap<>();
    private static Map<Integer, List<Integer>> exportedClassesUsage = new HashMap<>(); // Key = Exporter, Value = List of modules using classes from this package
    private static ModuleSystem INSTANCE;
    private static Map<String, List<ServiceRouter>> serviceProviders = new HashMap<>();
    private static File moduleFolder;
    private static Module.STATE defaultState = Module.STATE.RUNNING;
    private static Integer moduleSequence = 0;
    private static ClassLoader rootClassLoader = ClassLoader.getSystemClassLoader();
    private static boolean disabled = false;

    private ModuleSystem() {
    }

    public static List<String> getExportedPackages() {
        Set<String> strings = exportedPackages.keySet();
        return Arrays.asList(strings.toArray(new String[strings.size()]));
    }

    public static void main(String[] args) throws IOException {
        if (args.length > 0) {
            setModuleFolder(new File(args[0]));
        } else {
            setModuleFolder(new File("modules"));
        }
        // Start WebConsole if username/password is supplied
        if (args.length > 2)
            WebConsole.create().basicAuth(args[1], args[2]).start();

        init();
        System.in.read();
    }

    public static void init() {
        if (!isActive()) {
            INSTANCE = new ModuleSystem();
            logger.info("Tornado Inject Module System is active");
            if (moduleFolder != null)
                installModules();
        }
    }

    /**
     * Install all modules that are not already installed from module folder
     */
    public static void installModules() {
        if (moduleFolder.isFile()) {
            logger.error("Module folder " + moduleFolder + " is pointing to a file, , autoloading of modules is disabled.");
            return;
        }

        if (!moduleFolder.exists() && !moduleFolder.mkdirs()) {
            logger.error("Unable to create module folder, autoloading of modules is disabled.");
            return;
        }

        logger.info("Installing not currently installed modules from " + moduleFolder.getAbsolutePath() + "...");

        if (moduleFolder.exists()) {
            for (File moduleJar : moduleFolder.listFiles(new JarFilter())) {
                if (isAlreadyRegistered(moduleJar)) {
                    logger.debug("Module jar " + moduleJar + " already registered, skipping installation.");
                    continue;
                }
                try {
                    logger.info("Installing module " + moduleJar);
                    install(moduleJar.getAbsolutePath());
                } catch (Exception e) {
                    logger.error("Failed to install module " + moduleJar + ": " + e.getMessage());
                }
            }

            for (File mavenModuleCandidate : moduleFolder.listFiles(new FolderFilter())) {
                File pom = new File(mavenModuleCandidate, "pom.xml");
                if (pom.exists()) {
                    try {
                        Module module = createFromPom(pom);
                        if (isAlreadyRegistered(module)) {
                            logger.debug("Maven Module " + module + " already registered, skipping installation.");
                            continue;
                        }
                        try {
                            logger.info("Installing maven module " + module);
                            install(module);
                        } catch (Exception e) {
                            logger.error("Failed to install maven module " + module + ": " + e.getMessage());
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

            if (defaultState == Module.STATE.RUNNING) {
                for (Module module : modules.values()) {
                    try {
                        logger.info("Starting module " + module.toString());
                        module.start();
                    } catch (Exception e) {
                        logger.error("Failed to start module " + module + ": " + e.getMessage());
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    public static boolean isAlreadyRegistered(File moduleJar) {
        try {
            for (Module module : modules.values())
                if (module.getJarFile() != null && moduleJar.getCanonicalPath().equals(module.getJarFile().getCanonicalPath()))
                    return true;
        } catch (Exception ex) {
            return false;
        }
        return false;
    }

    public static boolean isAlreadyRegistered(Module checkModule) {
        for (Module module : modules.values())
            if (module.equals(checkModule))
                return true;
        return false;
    }

    public static boolean start(Integer moduleId) throws Exception {
        return getModule(moduleId).start();
    }

    public static boolean start(String jarFile) throws Exception {
        return install(jarFile).start();
    }

    public static void stop(Integer moduleId) throws Exception {
        getModule(moduleId).stop();
    }

    public static void restart(Integer moduleId) throws Exception {
        Module module = getModule(moduleId);
        module.stop();
        module.start();
    }

    public static void uninstall(Integer moduleId) throws Exception {
        stop(moduleId);
        removeExportedPackages(moduleId);
        Module module = modules.remove(moduleId);
        if (module != null)
            moduleNameMap.remove(module.getName());
    }

    public static Collection<Module> list() {
        return modules.values();
    }

    public static Module install(Module module) {
        init();
        Integer moduleId = getNextModuleId();
        modules.put(moduleId, module);
        module.setId(moduleId);
        moduleNameMap.put(module.getName(), moduleId);
        updateExportedPackages(moduleId);
        return module;
    }

    private static Integer getNextModuleId() {
        return ++moduleSequence;
    }

    public static Module install(String jarFile) throws Exception {
        File file = new File(jarFile);
        if (file.getName().equals("pom.xml"))
            return install(createFromPom(file));

        JarFile jar = new JarFile(file);
        Manifest manifest = jar.getManifest();
        Attributes attr = manifest.getMainAttributes();
        Module module = new Module();
        module.setJarFile(file);
        module.setName(attr.getValue(Module.MODULE_NAME));
        if (module.getName() == null)
            module.setName(file.getName());

        module.setDescription(attr.getValue(Module.MODULE_DESCRIPTION));
        module.setVersion(attr.getValue(Module.MODULE_VERSION));
        if (module.getVersion() == null)
            module.setVersion("SNAPSHOT");

        List<URL> classPathEntries = new ArrayList<>();
        classPathEntries.add(file.toURI().toURL());
        addTornadoInject(classPathEntries);
        module.setClassLoaderFactory(new ModuleClassLoaderFactory(module, classPathEntries));

        String contextClass = attr.getValue(Module.MODULE_CONTEXT);
        if (contextClass != null)
            module.setContextClass(contextClass);

        String exportPackage = attr.getValue(Module.EXPORT_PACKAGE);
        if (exportPackage != null) {
            List<String> exportPackages = new ArrayList<>();
            exportPackages.addAll(Arrays.asList(exportPackage.split(Module.EXPORT_PACKAGE_DELIMITER)));
            module.setExportPackages(exportPackages);
        }
        return install(module);
    }

    public static Module createFromPom(File pom) throws Exception {
        XPath x = XPathFactory.newInstance().newXPath();
        DocumentBuilder b = DocumentBuilderFactory.newInstance().newDocumentBuilder();

        Module module = new Module();
        module.setName(pom.getParentFile().getName());
        Document doc = b.parse(pom);

        String exportPackage = x.evaluate("//manifestEntries/Export-Package", doc).trim();
        if (exportPackage != null) {
            module.setExportPackages(new ArrayList<String>());
            module.getExportPackages().addAll(Arrays.asList(exportPackage.split(Module.EXPORT_PACKAGE_DELIMITER)));
        }

        String moduleContext = x.evaluate("//manifestEntries/Module-Context", doc);
        if (!"".equals(moduleContext))
            module.setContextClass(moduleContext);

        module.setVersion(x.evaluate("//project/version", doc));
        if (module.getVersion() == null)
            module.setVersion("SNAPSHOT");

        module.setDescription("Auto-created from " + pom.getAbsolutePath());
        List<URL> classPathEntries = new ArrayList<>();
        classPathEntries.add(new File(pom.getParentFile(), "target/classes").toURI().toURL());
        addTornadoInject(classPathEntries);
        module.setClassLoaderFactory(new ModuleClassLoaderFactory(module, classPathEntries));
        return module;
    }

    /**
     * Add Tornado Inject to this module's classpath
     *
     * @param classPathEntries The classpath for this module's classes and resources
     */
    private static void addTornadoInject(List<URL> classPathEntries) {
        String injectPackagePath = Inject.class.getPackage().getName().replace(".", "/");

        ClassLoader cl = getRootClassLoader();

        if (cl == null)
            throw new RuntimeException("Neither configured rootClassLoader or System Classloader can be retrieved.");

        URL injectPackageUrl = cl.getResource(injectPackagePath);
        if (injectPackageUrl == null)
            throw new RuntimeException("Could not find Tornado Inject in System Classloader or configured rootClassLoader!");

        try {
            classPathEntries.add(new URL(injectPackageUrl.toString().replaceAll(injectPackagePath + "$", "")));
        } catch (MalformedURLException ex) {
            throw new RuntimeException("Unable to create URL from Tornado Inject classpath entries!");
        }

    }

    private static void updateExportedPackages(Integer moduleId) {
        Module module = modules.get(moduleId);
        if (module.getExportPackages() != null) {
            for (String packageName : module.getExportPackages()) {
                if ("".equals(packageName.replaceAll(" ", "")))
                    continue;

                List<Integer> modulesProvidingPackage = exportedPackages.get(packageName);
                if (modulesProvidingPackage == null) {
                    modulesProvidingPackage = new ArrayList<>();
                    exportedPackages.put(packageName, modulesProvidingPackage);
                }
                modulesProvidingPackage.add(moduleId);
            }
        }
    }

    private static void removeExportedPackages(Integer moduleId) {
        Module module = modules.get(moduleId);
        if (module.getExportPackages() != null) {
            for (String packageName : module.getExportPackages()) {
                List<Integer> modulesProvidingPackage = exportedPackages.get(packageName);
                if (modulesProvidingPackage != null)
                    modulesProvidingPackage.remove(moduleId);
            }
        }
    }

    /**
     * Load exported class from the first module that exported it, according to module load order.
     * @param caller The calling Module
     * @param name class name
     * @throws java.lang.ClassNotFoundException if class was not found
     * @return The class if found
     */
    public static Class<?> loadExportedClass(Module caller, String name) throws ClassNotFoundException {
        if (!name.contains("."))
            throw new ClassNotFoundException("Unable to load exported class without package name: " + name);

        String packageName = name.substring(0, name.lastIndexOf("."));
        List<Integer> modulesProvidingPackage = exportedPackages.get(packageName);
        if (modulesProvidingPackage != null && !modulesProvidingPackage.isEmpty()) {
            Integer providingModuleId = modulesProvidingPackage.get(0);
            Class<?> clazz = modules.get(providingModuleId).getClassLoader().loadExportedClass(name);
            recordExportedClassUsage(caller, providingModuleId);
            return clazz;
        }
        throw new ClassNotFoundException("Unable to load class " + name + ", it was not exported from any modules.");
    }

    private static void recordExportedClassUsage(Module caller, Integer providingModuleId) {
        List<Integer> clients = exportedClassesUsage.get(providingModuleId);
        if (clients == null) {
            clients = new ArrayList<>();
            exportedClassesUsage.put(providingModuleId, clients);
        }

        if (caller != null && !clients.contains(caller.getId()))
            clients.add(caller.getId());
    }

    public static Module getModule(Integer moduleId) {
        return modules.get(moduleId);
    }

    public static Module getModule(String name) {
        Integer moduleId = moduleNameMap.get(name);
        return moduleId != null ? modules.get(moduleId) : null;
    }

    public static boolean isDisabled() {
        return disabled;
    }

    public static void disable() {
        disabled = true;
    }

    public static void disable(boolean disable) {
        disabled = disable;
    }

    public static boolean isActive() {
        return INSTANCE != null;
    }

    public static void addServiceRouter(ServiceRouter router) {
        logger.debug("Adding ServiceRouter " + router);
        List<ServiceRouter> servicesForClass = serviceProviders.get(router.getServiceClass());
        if (servicesForClass == null) {
            servicesForClass = new ArrayList<>();
            serviceProviders.put(router.getServiceClass(), servicesForClass);
        }
        servicesForClass.add(router);
        MessageBus.publish(new ServiceAvailabilityEvent(router, SERVICE_ADDED));
    }

    public static void shutdown() throws Exception {
        logger.debug("Shutting down module system");
        for (Module module : ModuleSystem.list().toArray(new Module[0]))
            ModuleSystem.uninstall(module.getId());

        INSTANCE = null;
    }

    public static void removeServiceRouter(String serviceName, int serviceHash) {
        List<ServiceRouter> routers = serviceProviders.get(serviceName);
        if (routers != null) {
            ListIterator<ServiceRouter> li = routers.listIterator();
            while (li.hasNext()) {
                ServiceRouter router = li.next();
                if (router.getServiceHash().equals(serviceHash)) {
                    logger.debug("Unregistering ServiceRouter " + router);
                    li.remove();
                    MessageBus.publish(new ServiceAvailabilityEvent(router, SERVICE_REMOVED));
                }
            }
        }
    }

    public static List<ServiceRouter> getServiceRouters(String serviceClass, String options) {
        List<ServiceRouter> routersForClass = serviceProviders.get(serviceClass);
        List<ServiceRouter> matchingOptions = new ArrayList<>();
        if (routersForClass == null)
            return Collections.emptyList();

        for (ServiceRouter router : routersForClass)
            if (router.getOptions().equals(options))
                matchingOptions.add(router);
        return matchingOptions;
    }

    public static void setDefaultState(Module.STATE defaultState) {
        ModuleSystem.defaultState = defaultState;
    }

    public static File getModuleFolder() {
        return moduleFolder;
    }

    public static void signoff(Module module) {
        List<Integer> clientModules = exportedClassesUsage.get(module.getId());
        if (clientModules != null) {
            for (Integer moduleId : clientModules) {
                Module client = getModule(moduleId);
                if (client != null) {
                    client.setClassloaderState(Module.CLASSLOADER_STATE.DIRTY);
                }
            }
        }
        exportedClassesUsage.remove(module.getId());
    }

    public static void setModuleFolder(File moduleFolder) {
        ModuleSystem.moduleFolder = moduleFolder;
    }

    public static void setRootClassLoader(ClassLoader rootClassLoader) {
        ModuleSystem.rootClassLoader = rootClassLoader;
    }

    public static ClassLoader getRootClassLoader() {
        return rootClassLoader;
    }

    public static class FolderFilter implements FileFilter {
        public boolean accept(File pathname) {
            return pathname.isDirectory();
        }
    }

    public static void runWhenAvailable(Class serviceClass, final Runnable work) {
        runWhenAvailable(serviceClass.getName(), Provides.NO_OPTIONS, work);
    }

    public static void runWhenAvailable(final String serviceClass, final String options, final Runnable work) {
        if (options == null)
            throw new IllegalArgumentException("Options must be not null, or Provides.NO_OPTIONS");

        List<ServiceRouter> routers = getServiceRouters(serviceClass, options);
        if (!routers.isEmpty()) {
            new Thread(work).start();
        } else {
            final InjectEventListener listener = new InjectEventListener() {
                public void onEvent(InjectEvent event) {
                    if (event instanceof ServiceAvailabilityEvent) {
                        ServiceAvailabilityEvent e = (ServiceAvailabilityEvent) event;
                        if (e.getEventType().equals(SERVICE_ADDED) && e.getRouter().getServiceClass().equals(serviceClass) && e.getRouter().getOptions().equals(options)) {
                            new Thread(work).start();
                            MessageBus.unsubscribe(this);
                        }
                    }
                }
            };
            MessageBus.subscribe(listener);
        }
    }

    private static class JarFilter implements FilenameFilter {
        public boolean accept(File dir, String name) {
            return name.toLowerCase().endsWith(".jar");
        }
    }

}
