package no.tornado.inject.module;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.UndeclaredThrowableException;
import java.util.List;

import static no.tornado.inject.module.ServiceAvailabilityEvent.EVENT_TYPE.SERVICE_ADDED;
import static no.tornado.inject.module.ServiceAvailabilityEvent.EVENT_TYPE.SERVICE_REMOVED;

public class ServiceProxy implements InvocationHandler, InjectEventListener {
    private static Log logger = LogFactory.getLog(ServiceProxy.class);

    private final String serviceClass;
    private final int timeout;
    private ServiceRouter router;
    private String options;

    public ServiceProxy(String serviceClass, int timeout, String options) {
        this.serviceClass = serviceClass;
        this.timeout = timeout;
        this.options = options;
        List<ServiceRouter> routers = ModuleSystem.getServiceRouters(serviceClass, options);
        this.router = routers.isEmpty() ? null : routers.get(routers.size() - 1);
    }

    public String toString() {
        return "ServiceProxy " + serviceClass;
    }

    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        if (router == null)
            waitOrDie();

        try {
            Object bean = router.getBean();
            Method beanMethod = bean.getClass().getMethod(method.getName(), method.getParameterTypes());
            return beanMethod.invoke(bean, args);
        } catch (UndeclaredThrowableException ex) {
            throw ex.getCause();
        } catch (InvocationTargetException ite) {
            if (ite.getTargetException() != null) {
                if (ite.getTargetException() instanceof UndeclaredThrowableException)
                    throw ite.getTargetException().getCause();

                throw ite.getTargetException();
            }
            else
                throw ite;
        }
    }

    private void waitOrDie() {
        int i = 0;
        while (router == null && i < timeout) {
            i++;
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        if (router == null)
            throw new ServiceNotAvailableException(serviceClass);
    }

    public void onEvent(InjectEvent event) {
        if (event instanceof ServiceAvailabilityEvent) {
            ServiceAvailabilityEvent sea = (ServiceAvailabilityEvent) event;
            if (SERVICE_REMOVED == sea.getEventType() && router.equals(sea.getRouter())) {
                logger.debug("Removing " + router + " from ServiceProxy " + this);
                router = null;
            }

            if (SERVICE_ADDED == sea.getEventType() && sea.getRouter().getServiceClass().equals(serviceClass) && sea.getRouter().getOptions().equals(options)) {
                logger.debug("Adding " + sea.getRouter() + " to ServiceProxy " + this + ", previous router was " + router);
                router = sea.getRouter();
            }
        }
    }
}
