package no.tornado.inject.servlet;

import no.tornado.inject.module.ModuleSystem;

import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.annotation.WebListener;
import java.io.File;
import java.io.IOException;

/**
 * <p>When running in a Servlet 3.0 container, this listener will be automatically
 * registered, and start the module system when the webapp starts.</p>
 * <p>You can supply the module folder with an init parameter called injectModuleFolder
 * to overwrite the default lookup mechanism. By default the module folder will be resolved like this:</p>
 * <ul>
 * <li>Look for a parent pom.xml as in a multimodule maven project</li>
 * <li>Look for WEB-INF/modules</li>
 * </ul>
 * <p>If you have already started the module system via some other means, this listener will do nothing.
 * You can disable the module system by calling ModuleSystem.disable() to prevent this listener from
 * starting the module system.</p>
 */
@WebListener
public class ModuleSystemServletListener implements ServletContextListener {
    public void contextInitialized(ServletContextEvent sce) {
        if (ModuleSystem.isDisabled() || "true".equals(sce.getServletContext().getInitParameter("disable-tornado-modulesystem")) || ModuleSystem.isActive())
            return;

        File moduleFolder = null;

        String configuredModuleFolder = sce.getServletContext().getInitParameter("injectModuleFolder");
        if (configuredModuleFolder != null)
            moduleFolder = new File(configuredModuleFolder);
        else {
            String devPomPath = sce.getServletContext().getRealPath("../../../pom.xml");
            if (devPomPath != null) {
                File developmentPom = new File(devPomPath);
                if (developmentPom.exists()) {
                    try {
                        moduleFolder = developmentPom.getParentFile().getCanonicalFile();
                    } catch (IOException e) {
                        moduleFolder = developmentPom.getParentFile();
                    }
                } else {
                    moduleFolder = new File(sce.getServletContext().getRealPath("WEB-INF/modules"));
                }
            }
        }

        if (moduleFolder != null && moduleFolder.exists()) {

            ModuleSystem.setRootClassLoader(getClass().getClassLoader());
            ModuleSystem.setModuleFolder(moduleFolder);
            ModuleSystem.init();
        }
    }

    public void contextDestroyed(ServletContextEvent sce) {
        if (ModuleSystem.isActive()) {
            try {
                ModuleSystem.shutdown();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

}
