package no.tornado.inject.webconsole;

import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;
import no.tornado.inject.IOTools;
import no.tornado.inject.module.Module;
import no.tornado.inject.module.ModuleSystem;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.*;
import java.net.InetSocketAddress;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@SuppressWarnings({"SynchronizeOnNonFinalField", "UnusedDeclaration"})
public class WebConsole implements HttpHandler {
    private static Log logger = LogFactory.getLog(WebConsole.class);
    private Integer port = 9000;
    private Properties mimeTypes = new Properties();
    private List<String> messages = Collections.synchronizedList(new ArrayList<String>());
    private String basicAuthString;

    public WebConsole basicAuth(String username, String password) {
        try {
            basicAuthString = "Basic " + Base64.getEncoder().encodeToString((username + ":" + password).getBytes("UTF-8"));
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Aj, aj, aj. No UTF-8 support!");
        }
        return this;
    }

    public static WebConsole create() {
        WebConsole webConsole = new WebConsole();
        try {
            webConsole.mimeTypes.load(WebConsole.class.getResourceAsStream("MimeTypes.conf"));
        } catch (IOException ignored) {
        }
        return webConsole;
    }

    public WebConsole start() throws IOException {
        InetSocketAddress addr = new InetSocketAddress(port);
        HttpServer server = HttpServer.create(addr, 0);
        server.createContext("/", this);
        server.setExecutor(Executors.newCachedThreadPool());
        server.start();
        logger.info("WebConsole available at http://localhost:" + port);
        return this;
    }

    public WebConsole port(Integer port) {
        this.port = port;
        return this;
    }

    public void handle(HttpExchange http) throws IOException {
        try {
            String uri = http.getRequestURI().toString();
            if ("/".equals(uri) || uri.startsWith("/?")) {
                handleAction(http);
            } else {
                handleFile(http);
            }
        } catch (Exception ex) {
            ex.printStackTrace();
            http.sendResponseHeaders(500, -1);
            http.getResponseBody().write(("Failed to process request: " + ex.getMessage()).getBytes());
        }
        http.getResponseBody().close();
    }

    private void handleFile(HttpExchange http) throws IOException {
        String filename = http.getRequestURI().toString().replaceAll("^/", "");
        String ext = filename.replaceAll("^.*\\.", "");
        try (InputStream fileInput = getClass().getResourceAsStream(filename)) {
            if (fileInput != null) {
                // Poor man's caching
                if (http.getRequestHeaders().containsKey("If-Modified-Since")) {
                    http.sendResponseHeaders(304, -1);
                } else {
                    http.getResponseHeaders().set("Content-Type", mimeTypes.getProperty(ext, "application/octet-stream"));
                    http.getResponseHeaders().set("Expires", "Wed, 1 Jan 2020 00:00:00 GMT");
                    http.getResponseHeaders().set("Last-Modified", "Wed, 28 Mar 2012 22:04:10 GMT");
                    byte[] data = IOTools.read(fileInput);
                    http.sendResponseHeaders(200, data.length);
                    http.getResponseBody().write(data);
                }
            } else {
                http.getResponseHeaders().set("Content-Type", "text/html; charset=UTF-8");
                http.sendResponseHeaders(404, 0);
                http.getResponseBody().write("<h1>404 Not found</h1>".getBytes());
            }
        }
    }


    private void handleAction(HttpExchange http) throws IOException {
        checkAuth(http);
        http.getResponseHeaders().set("Cache-Control", "no-cache");
        http.getResponseHeaders().set("Pragma", "no-cache");
        http.getResponseHeaders().set("Expires", "-1");

        String[] requestParts = http.getRequestURI().toString().replaceAll("^/\\?", "").split("&");
        Map<String, String> args = new HashMap<>();
        for (String arg : requestParts) {
            String[] requestPair = arg.split("=", 2);
            if (requestPair.length == 2) {
                String key = requestPair[0];
                String value = URLDecoder.decode(requestPair[1], "UTF-8");
                args.put(key, value);
            }
        }

        if (args.containsKey("cmd") && args.containsKey("m")) {
            try {
                Object returned = invokeOnModuleSystem(args.get("cmd"), args.get("m"));
                if ("true".equals(args.get("start")) && returned instanceof Module)
                    ((Module) returned).start();
            } catch (Exception ex) {
                synchronized (messages) {
                    messages.add("<div class=\"alert alert-error\"><a class=\"close\" data-dismiss=\"alert\" href=\"#\">&times;</a><span class=\"label label-warning\">Warning</span> Unable to invoke command " + args.get("cmd") + " on module " + args.get("m") + ".</div>");
                }
            }
            synchronized (messages) {
                messages.add("<div class=\"alert alert-success\"><a class=\"close\" data-dismiss=\"alert\" href=\"#\">&times;</a><strong>" + args.get("cmd").toUpperCase() + " executed</strong> on module " + args.get("m") + ".</div>");
            }
            http.getResponseHeaders().set("Location", "/");
            http.sendResponseHeaders(302, -1);
            return;
        }

        if (args.containsKey("changeModuleFolder")) {
            File newModuleFolder = new File(args.get("changeModuleFolder"));
            if (newModuleFolder.isDirectory()) {
                ModuleSystem.setModuleFolder(newModuleFolder);
                messages.add("<div class=\"alert alert-success\"><a class=\"close\" data-dismiss=\"alert\" href=\"#\">&times;</a><strong>Module folder updated.</div>");
            } else {
                messages.add("<div class=\"alert alert-error\"><a class=\"close\" data-dismiss=\"alert\" href=\"#\">&times;</a><span class=\"label label-warning\">Warning</span> " + args.get("changeModuleFolder") + " is not a valid folder.</div>");
            }
        }

        http.getResponseHeaders().set("Content-Type", "text/html; charset=UTF-8");
        InputStream fileInput = getClass().getResourceAsStream("index.html");
        String template = new String(IOTools.read(fileInput), "UTF-8");

        Matcher m = Pattern.compile("<tr class=\"moduleRow\">.*?</tr>", Pattern.DOTALL).matcher(template);
        if (!m.find())
            throw new RuntimeException("moduleRow invalid syntax in html template");

        boolean dirtyClassloaders = false;
        String rowTemplate = m.group(0);
        StringBuilder rows = new StringBuilder();
        for (Module module : ModuleSystem.list()) {
            rows.append(rowTemplate
                    .replace("#{id}", module.getId().toString())
                    .replace("#{name}", module.getName())
                    .replace("#{filename}", module.getJarFile() != null ? module.getJarFile().getName() : "N/A")
                    .replace("#{absoluteFilename}", module.getJarFile() != null ? module.getJarFile().getAbsolutePath().replace("\\", "\\\\") : "N/A")
                    .replace("#{version}", module.getVersion() != null ? module.getVersion() : "Unknown")
                    .replace("#{state}", module.getState().toString())
                    .replace("#{uptime}", module.getUpTime())
                    .replace("#{classloaderState}", module.getClassloaderState().toString())
            );

            if (module.getClassloaderState() == Module.CLASSLOADER_STATE.DIRTY)
                dirtyClassloaders = true;
        }
        template = m.replaceAll(rows.toString());

        template = template .replace("#{moduleTableTitle}", "Installed modules")
                            .replace("#{moduleFolderTableTitle}", "Available modules in module folder")
                            .replace("#{moduleSystemPath}", ModuleSystem.getModuleFolder() != null ? ModuleSystem.getModuleFolder().getAbsolutePath() : "");

        StringBuilder content = new StringBuilder();

        synchronized (messages) {
            ListIterator<String> li = messages.listIterator();
            while (li.hasNext()) {
                content.append(li.next());
                li.remove();
            }
        }

        if (dirtyClassloaders)
            content.append("<div class=\"alert alert-error\"><a class=\"close\" data-dismiss=\"alert\" href=\"#\">&times;</a><strong>Dirty classloaders!</strong> You have one or more modules using classes from modules that are either uninstalled or restarted. Please restart the modules to make sure you're using the latest available classes!</div>");

        template = template.replace("#{content}", content.toString());

        List<File> uninstalledModules = getUninstalledModules();
        if (uninstalledModules.isEmpty()) {
            m = Pattern.compile("<div class=\"container moduleFolder\">.*?</div>", Pattern.DOTALL).matcher(template);
            template = m.replaceAll("");
        } else {
            m = Pattern.compile("<tr class=\"moduleFolderRow\">.*?</tr>", Pattern.DOTALL).matcher(template);
            if (!m.find())
                throw new RuntimeException("moduleFolderRow invalid syntax in html template");

            rowTemplate = m.group(0);
            rows = new StringBuilder();
            for (File file : uninstalledModules) {
                rows.append(rowTemplate
                        .replace("#{filename}", file.getName().equals("pom.xml") ? file.getParentFile().getName() : file.getName())
                        .replace("#{absoluteFilename}", file.getAbsolutePath().replace("\\", "\\\\"))
                        .replace("#{absoluteFilenameEncoded}", URLEncoder.encode(file.getAbsolutePath(), "UTF-8"))
                );
            }
            template = m.replaceAll(rows.toString());
        }

        byte[] data = template.getBytes("UTF-8");
        http.sendResponseHeaders(200, data.length);
        http.getResponseBody().write(data);
    }

    private void checkAuth(HttpExchange http) throws IOException {
        if (basicAuthString != null) {
            if (!basicAuthString.equals(http.getRequestHeaders().getFirst("Authorization"))) {
                http.getResponseHeaders().set("WWW-Authenticate", "Basic realm=\"Tornado Inject WebConsole\"");
                http.sendResponseHeaders(401, 0);
                http.getResponseBody().write("Please log in".getBytes("UTF-8"));
            }
        }
    }

    private List<File> getUninstalledModules() {
        List<File> uninstalled = new ArrayList<>();
        if (ModuleSystem.getModuleFolder() != null && ModuleSystem.getModuleFolder().exists()) {
            for (File moduleJar : ModuleSystem.getModuleFolder().listFiles(new JarFilter())) {
                if (!ModuleSystem.isAlreadyRegistered(moduleJar))
                    uninstalled.add(moduleJar);
            }
            for (File mavenModuleCandidate : ModuleSystem.getModuleFolder().listFiles(new ModuleSystem.FolderFilter())) {
                File pom = new File(mavenModuleCandidate, "pom.xml");
                if (pom.exists()) {
                    try {
                        Module module = ModuleSystem.createFromPom(pom);
                        if (!ModuleSystem.isAlreadyRegistered(module))
                            uninstalled.add(pom);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return uninstalled;
    }

    private Object invokeOnModuleSystem(String cmd, String module) throws Exception {
        Integer moduleId = null;
        String moduleFile = null;
        try {
            moduleId = Integer.valueOf(module);
        } catch (NumberFormatException ex) {
            moduleFile = module;
        }
        switch (cmd) {
            case "start":
                if (moduleId != null)
                    return ModuleSystem.start(moduleId);
                else {
                    return ModuleSystem.install(moduleFile).start();
                }
            case "install":
                return ModuleSystem.install(moduleFile);
            case "stop":
                ModuleSystem.stop(moduleId);
                return null;
            case "restart":
                ModuleSystem.restart(moduleId);
                return null;
            case "uninstall":
                ModuleSystem.uninstall(moduleId);
                return null;
            default:
                return null;
        }
    }

    private static class JarFilter implements FilenameFilter {
        public boolean accept(File dir, String name) {
            return name.toLowerCase().endsWith(".jar");
        }
    }

}
