package no.tornado.inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>The ApplicationContext houses all your beans. You load it by supplying any object
 * you want to the ApplicationContext#registerBeans() method.</p>
 *
 * <p>All methods in the beanProvider object is concidered beans, and will be instantiated
 * lazily unless you annotate then with @Eager. You don't need to implement an interface,
 * and the methods in the beanProvider can have any visibility you want. The method name
 * is the bean name.</p>
 *
 * <p>Your beans can be automatically injected with other beans from your ApplicationContext
 * if you annotate the fields with the @Inject annotation. Circular dependencies are allowed.
 * You can supply the bean-argument to the @Inject annotation if you want to inject a spesific
 * bean based on name. If you don't, the best matching object based on type and name of the field
 * will be injected.</p>
 *
 * <p>You can even inject beans that are not part of your ApplicationContext via the
 * ApplicationContext.inject(object) method. Any @Inject annotated field will be injected.</p>
 *
 * <p>For more information, see these annotations:</p>
 *
 * @see Eager
 * @see Inject
 * @see Intercept
 * @see Destroy
 *
 */
public class ApplicationContext {
    private static Logger logger = LoggerFactory.getLogger(ApplicationContext.class);
    private static Map<String, BeanInfo> beansByName = new HashMap<>();
    private static Map<Class, List<BeanInfo>> beansByClass = new HashMap<>();

    static {
        Runtime.getRuntime().addShutdownHook(new AppContextShutdownHook());
    }

    /**
     * Register beans into your ApplicationContext. The beanProvider can be any object,
     * and all methods in the object will be conciders bean definitions. The methodName will
     * be the beanName in the context.
     *
     * @param beanProvider An object that provides beans for your ApplicationContext.
     *
     * <p>For more information:</p>
     *
     * @see ApplicationContext
     */
    public static void registerBeans(Object beanProvider) {
        Long before = new Date().getTime();
        // Create LazyLoadingProxy for all beans
        for (Method beanCreationMethod : getMethodsRecursive(beanProvider.getClass())) {
            logger.info("Registering bean " + beanCreationMethod.getName());
            BeanInfo beanInfo = new BeanInfo(beanCreationMethod, beanProvider);
            beansByName.put(beanInfo.getBeanName(), beanInfo);
            List<BeanInfo> beansBySameClass = beansByClass.get(beanInfo.getBeanCreationMethod().getReturnType());
            if (beansBySameClass == null) {
                beansBySameClass = new ArrayList<>();
                beansByClass.put(beanInfo.getBeanCreationMethod().getReturnType(), beansBySameClass);
            }
            beansBySameClass.add(beanInfo);
        }

        // Pre-instantiate beans configured for eager loading
        for (BeanInfo beanInfo : beansByName.values())
            if (beanInfo.isEager())
                beanInfo.getBean();

        logger.info("Registered beans from " + beanProvider + " in " + ((new Date().getTime() - before) / 1000) + " seconds");
    }

    /**
     * Manually get a bean from the ApplicationContext via bean name.
     *
     * @param beanName The name of the bean you want
     * @return The bean, if found, null if not.
     */
    public static Object getBean(String beanName) {
        BeanInfo beanInfo = getBeanInfo(beanName);
        return beanInfo == null ? null : beanInfo.getBean();
    }

    /**
     * <p>Get a bean from the ApplicationContext via className.</p>
     *
     * <p>If more beans of same class exists, the first is returned.</p>
     *
     * @param className The class name of the bean you want
     * @return The first bean of the supplied className if found, null if not
     */
    @SuppressWarnings("unchecked")
    public static <T> T getBean(Class<T> className) {
        List<BeanInfo> thisClassInfos = getBeanInfos(className);
        return thisClassInfos.isEmpty() ? null : (T) getBeanInfos(className).get(0).getBean();
    }

    /**
     * Get the BeanInfo object, containing various information about a bean, including the bean.
     *
     * @param beanName The name of the bean you want the BeanInfo for
     * @return The BeanInfo for the bean name, null if not found
     */
    public static BeanInfo getBeanInfo(String beanName) {
        return beansByName.get(beanName);
    }

    /**
     * Get all beans of the supplied class
     *
     * @param className The class name of the beans you want
     * @return A list of all beans for this class, empty list if none found.
     */
    public static List<Object> getBeans(Class className) {
        List<Object> beans = new ArrayList<>();
        List<BeanInfo> beansByThisClass = beansByClass.get(className);
        if (beansByThisClass == null)
            return new ArrayList<>();

        for (BeanInfo beanInfo : beansByThisClass)
            beans.add(beanInfo.getBean());
        return beans;
    }

    /**
     * Get all BeanInfo objects for the supplied class
     *
     * @param className The class name of the BeanInfos you want
     * @return A list of BeanInfo objects matching the supplied class name
     */
    public static List<BeanInfo> getBeanInfos(Class className) {
        return beansByClass.get(className);
    }

    /**
     * Inject beans from the ApplicationContext into an arbitrary object that is not
     * part of the ApplicationContext. The object must still contain @Inject annotations.
     *
     * @param object The object you want to inject with members from the ApplicationContext
     */
    public static void inject(final Object object) {
        BeanInfo mockBeanInfo = new BeanInfo() {
            public String getBeanName() {
                return object.getClass().getName();
            }
        };
        injectMembers(mockBeanInfo, object);
    }

    static void postProcessBean(BeanInfo beanInfo, Object bean) {
        injectMembers(beanInfo, bean);
        if (bean instanceof InitializingBean)
            ((InitializingBean) bean).afterPropertiesSet();
    }

    private static void injectMembers(BeanInfo beanInfo, Object bean) {
        logger.debug("Injecting members into " + beanInfo.getBeanName());
        for (Field field : getFieldsRecursive(bean.getClass())) {
            Inject inject = field.getAnnotation(Inject.class);
            if (inject != null) {
                field.setAccessible(true);
                if (inject.bean().equals("")) {
                    logger.debug("Injecting " + beanInfo.getBeanName() + "." + field.getName() + " by className");
                    List<Object> beansByClassName = getBeans(field.getType());

                    if (beansByClassName.isEmpty())
                        throw new BeanCreationException("Unable to inject " + beanInfo.getBeanName() + "." + field.getName() + ": No bean with type " + field.getType().getName() + " found in ApplicationContext", beanInfo);

                    try {
                        field.set(bean, beansByClassName.get(0));
                    } catch (IllegalAccessException e) {
                        throw new BeanCreationException("Unable to inject " + beanInfo.getBeanName() + "." + field.getName(), beanInfo, e);
                    }
                } else {
                    // Inject by bean name
                    logger.debug("Injecting " + beanInfo.getBeanName() + "." + field.getName() + " by beanName");
                    Object beanToInject = getBean(inject.bean());
                    if (beanToInject == null)
                        throw new BeanCreationException("Unable to inject " + beanInfo.getBeanName() + "." + field.getName() + ": No bean with name " + inject.bean() + " found in ApplicationContext", beanInfo);

                    try {
                        field.set(bean, beanToInject);
                    } catch (IllegalAccessException e) {
                        throw new BeanCreationException("Unable to inject " + beanInfo.getBeanName() + "." + field.getName(), beanInfo, e);
                    }
                }
            }
        }
    }

    public static void shutdown() {
        new AppContextShutdownHook().run();
    }

    private static List<Field> getFieldsRecursive(Class beanClass) {
        List<Field> fields = new ArrayList<>();
        addFieldsRecursive(fields, beanClass);
        return fields;
    }

    private static void addFieldsRecursive(List<Field> fields, Class beanClass) {
        fields.addAll(Arrays.asList(beanClass.getDeclaredFields()));
        if (beanClass.getSuperclass() != null)
            addFieldsRecursive(fields, beanClass.getSuperclass());
    }

    private static List<Method> getMethodsRecursive(Class beanClass) {
        List<Method> methods = new ArrayList<>();
        addMethodsRecursive(methods, beanClass);
        return methods;
    }

    private static void addMethodsRecursive(List<Method> methods, Class beanClass) {
        methods.addAll(Arrays.asList(beanClass.getDeclaredMethods()));
        if (beanClass.getSuperclass() != null)
            addMethodsRecursive(methods, beanClass.getSuperclass());
    }

    private static List<Object> getBeanProviders() {
        List<Object> beanProviders = new ArrayList<>();
        for (BeanInfo info : beansByName.values()) {
            if (!beanProviders.contains(info.getProvider()))
                beanProviders.add(info.getProvider());
        }
        return beanProviders;
    }

    private static class AppContextShutdownHook extends Thread {
        public void run() {
            // Call destroy methods
            for (Object beanProvider : getBeanProviders()) {
                for (Method beanCreationMethod : beanProvider.getClass().getDeclaredMethods()) {
                    Destroy destroy = beanCreationMethod.getAnnotation(Destroy.class);
                    if (destroy != null) {
                        BeanInfo beanInfo = getBeanInfo(beanCreationMethod.getName());
                        Object bean = beanInfo.getBean();
                        if (Proxy.isProxyClass(bean.getClass())) {
                            BeanProxy lazyLoader = (BeanProxy) Proxy.getInvocationHandler(bean);
                            Object delegate = lazyLoader.getDelegate();
                            if (delegate != null) {
                                logger.info("Running destroy method " + delegate + " " + beanInfo.getBeanName() + "." + destroy.value() + "()");
                                try {
                                    delegate.getClass().getMethod(destroy.value()).invoke(delegate);
                                } catch (Exception ignored) {
                                }
                            }
                        } else {
                            try {
                                logger.info("Running destroy method " + bean + "." + destroy.value() + "()");
                                bean.getClass().getMethod(destroy.value()).invoke(bean);
                            } catch (Exception ignored) {
                            }
                        }
                    }
                }
            }

            // Empty context
            beansByName.clear();
            beansByClass.clear();
        }
    }
}