package one.edee.darwin.locker.internal;

import lombok.extern.apachecommons.CommonsLog;
import one.edee.darwin.exception.ProcessIsLockedException;
import one.edee.darwin.locker.LockRestorer;
import one.edee.darwin.locker.Locker;

import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.TimerTask;

/**
 * Timer task to renew lock for not finished processes. Task is used in {@link Locker} class to automatically prolong
 * acquired lock when {@link LockRestorer#isFinished()} returns false.
 *
 * @author Michal Kolesnac, FG Forrest a.s. (c) 2009
 */
@CommonsLog
public class CheckLockTimerTask extends TimerTask {
    /**
     * Max count of renewing a lock
     */
    private static final int MAX_RENEW_COUNT = 10;
    /**
     * Reference to locker
     */
    private final Locker locker;
    /**
     * Name of process we want to locked
     */
    private final String processName;
    /**
     * Unlock key generated by leasing process
     */
    private final String unlockKey;
    /**
     * Time to renew of process lock
     */
    private final long renewTime;
    /**
     * Renewing counter
     */
    private int renewCount;

    public CheckLockTimerTask(Locker locker, String processName, String unlockKey, long renewTime) {
        this.locker = locker;
        this.processName = processName;
        this.unlockKey = unlockKey;
        this.renewTime = renewTime;
    }

    @Override
    public void run() {
        if (log.isDebugEnabled()) {
            log.debug("CheckLockTimerTask check lock of process: " + processName);
        }
        // Check max count of renewing lock
        renewCount++;
        if (renewCount > MAX_RENEW_COUNT) {
            if (log.isDebugEnabled()) {
                log.debug("Count of renew gain max value. Lock is not renewed.");
            }
            cancel();
            return;
        }
        // Check process map
        final Map<String, LockRestorer> processMap = locker.getProcessMap();
        final LockRestorer restorer = processMap.get(processName + unlockKey);
        if (restorer == null) {
            if (log.isDebugEnabled()) {
                log.debug("Process '" + processName + "' not found in processMap. Process was probably finished.");
            }
            cancel();
            return;
        }
        // Check if process is finished
        try {
            if (restorer.isFinished()) {
                if (log.isDebugEnabled()) {
                    log.debug("Process is finish. Don't renew a lock.");
                }
                processMap.remove(processName + unlockKey);
                cancel();
                return;
            }
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Cannot get status(running/finished) of process: " + processName, e);
            }
            processMap.remove(processName + unlockKey);
            cancel();
            return;
        }
        // Renew lock
        final LocalDateTime until = LocalDateTime.now().plus(renewTime, ChronoUnit.MILLIS);
        try {
            if (log.isDebugEnabled()) {
                SimpleDateFormat fmt = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss");
                log.debug("Process " + processName + " is renewed until " + fmt.format(until) + "");
            }
            this.locker.renewLease(processName, unlockKey, until);
        } catch (ProcessIsLockedException e) {
            if (log.isErrorEnabled()) {
                log.error("Cannot renew lock for process: " + processName, e);
            }
        }
    }
}
