package aj.org.objectweb.asm;

final class FieldWriter extends FieldVisitor {

    private final SymbolTable symbolTable;

    private final int accessFlags;

    private final int nameIndex;

    private final int descriptorIndex;

    private int signatureIndex;

    private int constantValueIndex;

    private AnnotationWriter lastRuntimeVisibleAnnotation;

    private AnnotationWriter lastRuntimeInvisibleAnnotation;

    private AnnotationWriter lastRuntimeVisibleTypeAnnotation;

    private AnnotationWriter lastRuntimeInvisibleTypeAnnotation;

    private Attribute firstAttribute;

    FieldWriter(final SymbolTable symbolTable, final int access, final String name, final String descriptor, final String signature, final Object constantValue) {
        super(Opcodes.ASM9);
        this.symbolTable = symbolTable;
        this.accessFlags = access;
        this.nameIndex = symbolTable.addConstantUtf8(name);
        this.descriptorIndex = symbolTable.addConstantUtf8(descriptor);
        if (signature != null) {
            this.signatureIndex = symbolTable.addConstantUtf8(signature);
        }
        if (constantValue != null) {
            this.constantValueIndex = symbolTable.addConstant(constantValue).index;
        }
    }

    @Override
    public AnnotationVisitor visitAnnotation(final String descriptor, final boolean visible) {
        if (visible) {
            return lastRuntimeVisibleAnnotation = AnnotationWriter.create(symbolTable, descriptor, lastRuntimeVisibleAnnotation);
        } else {
            return lastRuntimeInvisibleAnnotation = AnnotationWriter.create(symbolTable, descriptor, lastRuntimeInvisibleAnnotation);
        }
    }

    @Override
    public AnnotationVisitor visitTypeAnnotation(final int typeRef, final TypePath typePath, final String descriptor, final boolean visible) {
        if (visible) {
            return lastRuntimeVisibleTypeAnnotation = AnnotationWriter.create(symbolTable, typeRef, typePath, descriptor, lastRuntimeVisibleTypeAnnotation);
        } else {
            return lastRuntimeInvisibleTypeAnnotation = AnnotationWriter.create(symbolTable, typeRef, typePath, descriptor, lastRuntimeInvisibleTypeAnnotation);
        }
    }

    @Override
    public void visitAttribute(final Attribute attribute) {
        attribute.nextAttribute = firstAttribute;
        firstAttribute = attribute;
    }

    @Override
    public void visitEnd() {
    }

    int computeFieldInfoSize() {
        int size = 8;
        if (constantValueIndex != 0) {
            symbolTable.addConstantUtf8(Constants.CONSTANT_VALUE);
            size += 8;
        }
        size += Attribute.computeAttributesSize(symbolTable, accessFlags, signatureIndex);
        size += AnnotationWriter.computeAnnotationsSize(lastRuntimeVisibleAnnotation, lastRuntimeInvisibleAnnotation, lastRuntimeVisibleTypeAnnotation, lastRuntimeInvisibleTypeAnnotation);
        if (firstAttribute != null) {
            size += firstAttribute.computeAttributesSize(symbolTable);
        }
        return size;
    }

    void putFieldInfo(final ByteVector output) {
        boolean useSyntheticAttribute = symbolTable.getMajorVersion() < Opcodes.V1_5;
        int mask = useSyntheticAttribute ? Opcodes.ACC_SYNTHETIC : 0;
        output.putShort(accessFlags & ~mask).putShort(nameIndex).putShort(descriptorIndex);
        int attributesCount = 0;
        if (constantValueIndex != 0) {
            ++attributesCount;
        }
        if ((accessFlags & Opcodes.ACC_SYNTHETIC) != 0 && useSyntheticAttribute) {
            ++attributesCount;
        }
        if (signatureIndex != 0) {
            ++attributesCount;
        }
        if ((accessFlags & Opcodes.ACC_DEPRECATED) != 0) {
            ++attributesCount;
        }
        if (lastRuntimeVisibleAnnotation != null) {
            ++attributesCount;
        }
        if (lastRuntimeInvisibleAnnotation != null) {
            ++attributesCount;
        }
        if (lastRuntimeVisibleTypeAnnotation != null) {
            ++attributesCount;
        }
        if (lastRuntimeInvisibleTypeAnnotation != null) {
            ++attributesCount;
        }
        if (firstAttribute != null) {
            attributesCount += firstAttribute.getAttributeCount();
        }
        output.putShort(attributesCount);
        if (constantValueIndex != 0) {
            output.putShort(symbolTable.addConstantUtf8(Constants.CONSTANT_VALUE)).putInt(2).putShort(constantValueIndex);
        }
        Attribute.putAttributes(symbolTable, accessFlags, signatureIndex, output);
        AnnotationWriter.putAnnotations(symbolTable, lastRuntimeVisibleAnnotation, lastRuntimeInvisibleAnnotation, lastRuntimeVisibleTypeAnnotation, lastRuntimeInvisibleTypeAnnotation, output);
        if (firstAttribute != null) {
            firstAttribute.putAttributes(symbolTable, output);
        }
    }

    final void collectAttributePrototypes(final Attribute.Set attributePrototypes) {
        attributePrototypes.addAttributes(firstAttribute);
    }
}
