package aj.org.objectweb.asm;

public class Label {

    static final int FLAG_DEBUG_ONLY = 1;

    static final int FLAG_JUMP_TARGET = 2;

    static final int FLAG_RESOLVED = 4;

    static final int FLAG_REACHABLE = 8;

    static final int FLAG_SUBROUTINE_CALLER = 16;

    static final int FLAG_SUBROUTINE_START = 32;

    static final int FLAG_SUBROUTINE_END = 64;

    static final int LINE_NUMBERS_CAPACITY_INCREMENT = 4;

    static final int FORWARD_REFERENCES_CAPACITY_INCREMENT = 6;

    static final int FORWARD_REFERENCE_TYPE_MASK = 0xF0000000;

    static final int FORWARD_REFERENCE_TYPE_SHORT = 0x10000000;

    static final int FORWARD_REFERENCE_TYPE_WIDE = 0x20000000;

    static final int FORWARD_REFERENCE_HANDLE_MASK = 0x0FFFFFFF;

    static final Label EMPTY_LIST = new Label();

    public Object info;

    short flags;

    private short lineNumber;

    private int[] otherLineNumbers;

    int bytecodeOffset;

    private int[] forwardReferences;

    short inputStackSize;

    short outputStackSize;

    short outputStackMax;

    short subroutineId;

    Frame frame;

    Label nextBasicBlock;

    Edge outgoingEdges;

    Label nextListElement;

    public Label() {
    }

    public int getOffset() {
        if ((flags & FLAG_RESOLVED) == 0) {
            throw new IllegalStateException("Label offset position has not been resolved yet");
        }
        return bytecodeOffset;
    }

    final Label getCanonicalInstance() {
        return frame == null ? this : frame.owner;
    }

    final void addLineNumber(final int lineNumber) {
        if (this.lineNumber == 0) {
            this.lineNumber = (short) lineNumber;
        } else {
            if (otherLineNumbers == null) {
                otherLineNumbers = new int[LINE_NUMBERS_CAPACITY_INCREMENT];
            }
            int otherLineNumberIndex = ++otherLineNumbers[0];
            if (otherLineNumberIndex >= otherLineNumbers.length) {
                int[] newLineNumbers = new int[otherLineNumbers.length + LINE_NUMBERS_CAPACITY_INCREMENT];
                System.arraycopy(otherLineNumbers, 0, newLineNumbers, 0, otherLineNumbers.length);
                otherLineNumbers = newLineNumbers;
            }
            otherLineNumbers[otherLineNumberIndex] = lineNumber;
        }
    }

    final void accept(final MethodVisitor methodVisitor, final boolean visitLineNumbers) {
        methodVisitor.visitLabel(this);
        if (visitLineNumbers && lineNumber != 0) {
            methodVisitor.visitLineNumber(lineNumber & 0xFFFF, this);
            if (otherLineNumbers != null) {
                for (int i = 1; i <= otherLineNumbers[0]; ++i) {
                    methodVisitor.visitLineNumber(otherLineNumbers[i], this);
                }
            }
        }
    }

    final void put(final ByteVector code, final int sourceInsnBytecodeOffset, final boolean wideReference) {
        if ((flags & FLAG_RESOLVED) == 0) {
            if (wideReference) {
                addForwardReference(sourceInsnBytecodeOffset, FORWARD_REFERENCE_TYPE_WIDE, code.length);
                code.putInt(-1);
            } else {
                addForwardReference(sourceInsnBytecodeOffset, FORWARD_REFERENCE_TYPE_SHORT, code.length);
                code.putShort(-1);
            }
        } else {
            if (wideReference) {
                code.putInt(bytecodeOffset - sourceInsnBytecodeOffset);
            } else {
                code.putShort(bytecodeOffset - sourceInsnBytecodeOffset);
            }
        }
    }

    private void addForwardReference(final int sourceInsnBytecodeOffset, final int referenceType, final int referenceHandle) {
        if (forwardReferences == null) {
            forwardReferences = new int[FORWARD_REFERENCES_CAPACITY_INCREMENT];
        }
        int lastElementIndex = forwardReferences[0];
        if (lastElementIndex + 2 >= forwardReferences.length) {
            int[] newValues = new int[forwardReferences.length + FORWARD_REFERENCES_CAPACITY_INCREMENT];
            System.arraycopy(forwardReferences, 0, newValues, 0, forwardReferences.length);
            forwardReferences = newValues;
        }
        forwardReferences[++lastElementIndex] = sourceInsnBytecodeOffset;
        forwardReferences[++lastElementIndex] = referenceType | referenceHandle;
        forwardReferences[0] = lastElementIndex;
    }

    final boolean resolve(final byte[] code, final int bytecodeOffset) {
        this.flags |= FLAG_RESOLVED;
        this.bytecodeOffset = bytecodeOffset;
        if (forwardReferences == null) {
            return false;
        }
        boolean hasAsmInstructions = false;
        for (int i = forwardReferences[0]; i > 0; i -= 2) {
            final int sourceInsnBytecodeOffset = forwardReferences[i - 1];
            final int reference = forwardReferences[i];
            final int relativeOffset = bytecodeOffset - sourceInsnBytecodeOffset;
            int handle = reference & FORWARD_REFERENCE_HANDLE_MASK;
            if ((reference & FORWARD_REFERENCE_TYPE_MASK) == FORWARD_REFERENCE_TYPE_SHORT) {
                if (relativeOffset < Short.MIN_VALUE || relativeOffset > Short.MAX_VALUE) {
                    int opcode = code[sourceInsnBytecodeOffset] & 0xFF;
                    if (opcode < Opcodes.IFNULL) {
                        code[sourceInsnBytecodeOffset] = (byte) (opcode + Constants.ASM_OPCODE_DELTA);
                    } else {
                        code[sourceInsnBytecodeOffset] = (byte) (opcode + Constants.ASM_IFNULL_OPCODE_DELTA);
                    }
                    hasAsmInstructions = true;
                }
                code[handle++] = (byte) (relativeOffset >>> 8);
                code[handle] = (byte) relativeOffset;
            } else {
                code[handle++] = (byte) (relativeOffset >>> 24);
                code[handle++] = (byte) (relativeOffset >>> 16);
                code[handle++] = (byte) (relativeOffset >>> 8);
                code[handle] = (byte) relativeOffset;
            }
        }
        return hasAsmInstructions;
    }

    final void markSubroutine(final short subroutineId) {
        Label listOfBlocksToProcess = this;
        listOfBlocksToProcess.nextListElement = EMPTY_LIST;
        while (listOfBlocksToProcess != EMPTY_LIST) {
            Label basicBlock = listOfBlocksToProcess;
            listOfBlocksToProcess = listOfBlocksToProcess.nextListElement;
            basicBlock.nextListElement = null;
            if (basicBlock.subroutineId == 0) {
                basicBlock.subroutineId = subroutineId;
                listOfBlocksToProcess = basicBlock.pushSuccessors(listOfBlocksToProcess);
            }
        }
    }

    final void addSubroutineRetSuccessors(final Label subroutineCaller) {
        Label listOfProcessedBlocks = EMPTY_LIST;
        Label listOfBlocksToProcess = this;
        listOfBlocksToProcess.nextListElement = EMPTY_LIST;
        while (listOfBlocksToProcess != EMPTY_LIST) {
            Label basicBlock = listOfBlocksToProcess;
            listOfBlocksToProcess = basicBlock.nextListElement;
            basicBlock.nextListElement = listOfProcessedBlocks;
            listOfProcessedBlocks = basicBlock;
            if ((basicBlock.flags & FLAG_SUBROUTINE_END) != 0 && basicBlock.subroutineId != subroutineCaller.subroutineId) {
                basicBlock.outgoingEdges = new Edge(basicBlock.outputStackSize, subroutineCaller.outgoingEdges.successor, basicBlock.outgoingEdges);
            }
            listOfBlocksToProcess = basicBlock.pushSuccessors(listOfBlocksToProcess);
        }
        while (listOfProcessedBlocks != EMPTY_LIST) {
            Label newListOfProcessedBlocks = listOfProcessedBlocks.nextListElement;
            listOfProcessedBlocks.nextListElement = null;
            listOfProcessedBlocks = newListOfProcessedBlocks;
        }
    }

    private Label pushSuccessors(final Label listOfLabelsToProcess) {
        Label newListOfLabelsToProcess = listOfLabelsToProcess;
        Edge outgoingEdge = outgoingEdges;
        while (outgoingEdge != null) {
            boolean isJsrTarget = (flags & Label.FLAG_SUBROUTINE_CALLER) != 0 && outgoingEdge == outgoingEdges.nextEdge;
            if (!isJsrTarget && outgoingEdge.successor.nextListElement == null) {
                outgoingEdge.successor.nextListElement = newListOfLabelsToProcess;
                newListOfLabelsToProcess = outgoingEdge.successor;
            }
            outgoingEdge = outgoingEdge.nextEdge;
        }
        return newListOfLabelsToProcess;
    }

    @Override
    public String toString() {
        return "L" + System.identityHashCode(this);
    }
}
