package aj.org.objectweb.asm;

final class RecordComponentWriter extends RecordComponentVisitor {

    private final SymbolTable symbolTable;

    private final int nameIndex;

    private final int descriptorIndex;

    private int signatureIndex;

    private AnnotationWriter lastRuntimeVisibleAnnotation;

    private AnnotationWriter lastRuntimeInvisibleAnnotation;

    private AnnotationWriter lastRuntimeVisibleTypeAnnotation;

    private AnnotationWriter lastRuntimeInvisibleTypeAnnotation;

    private Attribute firstAttribute;

    RecordComponentWriter(final SymbolTable symbolTable, final String name, final String descriptor, final String signature) {
        super(Opcodes.ASM9);
        this.symbolTable = symbolTable;
        this.nameIndex = symbolTable.addConstantUtf8(name);
        this.descriptorIndex = symbolTable.addConstantUtf8(descriptor);
        if (signature != null) {
            this.signatureIndex = symbolTable.addConstantUtf8(signature);
        }
    }

    @Override
    public AnnotationVisitor visitAnnotation(final String descriptor, final boolean visible) {
        if (visible) {
            return lastRuntimeVisibleAnnotation = AnnotationWriter.create(symbolTable, descriptor, lastRuntimeVisibleAnnotation);
        } else {
            return lastRuntimeInvisibleAnnotation = AnnotationWriter.create(symbolTable, descriptor, lastRuntimeInvisibleAnnotation);
        }
    }

    @Override
    public AnnotationVisitor visitTypeAnnotation(final int typeRef, final TypePath typePath, final String descriptor, final boolean visible) {
        if (visible) {
            return lastRuntimeVisibleTypeAnnotation = AnnotationWriter.create(symbolTable, typeRef, typePath, descriptor, lastRuntimeVisibleTypeAnnotation);
        } else {
            return lastRuntimeInvisibleTypeAnnotation = AnnotationWriter.create(symbolTable, typeRef, typePath, descriptor, lastRuntimeInvisibleTypeAnnotation);
        }
    }

    @Override
    public void visitAttribute(final Attribute attribute) {
        attribute.nextAttribute = firstAttribute;
        firstAttribute = attribute;
    }

    @Override
    public void visitEnd() {
    }

    int computeRecordComponentInfoSize() {
        int size = 6;
        size += Attribute.computeAttributesSize(symbolTable, 0, signatureIndex);
        size += AnnotationWriter.computeAnnotationsSize(lastRuntimeVisibleAnnotation, lastRuntimeInvisibleAnnotation, lastRuntimeVisibleTypeAnnotation, lastRuntimeInvisibleTypeAnnotation);
        if (firstAttribute != null) {
            size += firstAttribute.computeAttributesSize(symbolTable);
        }
        return size;
    }

    void putRecordComponentInfo(final ByteVector output) {
        output.putShort(nameIndex).putShort(descriptorIndex);
        int attributesCount = 0;
        if (signatureIndex != 0) {
            ++attributesCount;
        }
        if (lastRuntimeVisibleAnnotation != null) {
            ++attributesCount;
        }
        if (lastRuntimeInvisibleAnnotation != null) {
            ++attributesCount;
        }
        if (lastRuntimeVisibleTypeAnnotation != null) {
            ++attributesCount;
        }
        if (lastRuntimeInvisibleTypeAnnotation != null) {
            ++attributesCount;
        }
        if (firstAttribute != null) {
            attributesCount += firstAttribute.getAttributeCount();
        }
        output.putShort(attributesCount);
        Attribute.putAttributes(symbolTable, 0, signatureIndex, output);
        AnnotationWriter.putAnnotations(symbolTable, lastRuntimeVisibleAnnotation, lastRuntimeInvisibleAnnotation, lastRuntimeVisibleTypeAnnotation, lastRuntimeInvisibleTypeAnnotation, output);
        if (firstAttribute != null) {
            firstAttribute.putAttributes(symbolTable, output);
        }
    }

    final void collectAttributePrototypes(final Attribute.Set attributePrototypes) {
        attributePrototypes.addAttributes(firstAttribute);
    }
}
