package aj.org.objectweb.asm.signature;

public class SignatureReader {

    private final String signatureValue;

    public SignatureReader(final String signature) {
        this.signatureValue = signature;
    }

    public void accept(final SignatureVisitor signatureVistor) {
        String signature = this.signatureValue;
        int length = signature.length();
        int offset;
        char currentChar;
        if (signature.charAt(0) == '<') {
            offset = 2;
            do {
                int classBoundStartOffset = signature.indexOf(':', offset);
                signatureVistor.visitFormalTypeParameter(signature.substring(offset - 1, classBoundStartOffset));
                offset = classBoundStartOffset + 1;
                currentChar = signature.charAt(offset);
                if (currentChar == 'L' || currentChar == '[' || currentChar == 'T') {
                    offset = parseType(signature, offset, signatureVistor.visitClassBound());
                }
                while ((currentChar = signature.charAt(offset++)) == ':') {
                    offset = parseType(signature, offset, signatureVistor.visitInterfaceBound());
                }
            } while (currentChar != '>');
        } else {
            offset = 0;
        }
        if (signature.charAt(offset) == '(') {
            offset++;
            while (signature.charAt(offset) != ')') {
                offset = parseType(signature, offset, signatureVistor.visitParameterType());
            }
            offset = parseType(signature, offset + 1, signatureVistor.visitReturnType());
            while (offset < length) {
                offset = parseType(signature, offset + 1, signatureVistor.visitExceptionType());
            }
        } else {
            offset = parseType(signature, offset, signatureVistor.visitSuperclass());
            while (offset < length) {
                offset = parseType(signature, offset, signatureVistor.visitInterface());
            }
        }
    }

    public void acceptType(final SignatureVisitor signatureVisitor) {
        parseType(signatureValue, 0, signatureVisitor);
    }

    private static int parseType(final String signature, final int startOffset, final SignatureVisitor signatureVisitor) {
        int offset = startOffset;
        char currentChar = signature.charAt(offset++);
        switch (currentChar) {
            case 'Z':
            case 'C':
            case 'B':
            case 'S':
            case 'I':
            case 'F':
            case 'J':
            case 'D':
            case 'V':
                signatureVisitor.visitBaseType(currentChar);
                return offset;
            case '[':
                return parseType(signature, offset, signatureVisitor.visitArrayType());
            case 'T':
                int endOffset = signature.indexOf(';', offset);
                signatureVisitor.visitTypeVariable(signature.substring(offset, endOffset));
                return endOffset + 1;
            case 'L':
                int start = offset;
                boolean visited = false;
                boolean inner = false;
                while (true) {
                    currentChar = signature.charAt(offset++);
                    if (currentChar == '.' || currentChar == ';') {
                        if (!visited) {
                            String name = signature.substring(start, offset - 1);
                            if (inner) {
                                signatureVisitor.visitInnerClassType(name);
                            } else {
                                signatureVisitor.visitClassType(name);
                            }
                        }
                        if (currentChar == ';') {
                            signatureVisitor.visitEnd();
                            break;
                        }
                        start = offset;
                        visited = false;
                        inner = true;
                    } else if (currentChar == '<') {
                        String name = signature.substring(start, offset - 1);
                        if (inner) {
                            signatureVisitor.visitInnerClassType(name);
                        } else {
                            signatureVisitor.visitClassType(name);
                        }
                        visited = true;
                        while ((currentChar = signature.charAt(offset)) != '>') {
                            switch (currentChar) {
                                case '*':
                                    ++offset;
                                    signatureVisitor.visitTypeArgument();
                                    break;
                                case '+':
                                case '-':
                                    offset = parseType(signature, offset + 1, signatureVisitor.visitTypeArgument(currentChar));
                                    break;
                                default:
                                    offset = parseType(signature, offset, signatureVisitor.visitTypeArgument('='));
                                    break;
                            }
                        }
                    }
                }
                return offset;
            default:
                throw new IllegalArgumentException();
        }
    }
}
