package aj.org.objectweb.asm.signature;

import aj.org.objectweb.asm.Opcodes;

public class SignatureWriter extends SignatureVisitor {

    private final StringBuilder stringBuilder = new StringBuilder();

    private boolean hasFormals;

    private boolean hasParameters;

    private int argumentStack;

    public SignatureWriter() {
        super(Opcodes.ASM9);
    }

    @Override
    public void visitFormalTypeParameter(final String name) {
        if (!hasFormals) {
            hasFormals = true;
            stringBuilder.append('<');
        }
        stringBuilder.append(name);
        stringBuilder.append(':');
    }

    @Override
    public SignatureVisitor visitClassBound() {
        return this;
    }

    @Override
    public SignatureVisitor visitInterfaceBound() {
        stringBuilder.append(':');
        return this;
    }

    @Override
    public SignatureVisitor visitSuperclass() {
        endFormals();
        return this;
    }

    @Override
    public SignatureVisitor visitInterface() {
        return this;
    }

    @Override
    public SignatureVisitor visitParameterType() {
        endFormals();
        if (!hasParameters) {
            hasParameters = true;
            stringBuilder.append('(');
        }
        return this;
    }

    @Override
    public SignatureVisitor visitReturnType() {
        endFormals();
        if (!hasParameters) {
            stringBuilder.append('(');
        }
        stringBuilder.append(')');
        return this;
    }

    @Override
    public SignatureVisitor visitExceptionType() {
        stringBuilder.append('^');
        return this;
    }

    @Override
    public void visitBaseType(final char descriptor) {
        stringBuilder.append(descriptor);
    }

    @Override
    public void visitTypeVariable(final String name) {
        stringBuilder.append('T');
        stringBuilder.append(name);
        stringBuilder.append(';');
    }

    @Override
    public SignatureVisitor visitArrayType() {
        stringBuilder.append('[');
        return this;
    }

    @Override
    public void visitClassType(final String name) {
        stringBuilder.append('L');
        stringBuilder.append(name);
        argumentStack *= 2;
    }

    @Override
    public void visitInnerClassType(final String name) {
        endArguments();
        stringBuilder.append('.');
        stringBuilder.append(name);
        argumentStack *= 2;
    }

    @Override
    public void visitTypeArgument() {
        if (argumentStack % 2 == 0) {
            argumentStack |= 1;
            stringBuilder.append('<');
        }
        stringBuilder.append('*');
    }

    @Override
    public SignatureVisitor visitTypeArgument(final char wildcard) {
        if (argumentStack % 2 == 0) {
            argumentStack |= 1;
            stringBuilder.append('<');
        }
        if (wildcard != '=') {
            stringBuilder.append(wildcard);
        }
        return this;
    }

    @Override
    public void visitEnd() {
        endArguments();
        stringBuilder.append(';');
    }

    @Override
    public String toString() {
        return stringBuilder.toString();
    }

    private void endFormals() {
        if (hasFormals) {
            hasFormals = false;
            stringBuilder.append('>');
        }
    }

    private void endArguments() {
        if (argumentStack % 2 == 1) {
            stringBuilder.append('>');
        }
        argumentStack /= 2;
    }
}
