package org.aspectj.apache.bcel.classfile;

import org.aspectj.apache.bcel.Constants;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public final class BootstrapMethods extends Attribute {

    private boolean isInPackedState = false;

    private byte[] data;

    private int numBootstrapMethods;

    private BootstrapMethod[] bootstrapMethods;

    public BootstrapMethods(BootstrapMethods c) {
        this(c.getNameIndex(), c.getLength(), c.getBootstrapMethods(), c.getConstantPool());
    }

    public BootstrapMethods(int nameIndex, int length, BootstrapMethod[] lineNumberTable, ConstantPool constantPool) {
        super(Constants.ATTR_BOOTSTRAPMETHODS, nameIndex, length, constantPool);
        setBootstrapMethods(lineNumberTable);
        isInPackedState = false;
    }

    public final void setBootstrapMethods(BootstrapMethod[] bootstrapMethods) {
        this.data = null;
        this.isInPackedState = false;
        this.bootstrapMethods = bootstrapMethods;
        this.numBootstrapMethods = bootstrapMethods == null ? 0 : bootstrapMethods.length;
    }

    BootstrapMethods(int name_index, int length, DataInputStream file, ConstantPool constant_pool) throws IOException {
        this(name_index, length, (BootstrapMethod[]) null, constant_pool);
        data = new byte[length];
        file.readFully(data);
        isInPackedState = true;
    }

    public static class BootstrapMethod {

        private int bootstrapMethodRef;

        private int[] bootstrapArguments;

        BootstrapMethod(DataInputStream file) throws IOException {
            this(file.readUnsignedShort(), readBootstrapArguments(file));
        }

        private static int[] readBootstrapArguments(DataInputStream dis) throws IOException {
            int numBootstrapMethods = dis.readUnsignedShort();
            int[] bootstrapArguments = new int[numBootstrapMethods];
            for (int i = 0; i < numBootstrapMethods; i++) {
                bootstrapArguments[i] = dis.readUnsignedShort();
            }
            return bootstrapArguments;
        }

        public BootstrapMethod(int bootstrapMethodRef, int[] bootstrapArguments) {
            this.bootstrapMethodRef = bootstrapMethodRef;
            this.bootstrapArguments = bootstrapArguments;
        }

        public int getBootstrapMethodRef() {
            return bootstrapMethodRef;
        }

        public int[] getBootstrapArguments() {
            return bootstrapArguments;
        }

        public final void dump(DataOutputStream file) throws IOException {
            file.writeShort(bootstrapMethodRef);
            int len = bootstrapArguments.length;
            file.writeShort(len);
            for (int bootstrapArgument : bootstrapArguments) {
                file.writeShort(bootstrapArgument);
            }
        }

        public final int getLength() {
            return 2 + 2 + 2 * bootstrapArguments.length;
        }
    }

    private void unpack() {
        if (isInPackedState) {
            try {
                ByteArrayInputStream bs = new ByteArrayInputStream(data);
                DataInputStream dis = new DataInputStream(bs);
                numBootstrapMethods = dis.readUnsignedShort();
                bootstrapMethods = new BootstrapMethod[numBootstrapMethods];
                for (int i = 0; i < numBootstrapMethods; i++) {
                    bootstrapMethods[i] = new BootstrapMethod(dis);
                }
                dis.close();
                data = null;
            } catch (IOException e) {
                throw new RuntimeException("Unpacking of LineNumberTable attribute failed");
            }
            isInPackedState = false;
        }
    }

    @Override
    public void accept(ClassVisitor v) {
        unpack();
        v.visitBootstrapMethods(this);
    }

    @Override
    public final void dump(DataOutputStream file) throws IOException {
        super.dump(file);
        if (isInPackedState) {
            file.write(data);
        } else {
            int blen = bootstrapMethods.length;
            file.writeShort(blen);
            for (BootstrapMethod bootstrapMethod : bootstrapMethods) {
                bootstrapMethod.dump(file);
            }
        }
    }

    public final BootstrapMethod[] getBootstrapMethods() {
        unpack();
        return bootstrapMethods;
    }

    @Override
    public final String toString() {
        unpack();
        StringBuilder buf = new StringBuilder();
        StringBuilder line = new StringBuilder();
        for (int i = 0; i < numBootstrapMethods; i++) {
            BootstrapMethod bm = bootstrapMethods[i];
            line.append("BootstrapMethod[").append(i).append("]:");
            int ref = bm.getBootstrapMethodRef();
            ConstantMethodHandle mh = (ConstantMethodHandle) getConstantPool().getConstant(ref);
            line.append("#" + ref + ":");
            line.append(ConstantMethodHandle.kindToString(mh.getReferenceKind()));
            line.append(" ").append(getConstantPool().getConstant(mh.getReferenceIndex()));
            int[] args = bm.getBootstrapArguments();
            line.append(" argcount:").append(args == null ? 0 : args.length).append(" ");
            if (args != null) {
                for (int arg : args) {
                    line.append(arg).append("(").append(getConstantPool().getConstant(arg)).append(") ");
                }
            }
            if (i < numBootstrapMethods - 1) {
                line.append(", ");
            }
            if (line.length() > 72) {
                line.append('\n');
                buf.append(line);
                line.setLength(0);
            }
        }
        buf.append(line);
        return buf.toString();
    }

    public final int getNumBootstrapMethods() {
        unpack();
        return bootstrapMethods.length;
    }
}
