package org.aspectj.apache.bcel.classfile;

import org.aspectj.apache.bcel.Constants;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

public final class ClassParser {

    private DataInputStream file;

    private String filename;

    private int classnameIndex;

    private int superclassnameIndex;

    private int major, minor;

    private int accessflags;

    private int[] interfaceIndices;

    private ConstantPool cpool;

    private Field[] fields;

    private Method[] methods;

    private Attribute[] attributes;

    private static final int BUFSIZE = 8192;

    public ClassParser(InputStream file, String filename) {
        this.filename = filename;
        if (file instanceof DataInputStream)
            this.file = (DataInputStream) file;
        else
            this.file = new DataInputStream(new BufferedInputStream(file, BUFSIZE));
    }

    public ClassParser(ByteArrayInputStream baos, String filename) {
        this.filename = filename;
        this.file = new DataInputStream(baos);
    }

    public ClassParser(String file_name) throws IOException {
        this.filename = file_name;
        file = new DataInputStream(new BufferedInputStream(new FileInputStream(file_name), BUFSIZE));
    }

    public JavaClass parse() throws IOException, ClassFormatException {
        readID();
        readVersion();
        readConstantPool();
        readClassInfo();
        readInterfaces();
        readFields();
        readMethods();
        readAttributes();
        file.close();
        JavaClass jc = new JavaClass(classnameIndex, superclassnameIndex, filename, major, minor, accessflags, cpool, interfaceIndices, fields, methods, attributes);
        return jc;
    }

    private final void readAttributes() {
        attributes = AttributeUtils.readAttributes(file, cpool);
    }

    private final void readClassInfo() throws IOException {
        accessflags = file.readUnsignedShort();
        if ((accessflags & Constants.ACC_INTERFACE) != 0)
            accessflags |= Constants.ACC_ABSTRACT;
        classnameIndex = file.readUnsignedShort();
        superclassnameIndex = file.readUnsignedShort();
    }

    private final void readConstantPool() throws IOException {
        try {
            cpool = new ConstantPool(file);
        } catch (ClassFormatException cfe) {
            cfe.printStackTrace();
            if (filename != null) {
                String newmessage = "File: '" + filename + "': " + cfe.getMessage();
                throw new ClassFormatException(newmessage);
            }
            throw cfe;
        }
    }

    private final void readFields() throws IOException, ClassFormatException {
        int fieldCount = file.readUnsignedShort();
        if (fieldCount == 0) {
            fields = Field.NoFields;
        } else {
            fields = new Field[fieldCount];
            for (int i = 0; i < fieldCount; i++) fields[i] = new Field(file, cpool);
        }
    }

    private final void readID() throws IOException {
        int magic = 0xCAFEBABE;
        if (file.readInt() != magic)
            throw new ClassFormatException(filename + " is not a Java .class file");
    }

    private static final int[] NO_INTERFACES = new int[0];

    private final void readInterfaces() throws IOException {
        int interfacesCount = file.readUnsignedShort();
        if (interfacesCount == 0) {
            interfaceIndices = NO_INTERFACES;
        } else {
            interfaceIndices = new int[interfacesCount];
            for (int i = 0; i < interfacesCount; i++)
                interfaceIndices[i] = file.readUnsignedShort();
        }
    }

    private final void readMethods() throws IOException {
        int methodsCount = file.readUnsignedShort();
        if (methodsCount == 0) {
            methods = Method.NoMethods;
        } else {
            methods = new Method[methodsCount];
            for (int i = 0; i < methodsCount; i++) methods[i] = new Method(file, cpool);
        }
    }

    private final void readVersion() throws IOException {
        minor = file.readUnsignedShort();
        major = file.readUnsignedShort();
    }
}
