package org.aspectj.apache.bcel.classfile;

import org.aspectj.apache.bcel.Constants;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public final class Code extends Attribute {

    private int maxStack;

    private int maxLocals;

    private byte[] code;

    private CodeException[] exceptionTable;

    private Attribute[] attributes;

    private static final CodeException[] NO_EXCEPTIONS = new CodeException[]{};

    public Code(Code c) {
        this(c.getNameIndex(), c.getLength(), c.getMaxStack(), c.getMaxLocals(), c.getCode(), c.getExceptionTable(), c.getAttributes(), c.getConstantPool());
    }

    Code(int name_index, int length, DataInputStream file, ConstantPool constant_pool) throws IOException {
        this(name_index, length, file.readUnsignedShort(), file.readUnsignedShort(), (byte[]) null, (CodeException[]) null, (Attribute[]) null, constant_pool);
        int len = file.readInt();
        code = new byte[len];
        file.readFully(code);
        len = file.readUnsignedShort();
        if (len == 0) {
            exceptionTable = NO_EXCEPTIONS;
        } else {
            exceptionTable = new CodeException[len];
            for (int i = 0; i < len; i++) {
                exceptionTable[i] = new CodeException(file);
            }
        }
        attributes = AttributeUtils.readAttributes(file, constant_pool);
        this.length = length;
    }

    public Code(int name_index, int length, int max_stack, int max_locals, byte[] code, CodeException[] exception_table, Attribute[] attributes, ConstantPool constant_pool) {
        super(Constants.ATTR_CODE, name_index, length, constant_pool);
        this.maxStack = max_stack;
        this.maxLocals = max_locals;
        setCode(code);
        setExceptionTable(exception_table);
        setAttributes(attributes);
    }

    @Override
    public void accept(ClassVisitor v) {
        v.visitCode(this);
    }

    @Override
    public final void dump(DataOutputStream file) throws IOException {
        super.dump(file);
        file.writeShort(maxStack);
        file.writeShort(maxLocals);
        file.writeInt(code.length);
        file.write(code, 0, code.length);
        file.writeShort(exceptionTable.length);
        for (CodeException e : exceptionTable) {
            e.dump(file);
        }
        file.writeShort(attributes.length);
        for (Attribute attribute : attributes) {
            attribute.dump(file);
        }
    }

    public final Attribute[] getAttributes() {
        return attributes;
    }

    public LineNumberTable getLineNumberTable() {
        for (Attribute attribute : attributes) {
            if (attribute.tag == Constants.ATTR_LINE_NUMBER_TABLE) {
                return (LineNumberTable) attribute;
            }
        }
        return null;
    }

    public LocalVariableTable getLocalVariableTable() {
        for (Attribute attribute : attributes) {
            if (attribute.tag == Constants.ATTR_LOCAL_VARIABLE_TABLE) {
                return (LocalVariableTable) attribute;
            }
        }
        return null;
    }

    public final byte[] getCode() {
        return code;
    }

    public final CodeException[] getExceptionTable() {
        return exceptionTable;
    }

    public final int getMaxLocals() {
        return maxLocals;
    }

    public final int getMaxStack() {
        return maxStack;
    }

    private final int getInternalLength() {
        return 2 + 2 + 4 + (code == null ? 0 : code.length) + 2 + 8 * (exceptionTable == null ? 0 : exceptionTable.length) + 2;
    }

    private final int calculateLength() {
        int len = 0;
        if (attributes != null) {
            for (Attribute attribute : attributes) {
                len += attribute.length + 6;
            }
        }
        return len + getInternalLength();
    }

    public final void setAttributes(Attribute[] attributes) {
        this.attributes = attributes;
        length = calculateLength();
    }

    public final void setCode(byte[] code) {
        this.code = code;
    }

    public final void setExceptionTable(CodeException[] exception_table) {
        this.exceptionTable = exception_table;
    }

    public final void setMaxLocals(int max_locals) {
        this.maxLocals = max_locals;
    }

    public final void setMaxStack(int max_stack) {
        this.maxStack = max_stack;
    }

    public final String toString(boolean verbose) {
        StringBuilder buf;
        buf = new StringBuilder("Code(max_stack = " + maxStack + ", max_locals = " + maxLocals + ", code_length = " + code.length + ")\n" + Utility.codeToString(code, cpool, 0, -1, verbose));
        if (exceptionTable.length > 0) {
            buf.append("\nException handler(s) = \n" + "From\tTo\tHandler\tType\n");
            for (CodeException e : exceptionTable) {
                buf.append(e.toString(cpool, verbose) + "\n");
            }
        }
        if (attributes.length > 0) {
            buf.append("\nAttribute(s) = \n");
            for (Attribute attribute : attributes) {
                buf.append(attribute.toString() + "\n");
            }
        }
        return buf.toString();
    }

    @Override
    public final String toString() {
        return toString(true);
    }

    public String getCodeString() {
        StringBuilder codeString = new StringBuilder();
        codeString.append("Code(max_stack = ").append(maxStack);
        codeString.append(", max_locals = ").append(maxLocals);
        codeString.append(", code_length = ").append(code.length).append(")\n");
        codeString.append(Utility.codeToString(code, cpool, 0, -1, true));
        if (exceptionTable.length > 0) {
            codeString.append("\n").append("Exception entries =  ").append(exceptionTable.length).append("\n");
            for (CodeException exc : exceptionTable) {
                int type = exc.getCatchType();
                String name = "finally";
                if (type != 0) {
                    name = this.cpool.getConstantString(type, Constants.CONSTANT_Class);
                }
                codeString.append(name).append("[");
                codeString.append(exc.getStartPC()).append(">").append(exc.getEndPC()).append("]\n");
            }
        }
        return codeString.toString();
    }
}
