package org.aspectj.apache.bcel.classfile;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.classfile.annotation.AnnotationGen;
import org.aspectj.apache.bcel.classfile.annotation.RuntimeAnnos;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public abstract class FieldOrMethod extends Modifiers implements Node {

    protected int nameIndex;

    protected int signatureIndex;

    protected Attribute[] attributes;

    protected ConstantPool cpool;

    private String name;

    private String signature;

    private AnnotationGen[] annotations;

    private String signatureAttributeString = null;

    private boolean searchedForSignatureAttribute = false;

    protected FieldOrMethod() {
    }

    protected FieldOrMethod(FieldOrMethod c) {
        this(c.getModifiers(), c.getNameIndex(), c.getSignatureIndex(), c.getAttributes(), c.getConstantPool());
    }

    protected FieldOrMethod(DataInputStream file, ConstantPool cpool) throws IOException {
        this(file.readUnsignedShort(), file.readUnsignedShort(), file.readUnsignedShort(), null, cpool);
        attributes = AttributeUtils.readAttributes(file, cpool);
    }

    protected FieldOrMethod(int accessFlags, int nameIndex, int signatureIndex, Attribute[] attributes, ConstantPool cpool) {
        this.modifiers = accessFlags;
        this.nameIndex = nameIndex;
        this.signatureIndex = signatureIndex;
        this.cpool = cpool;
        this.attributes = attributes;
    }

    public void setAttributes(Attribute[] attributes) {
        this.attributes = attributes;
    }

    public final void dump(DataOutputStream file) throws IOException {
        file.writeShort(modifiers);
        file.writeShort(nameIndex);
        file.writeShort(signatureIndex);
        AttributeUtils.writeAttributes(attributes, file);
    }

    public final Attribute[] getAttributes() {
        return attributes;
    }

    public final ConstantPool getConstantPool() {
        return cpool;
    }

    public final int getNameIndex() {
        return nameIndex;
    }

    public final int getSignatureIndex() {
        return signatureIndex;
    }

    public final String getName() {
        if (name == null) {
            ConstantUtf8 c = (ConstantUtf8) cpool.getConstant(nameIndex, Constants.CONSTANT_Utf8);
            name = c.getValue();
        }
        return name;
    }

    public final String getSignature() {
        if (signature == null) {
            ConstantUtf8 c = (ConstantUtf8) cpool.getConstant(signatureIndex, Constants.CONSTANT_Utf8);
            signature = c.getValue();
        }
        return signature;
    }

    public final String getDeclaredSignature() {
        if (getGenericSignature() != null)
            return getGenericSignature();
        return getSignature();
    }

    public AnnotationGen[] getAnnotations() {
        if (annotations == null) {
            List<AnnotationGen> accumulatedAnnotations = new ArrayList<>();
            for (Attribute attribute : attributes) {
                if (attribute instanceof RuntimeAnnos) {
                    RuntimeAnnos runtimeAnnotations = (RuntimeAnnos) attribute;
                    accumulatedAnnotations.addAll(runtimeAnnotations.getAnnotations());
                }
            }
            if (accumulatedAnnotations.size() == 0) {
                annotations = AnnotationGen.NO_ANNOTATIONS;
            } else {
                annotations = accumulatedAnnotations.toArray(AnnotationGen.NO_ANNOTATIONS);
            }
        }
        return annotations;
    }

    public final String getGenericSignature() {
        if (!searchedForSignatureAttribute) {
            Signature sig = AttributeUtils.getSignatureAttribute(attributes);
            signatureAttributeString = (sig == null ? null : sig.getSignature());
            searchedForSignatureAttribute = true;
        }
        return signatureAttributeString;
    }
}
