package org.aspectj.apache.bcel.classfile;

import org.aspectj.apache.bcel.Constants;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public final class LineNumberTable extends Attribute {

    private boolean isInPackedState = false;

    private byte[] data;

    private int tableLength;

    private LineNumber[] table;

    public LineNumberTable(LineNumberTable c) {
        this(c.getNameIndex(), c.getLength(), c.getLineNumberTable(), c.getConstantPool());
    }

    public LineNumberTable(int nameIndex, int length, LineNumber[] lineNumberTable, ConstantPool constantPool) {
        super(Constants.ATTR_LINE_NUMBER_TABLE, nameIndex, length, constantPool);
        setLineNumberTable(lineNumberTable);
        isInPackedState = false;
    }

    LineNumberTable(int name_index, int length, DataInputStream file, ConstantPool constant_pool) throws IOException {
        this(name_index, length, (LineNumber[]) null, constant_pool);
        data = new byte[length];
        file.readFully(data);
        isInPackedState = true;
    }

    private void unpack() {
        if (isInPackedState) {
            try {
                ByteArrayInputStream bs = new ByteArrayInputStream(data);
                DataInputStream dis = new DataInputStream(bs);
                tableLength = (dis.readUnsignedShort());
                table = new LineNumber[tableLength];
                for (int i = 0; i < tableLength; i++) {
                    table[i] = new LineNumber(dis);
                }
                dis.close();
                data = null;
            } catch (IOException e) {
                throw new RuntimeException("Unpacking of LineNumberTable attribute failed");
            }
            isInPackedState = false;
        }
    }

    @Override
    public void accept(ClassVisitor v) {
        unpack();
        v.visitLineNumberTable(this);
    }

    @Override
    public final void dump(DataOutputStream file) throws IOException {
        super.dump(file);
        if (isInPackedState) {
            file.write(data);
        } else {
            file.writeShort(tableLength);
            for (int i = 0; i < tableLength; i++) {
                table[i].dump(file);
            }
        }
    }

    public final LineNumber[] getLineNumberTable() {
        unpack();
        return table;
    }

    public final void setLineNumberTable(LineNumber[] line_number_table) {
        this.data = null;
        this.isInPackedState = false;
        this.table = line_number_table;
        this.tableLength = (line_number_table == null) ? 0 : line_number_table.length;
    }

    @Override
    public final String toString() {
        unpack();
        StringBuilder buf = new StringBuilder();
        StringBuilder line = new StringBuilder();
        for (int i = 0; i < tableLength; i++) {
            line.append(table[i].toString());
            if (i < tableLength - 1) {
                line.append(", ");
            }
            if (line.length() > 72) {
                line.append('\n');
                buf.append(line);
                line.setLength(0);
            }
        }
        buf.append(line);
        return buf.toString();
    }

    public int getSourceLine(int pos) {
        unpack();
        int l = 0, r = tableLength - 1;
        if (r < 0)
            return -1;
        int min_index = -1, min = -1;
        do {
            int i = (l + r) / 2;
            int j = table[i].getStartPC();
            if (j == pos)
                return table[i].getLineNumber();
            else if (pos < j)
                r = i - 1;
            else
                l = i + 1;
            if (j < pos && j > min) {
                min = j;
                min_index = i;
            }
        } while (l <= r);
        if (min_index < 0)
            return -1;
        return table[min_index].getLineNumber();
    }

    public final int getTableLength() {
        unpack();
        return tableLength;
    }
}
