package org.aspectj.apache.bcel.classfile;

import org.aspectj.apache.bcel.Constants;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public class LocalVariableTable extends Attribute {

    private boolean isInPackedState = false;

    private byte[] data;

    private int localVariableTableLength;

    private LocalVariable[] localVariableTable;

    public LocalVariableTable(LocalVariableTable c) {
        this(c.getNameIndex(), c.getLength(), c.getLocalVariableTable(), c.getConstantPool());
    }

    public LocalVariableTable(int name_index, int length, LocalVariable[] local_variable_table, ConstantPool constant_pool) {
        super(Constants.ATTR_LOCAL_VARIABLE_TABLE, name_index, length, constant_pool);
        setLocalVariableTable(local_variable_table);
    }

    LocalVariableTable(int name_index, int length, DataInputStream file, ConstantPool constant_pool) throws IOException {
        super(Constants.ATTR_LOCAL_VARIABLE_TABLE, name_index, length, constant_pool);
        data = new byte[length];
        file.readFully(data);
        isInPackedState = true;
    }

    @Override
    public void accept(ClassVisitor v) {
        unpack();
        v.visitLocalVariableTable(this);
    }

    @Override
    public final synchronized void dump(DataOutputStream file) throws IOException {
        super.dump(file);
        if (isInPackedState) {
            file.write(data);
        } else {
            file.writeShort(localVariableTableLength);
            for (int i = 0; i < localVariableTableLength; i++) localVariableTable[i].dump(file);
        }
    }

    public final LocalVariable[] getLocalVariableTable() {
        unpack();
        return localVariableTable;
    }

    public final LocalVariable getLocalVariable(int index) {
        unpack();
        for (int i = 0; i < localVariableTableLength; i++) {
            if (localVariableTable[i] != null && localVariableTable[i].getIndex() == index) {
                return localVariableTable[i];
            }
        }
        return null;
    }

    public synchronized final void setLocalVariableTable(LocalVariable[] local_variable_table) {
        data = null;
        isInPackedState = false;
        this.localVariableTable = local_variable_table;
        localVariableTableLength = (local_variable_table == null) ? 0 : local_variable_table.length;
    }

    @Override
    public final String toString() {
        StringBuilder buf = new StringBuilder("");
        unpack();
        for (int i = 0; i < localVariableTableLength; i++) {
            buf.append(localVariableTable[i].toString());
            if (i < localVariableTableLength - 1)
                buf.append('\n');
        }
        return buf.toString();
    }

    public synchronized LocalVariableTable copyFromPackedState() {
        if (!isInPackedState)
            throw new IllegalStateException("No in packed state");
        try {
            return new LocalVariableTable(nameIndex, length, new DataInputStream(new ByteArrayInputStream(data)), getConstantPool());
        } catch (IOException e) {
            throw new RuntimeException("Failed to unpack clone", e);
        }
    }

    public final int getTableLength() {
        unpack();
        return localVariableTableLength;
    }

    private synchronized void unpack() {
        if (!isInPackedState)
            return;
        try {
            ByteArrayInputStream bs = new ByteArrayInputStream(data);
            DataInputStream dis = new DataInputStream(bs);
            localVariableTableLength = (dis.readUnsignedShort());
            localVariableTable = new LocalVariable[localVariableTableLength];
            for (int i = 0; i < localVariableTableLength; i++)
                localVariableTable[i] = new LocalVariable(dis, cpool);
            dis.close();
            data = null;
        } catch (IOException e) {
            throw new RuntimeException("Unpacking of LocalVariableTable attribute failed", e);
        }
        isInPackedState = false;
    }
}
