package org.aspectj.apache.bcel.classfile;

import org.aspectj.apache.bcel.Constants;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public final class NestMembers extends Attribute {

    private int numberOfClasses;

    private int[] classes;

    public NestMembers(NestMembers c) {
        this(c.getNameIndex(), c.getLength(), c.getClasses(), c.getConstantPool());
    }

    public NestMembers(int nameIndex, int length, int[] classes, ConstantPool cp) {
        super(Constants.ATTR_NEST_MEMBERS, nameIndex, length, cp);
        setClasses(classes);
    }

    NestMembers(int nameIndex, int length, DataInputStream file, ConstantPool constant_pool) throws IOException {
        this(nameIndex, length, (int[]) null, constant_pool);
        numberOfClasses = file.readUnsignedShort();
        classes = new int[numberOfClasses];
        for (int i = 0; i < numberOfClasses; i++) {
            classes[i] = file.readUnsignedShort();
        }
    }

    @Override
    public void accept(ClassVisitor v) {
        v.visitNestMembers(this);
    }

    @Override
    public final void dump(DataOutputStream file) throws IOException {
        super.dump(file);
        file.writeShort(numberOfClasses);
        for (int i = 0; i < numberOfClasses; i++) {
            file.writeShort(classes[i]);
        }
    }

    public final int[] getClasses() {
        return classes;
    }

    public final void setClasses(int[] inner_classes) {
        this.classes = inner_classes;
        numberOfClasses = (inner_classes == null) ? 0 : inner_classes.length;
    }

    public final String[] getClassesNames() {
        String[] result = new String[numberOfClasses];
        for (int i = 0; i < numberOfClasses; i++) {
            ConstantClass constantClass = (ConstantClass) cpool.getConstant(classes[i], Constants.CONSTANT_Class);
            result[i] = constantClass.getClassname(cpool);
        }
        return result;
    }

    @Override
    public final String toString() {
        StringBuilder buf = new StringBuilder();
        for (int i = 0; i < numberOfClasses; i++) {
            ConstantClass constantClass = (ConstantClass) cpool.getConstant(classes[i], Constants.CONSTANT_Class);
            buf.append(constantClass.getClassname(cpool)).append(" ");
        }
        return "NestMembers(" + buf.toString().trim() + ")";
    }
}
