package org.aspectj.apache.bcel.classfile;

import org.aspectj.apache.bcel.Constants;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public final class Signature extends Attribute {

    private int signature_index;

    public Signature(Signature c) {
        this(c.getNameIndex(), c.getLength(), c.getSignatureIndex(), c.getConstantPool());
    }

    Signature(int name_index, int length, DataInputStream file, ConstantPool constant_pool) throws IOException {
        this(name_index, length, file.readUnsignedShort(), constant_pool);
    }

    public Signature(int name_index, int length, int signature_index, ConstantPool constant_pool) {
        super(Constants.ATTR_SIGNATURE, name_index, length, constant_pool);
        this.signature_index = signature_index;
    }

    @Override
    public void accept(ClassVisitor v) {
        System.err.println("Visiting non-standard Signature object");
        v.visitSignature(this);
    }

    @Override
    public final void dump(DataOutputStream file) throws IOException {
        super.dump(file);
        file.writeShort(signature_index);
    }

    public final int getSignatureIndex() {
        return signature_index;
    }

    public final void setSignatureIndex(int signature_index) {
        this.signature_index = signature_index;
    }

    public final String getSignature() {
        ConstantUtf8 c = (ConstantUtf8) cpool.getConstant(signature_index, Constants.CONSTANT_Utf8);
        return c.getValue();
    }

    private static final class MyByteArrayInputStream extends ByteArrayInputStream {

        MyByteArrayInputStream(String data) {
            super(data.getBytes());
        }

        final int mark() {
            return pos;
        }

        final String getData() {
            return new String(buf);
        }

        final void reset(int p) {
            pos = p;
        }

        final void unread() {
            if (pos > 0)
                pos--;
        }
    }

    private static boolean identStart(int ch) {
        return ch == 'T' || ch == 'L';
    }

    private static final void matchIdent(MyByteArrayInputStream in, StringBuilder buf) {
        int ch;
        if ((ch = in.read()) == -1)
            throw new RuntimeException("Illegal signature: " + in.getData() + " no ident, reaching EOF");
        if (!identStart(ch)) {
            StringBuilder buf2 = new StringBuilder();
            int count = 1;
            while (Character.isJavaIdentifierPart((char) ch)) {
                buf2.append((char) ch);
                count++;
                ch = in.read();
            }
            if (ch == ':') {
                in.skip("Ljava/lang/Object".length());
                buf.append(buf2);
                ch = in.read();
                in.unread();
            } else {
                for (int i = 0; i < count; i++) in.unread();
            }
            return;
        }
        StringBuilder buf2 = new StringBuilder();
        ch = in.read();
        do {
            buf2.append((char) ch);
            ch = in.read();
        } while ((ch != -1) && (Character.isJavaIdentifierPart((char) ch) || (ch == '/')));
        buf.append(buf2.toString().replace('/', '.'));
        if (ch != -1)
            in.unread();
    }

    private static final void matchGJIdent(MyByteArrayInputStream in, StringBuilder buf) {
        int ch;
        matchIdent(in, buf);
        ch = in.read();
        if ((ch == '<') || ch == '(') {
            buf.append((char) ch);
            matchGJIdent(in, buf);
            while (((ch = in.read()) != '>') && (ch != ')')) {
                if (ch == -1)
                    throw new RuntimeException("Illegal signature: " + in.getData() + " reaching EOF");
                buf.append(", ");
                in.unread();
                matchGJIdent(in, buf);
            }
            buf.append((char) ch);
        } else
            in.unread();
        ch = in.read();
        if (identStart(ch)) {
            in.unread();
            matchGJIdent(in, buf);
        } else if (ch == ')') {
            in.unread();
            return;
        } else if (ch != ';')
            throw new RuntimeException("Illegal signature: " + in.getData() + " read " + (char) ch);
    }

    public static String translate(String s) {
        StringBuilder buf = new StringBuilder();
        matchGJIdent(new MyByteArrayInputStream(s), buf);
        return buf.toString();
    }

    public static final boolean isFormalParameterList(String s) {
        return s.startsWith("<") && (s.indexOf(':') > 0);
    }

    public static final boolean isActualParameterList(String s) {
        return s.startsWith("L") && s.endsWith(">;");
    }

    @Override
    public final String toString() {
        String s = getSignature();
        return "Signature(" + s + ")";
    }
}
