package org.aspectj.apache.bcel.classfile.annotation;

import org.aspectj.apache.bcel.classfile.ConstantPool;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public abstract class ElementValue {

    public static final int STRING = 's';

    public static final int ENUM_CONSTANT = 'e';

    public static final int CLASS = 'c';

    public static final int ANNOTATION = '@';

    public static final int ARRAY = '[';

    public static final int PRIMITIVE_INT = 'I';

    public static final int PRIMITIVE_BYTE = 'B';

    public static final int PRIMITIVE_CHAR = 'C';

    public static final int PRIMITIVE_DOUBLE = 'D';

    public static final int PRIMITIVE_FLOAT = 'F';

    public static final int PRIMITIVE_LONG = 'J';

    public static final int PRIMITIVE_SHORT = 'S';

    public static final int PRIMITIVE_BOOLEAN = 'Z';

    protected int type;

    protected ConstantPool cpool;

    protected ElementValue(int type, ConstantPool cpool) {
        this.type = type;
        this.cpool = cpool;
    }

    public int getElementValueType() {
        return type;
    }

    public abstract String stringifyValue();

    public abstract void dump(DataOutputStream dos) throws IOException;

    public static ElementValue readElementValue(DataInputStream dis, ConstantPool cpGen) throws IOException {
        int type = dis.readUnsignedByte();
        switch (type) {
            case 'B':
                return new SimpleElementValue(PRIMITIVE_BYTE, dis.readUnsignedShort(), cpGen);
            case 'C':
                return new SimpleElementValue(PRIMITIVE_CHAR, dis.readUnsignedShort(), cpGen);
            case 'D':
                return new SimpleElementValue(PRIMITIVE_DOUBLE, dis.readUnsignedShort(), cpGen);
            case 'F':
                return new SimpleElementValue(PRIMITIVE_FLOAT, dis.readUnsignedShort(), cpGen);
            case 'I':
                return new SimpleElementValue(PRIMITIVE_INT, dis.readUnsignedShort(), cpGen);
            case 'J':
                return new SimpleElementValue(PRIMITIVE_LONG, dis.readUnsignedShort(), cpGen);
            case 'S':
                return new SimpleElementValue(PRIMITIVE_SHORT, dis.readUnsignedShort(), cpGen);
            case 'Z':
                return new SimpleElementValue(PRIMITIVE_BOOLEAN, dis.readUnsignedShort(), cpGen);
            case 's':
                return new SimpleElementValue(STRING, dis.readUnsignedShort(), cpGen);
            case 'e':
                return new EnumElementValue(dis.readUnsignedShort(), dis.readUnsignedShort(), cpGen);
            case 'c':
                return new ClassElementValue(dis.readUnsignedShort(), cpGen);
            case '@':
                return new AnnotationElementValue(ANNOTATION, AnnotationGen.read(dis, cpGen, true), cpGen);
            case '[':
                int numArrayVals = dis.readUnsignedShort();
                ElementValue[] evalues = new ElementValue[numArrayVals];
                for (int j = 0; j < numArrayVals; j++) {
                    evalues[j] = ElementValue.readElementValue(dis, cpGen);
                }
                return new ArrayElementValue(ARRAY, evalues, cpGen);
            default:
                throw new RuntimeException("Unexpected element value kind in annotation: " + type);
        }
    }

    protected ConstantPool getConstantPool() {
        return cpool;
    }

    public static ElementValue copy(ElementValue value, ConstantPool cpool, boolean copyPoolEntries) {
        switch (value.getElementValueType()) {
            case 'B':
            case 'C':
            case 'D':
            case 'F':
            case 'I':
            case 'J':
            case 'S':
            case 'Z':
            case 's':
                return new SimpleElementValue((SimpleElementValue) value, cpool, copyPoolEntries);
            case 'e':
                return new EnumElementValue((EnumElementValue) value, cpool, copyPoolEntries);
            case '@':
                return new AnnotationElementValue((AnnotationElementValue) value, cpool, copyPoolEntries);
            case '[':
                return new ArrayElementValue((ArrayElementValue) value, cpool, copyPoolEntries);
            case 'c':
                return new ClassElementValue((ClassElementValue) value, cpool, copyPoolEntries);
            default:
                throw new RuntimeException("Not implemented yet! (" + value.getElementValueType() + ")");
        }
    }
}
