package org.aspectj.apache.bcel.classfile.annotation;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.classfile.ConstantPool;
import org.aspectj.apache.bcel.classfile.ConstantUtf8;
import org.aspectj.apache.bcel.generic.ObjectType;

import java.io.DataOutputStream;
import java.io.IOException;

public class EnumElementValue extends ElementValue {

    private int typeIdx;

    private int valueIdx;

    protected EnumElementValue(int typeIdx, int valueIdx, ConstantPool cpool) {
        super(ElementValue.ENUM_CONSTANT, cpool);
        if (type != ENUM_CONSTANT) {
            throw new RuntimeException("Only element values of type enum can be built with this ctor");
        }
        this.typeIdx = typeIdx;
        this.valueIdx = valueIdx;
    }

    public EnumElementValue(ObjectType t, String value, ConstantPool cpool) {
        super(ElementValue.ENUM_CONSTANT, cpool);
        typeIdx = cpool.addUtf8(t.getSignature());
        valueIdx = cpool.addUtf8(value);
    }

    public EnumElementValue(EnumElementValue value, ConstantPool cpool, boolean copyPoolEntries) {
        super(ENUM_CONSTANT, cpool);
        if (copyPoolEntries) {
            typeIdx = cpool.addUtf8(value.getEnumTypeString());
            valueIdx = cpool.addUtf8(value.getEnumValueString());
        } else {
            typeIdx = value.getTypeIndex();
            valueIdx = value.getValueIndex();
        }
    }

    @Override
    public void dump(DataOutputStream dos) throws IOException {
        dos.writeByte(type);
        dos.writeShort(typeIdx);
        dos.writeShort(valueIdx);
    }

    @Override
    public String stringifyValue() {
        StringBuilder sb = new StringBuilder();
        ConstantUtf8 cu8 = (ConstantUtf8) cpool.getConstant(typeIdx, Constants.CONSTANT_Utf8);
        sb.append(cu8.getValue());
        cu8 = (ConstantUtf8) cpool.getConstant(valueIdx, Constants.CONSTANT_Utf8);
        sb.append(cu8.getValue());
        return sb.toString();
    }

    public String toString() {
        StringBuilder s = new StringBuilder("E(");
        s.append(getEnumTypeString()).append(" ").append(getEnumValueString()).append(")");
        return s.toString();
    }

    public String getEnumTypeString() {
        return ((ConstantUtf8) getConstantPool().getConstant(typeIdx)).getValue();
    }

    public String getEnumValueString() {
        return ((ConstantUtf8) getConstantPool().getConstant(valueIdx)).getValue();
    }

    public int getValueIndex() {
        return valueIdx;
    }

    public int getTypeIndex() {
        return typeIdx;
    }
}
