package org.aspectj.apache.bcel.classfile.annotation;

import org.aspectj.apache.bcel.classfile.ConstantPool;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public class TypeAnnotationGen {

    public static final TypeAnnotationGen[] NO_TYPE_ANNOTATIONS = new TypeAnnotationGen[0];

    public static final int[] NO_TYPE_PATH = new int[0];

    private ConstantPool cpool;

    private TypeAnnotationGen(ConstantPool cpool) {
        this.cpool = cpool;
    }

    private int targetType;

    private int[] typePath;

    private AnnotationGen annotation;

    private int info;

    private int info2;

    private int[] localVarTarget;

    public final static int CLASS_TYPE_PARAMETER = 0x00;

    public final static int METHOD_TYPE_PARAMETER = 0x01;

    public final static int CLASS_EXTENDS = 0x10;

    public final static int CLASS_TYPE_PARAMETER_BOUND = 0x11;

    public final static int METHOD_TYPE_PARAMETER_BOUND = 0x12;

    public final static int FIELD = 0x13;

    public final static int METHOD_RETURN = 0x14;

    public final static int METHOD_RECEIVER = 0x15;

    public final static int METHOD_FORMAL_PARAMETER = 0x16;

    public final static int THROWS = 0x17;

    public final static int LOCAL_VARIABLE = 0x40;

    public final static int RESOURCE_VARIABLE = 0x41;

    public final static int EXCEPTION_PARAMETER = 0x42;

    public final static int INSTANCEOF = 0x43;

    public final static int NEW = 0x44;

    public final static int CONSTRUCTOR_REFERENCE = 0x45;

    public final static int METHOD_REFERENCE = 0x46;

    public final static int CAST = 0x47;

    public final static int CONSTRUCTOR_INVOCATION_TYPE_ARGUMENT = 0x48;

    public final static int METHOD_INVOCATION_TYPE_ARGUMENT = 0x49;

    public final static int CONSTRUCTOR_REFERENCE_TYPE_ARGUMENT = 0x4A;

    public final static int METHOD_REFERENCE_TYPE_ARGUMENT = 0x4B;

    public final static int TYPE_PATH_ENTRY_KIND_ARRAY = 0;

    public final static int TYPE_PATH_ENTRY_KIND_INNER_TYPE = 1;

    public final static int TYPE_PATH_ENTRY_KIND_WILDCARD = 2;

    public final static int TYPE_PATH_ENTRY_KIND_TYPE_ARGUMENT = 3;

    public static TypeAnnotationGen read(DataInputStream dis, ConstantPool cpool, boolean isVisible) throws IOException {
        TypeAnnotationGen typeAnno = new TypeAnnotationGen(cpool);
        typeAnno.targetType = dis.readUnsignedByte();
        switch (typeAnno.targetType) {
            case CLASS_TYPE_PARAMETER:
                typeAnno.info = dis.readUnsignedByte();
                break;
            case METHOD_TYPE_PARAMETER:
                typeAnno.info = dis.readUnsignedByte();
                break;
            case CLASS_EXTENDS:
                int superTypeIndex = dis.readUnsignedShort();
                if (superTypeIndex == 65535) {
                    typeAnno.info = -1;
                } else {
                    typeAnno.info = superTypeIndex;
                }
                break;
            case CLASS_TYPE_PARAMETER_BOUND:
            case METHOD_TYPE_PARAMETER_BOUND:
                typeAnno.info = dis.readUnsignedByte();
                typeAnno.info2 = dis.readUnsignedByte();
                break;
            case FIELD:
            case METHOD_RETURN:
            case METHOD_RECEIVER:
                break;
            case METHOD_FORMAL_PARAMETER:
                typeAnno.info = dis.readUnsignedByte();
                break;
            case THROWS:
                typeAnno.info = dis.readUnsignedShort();
                break;
            case LOCAL_VARIABLE:
            case RESOURCE_VARIABLE:
                typeAnno.localVarTarget = readLocalVarTarget(dis);
                break;
            case EXCEPTION_PARAMETER:
                typeAnno.info = dis.readUnsignedByte();
                break;
            case INSTANCEOF:
            case NEW:
            case CONSTRUCTOR_REFERENCE:
            case METHOD_REFERENCE:
                typeAnno.info = dis.readUnsignedShort();
                break;
            case CAST:
            case CONSTRUCTOR_INVOCATION_TYPE_ARGUMENT:
            case METHOD_INVOCATION_TYPE_ARGUMENT:
            case CONSTRUCTOR_REFERENCE_TYPE_ARGUMENT:
            case METHOD_REFERENCE_TYPE_ARGUMENT:
                typeAnno.info = dis.readUnsignedShort();
                typeAnno.info2 = dis.readUnsignedByte();
                break;
            default:
                throw new IllegalStateException("nyi " + typeAnno.targetType);
        }
        int typepathlength = dis.readUnsignedByte();
        if (typepathlength == 0) {
            typeAnno.typePath = NO_TYPE_PATH;
        } else {
            typeAnno.typePath = new int[typepathlength * 2];
            for (int i = 0, max = typepathlength * 2; i < max; i++) {
                typeAnno.typePath[i] = dis.readUnsignedByte();
            }
        }
        typeAnno.annotation = AnnotationGen.read(dis, cpool, isVisible);
        return typeAnno;
    }

    public static int[] readLocalVarTarget(DataInputStream dis) throws IOException {
        int tableLength = dis.readUnsignedShort();
        int[] table = new int[tableLength * 3];
        int count = 0;
        for (int i = 0; i < tableLength; i++) {
            table[count++] = dis.readUnsignedShort();
            table[count++] = dis.readUnsignedShort();
            table[count++] = dis.readUnsignedShort();
        }
        return table;
    }

    public void dump(DataOutputStream dos) throws IOException {
        dos.writeByte(targetType);
        switch (targetType) {
            case CLASS_TYPE_PARAMETER:
                dos.writeByte(this.info);
                break;
            case METHOD_TYPE_PARAMETER:
                dos.writeByte(info);
                break;
            case CLASS_EXTENDS:
                dos.writeShort(info);
                break;
            case CLASS_TYPE_PARAMETER_BOUND:
            case METHOD_TYPE_PARAMETER_BOUND:
                dos.writeByte(info);
                dos.writeByte(info2);
                break;
            case FIELD:
            case METHOD_RETURN:
            case METHOD_RECEIVER:
                break;
            case METHOD_FORMAL_PARAMETER:
                dos.writeByte(info);
                break;
            case THROWS:
                dos.writeShort(info);
                break;
            case LOCAL_VARIABLE:
            case RESOURCE_VARIABLE:
                dos.writeShort(localVarTarget.length / 3);
                for (int j : localVarTarget) {
                    dos.writeShort(j);
                }
                break;
            case EXCEPTION_PARAMETER:
                dos.writeByte(info);
                break;
            case INSTANCEOF:
            case NEW:
            case CONSTRUCTOR_REFERENCE:
            case METHOD_REFERENCE:
                dos.writeShort(info);
                break;
            case CAST:
            case CONSTRUCTOR_INVOCATION_TYPE_ARGUMENT:
            case METHOD_INVOCATION_TYPE_ARGUMENT:
            case CONSTRUCTOR_REFERENCE_TYPE_ARGUMENT:
            case METHOD_REFERENCE_TYPE_ARGUMENT:
                dos.writeShort(info);
                dos.writeByte(info);
                break;
            default:
                throw new IllegalStateException("nyi " + targetType);
        }
        dos.writeByte(typePath.length);
        for (int j : typePath) {
            dos.writeByte(j);
        }
        annotation.dump(dos);
    }

    public int getSupertypeIndex() {
        assert (targetType == CLASS_EXTENDS);
        return info;
    }

    public int getOffset() {
        assert (targetType == INSTANCEOF || targetType == NEW || targetType == CONSTRUCTOR_REFERENCE || targetType == METHOD_REFERENCE || targetType == CAST || targetType == CONSTRUCTOR_INVOCATION_TYPE_ARGUMENT || targetType == METHOD_INVOCATION_TYPE_ARGUMENT || targetType == CONSTRUCTOR_REFERENCE_TYPE_ARGUMENT || targetType == METHOD_REFERENCE_TYPE_ARGUMENT);
        return info;
    }

    public int getTypeParameterIndex() {
        assert (targetType == CLASS_TYPE_PARAMETER || targetType == METHOD_TYPE_PARAMETER || targetType == CLASS_TYPE_PARAMETER_BOUND || targetType == METHOD_TYPE_PARAMETER_BOUND);
        return info;
    }

    public int getTypeArgumentIndex() {
        assert (targetType == CAST || targetType == CONSTRUCTOR_INVOCATION_TYPE_ARGUMENT || targetType == METHOD_INVOCATION_TYPE_ARGUMENT || targetType == CONSTRUCTOR_REFERENCE_TYPE_ARGUMENT || targetType == METHOD_REFERENCE_TYPE_ARGUMENT);
        return info2;
    }

    public int getBoundIndex() {
        assert (targetType == CLASS_TYPE_PARAMETER_BOUND || targetType == METHOD_TYPE_PARAMETER_BOUND);
        return info2;
    }

    public int getMethodFormalParameterIndex() {
        assert (targetType == METHOD_FORMAL_PARAMETER);
        return info;
    }

    public int getThrowsTypeIndex() {
        assert (targetType == THROWS);
        return info;
    }

    public int[] getLocalVarTarget() {
        assert (targetType == LOCAL_VARIABLE || targetType == RESOURCE_VARIABLE);
        return localVarTarget;
    }

    public int getExceptionTableIndex() {
        assert (targetType == EXCEPTION_PARAMETER);
        return info;
    }

    public int getTargetType() {
        return targetType;
    }

    public AnnotationGen getAnnotation() {
        return annotation;
    }

    public int[] getTypePath() {
        return typePath;
    }

    public String getTypePathString() {
        return toTypePathString(typePath);
    }

    public static String toTypePathString(int[] typepath) {
        StringBuilder sb = new StringBuilder();
        int count = 0;
        sb.append("[");
        while (count < typepath.length) {
            if (count > 0)
                sb.append(",");
            switch (typepath[count++]) {
                case TYPE_PATH_ENTRY_KIND_ARRAY:
                    sb.append("ARRAY");
                    count++;
                    break;
                case TYPE_PATH_ENTRY_KIND_INNER_TYPE:
                    sb.append("INNER_TYPE");
                    count++;
                    break;
                case TYPE_PATH_ENTRY_KIND_WILDCARD:
                    sb.append("WILDCARD");
                    count++;
                    break;
                case TYPE_PATH_ENTRY_KIND_TYPE_ARGUMENT:
                    sb.append("TYPE_ARGUMENT(").append(typepath[count++]).append(")");
                    break;
            }
        }
        sb.append("]");
        return sb.toString();
    }
}
