package org.aspectj.apache.bcel.generic;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.classfile.Attribute;
import org.aspectj.apache.bcel.classfile.Constant;
import org.aspectj.apache.bcel.classfile.ConstantDouble;
import org.aspectj.apache.bcel.classfile.ConstantFloat;
import org.aspectj.apache.bcel.classfile.ConstantInteger;
import org.aspectj.apache.bcel.classfile.ConstantLong;
import org.aspectj.apache.bcel.classfile.ConstantObject;
import org.aspectj.apache.bcel.classfile.ConstantPool;
import org.aspectj.apache.bcel.classfile.ConstantString;
import org.aspectj.apache.bcel.classfile.ConstantValue;
import org.aspectj.apache.bcel.classfile.Field;
import org.aspectj.apache.bcel.classfile.Utility;
import org.aspectj.apache.bcel.classfile.annotation.AnnotationGen;
import org.aspectj.apache.bcel.classfile.annotation.RuntimeAnnos;

import java.util.List;

public class FieldGen extends FieldGenOrMethodGen {

    private Object value = null;

    public FieldGen(int modifiers, Type type, String name, ConstantPool cpool) {
        setModifiers(modifiers);
        setType(type);
        setName(name);
        setConstantPool(cpool);
    }

    public FieldGen(Field field, ConstantPool cp) {
        this(field.getModifiers(), Type.getType(field.getSignature()), field.getName(), cp);
        Attribute[] attrs = field.getAttributes();
        for (Attribute attr : attrs) {
            if (attr instanceof ConstantValue) {
                setValue(((ConstantValue) attr).getConstantValueIndex());
            } else if (attr instanceof RuntimeAnnos) {
                RuntimeAnnos runtimeAnnotations = (RuntimeAnnos) attr;
                List<AnnotationGen> l = runtimeAnnotations.getAnnotations();
                for (AnnotationGen element : l) {
                    addAnnotation(new AnnotationGen(element, cp, false));
                }
            } else {
                addAttribute(attr);
            }
        }
    }

    public void setValue(int index) {
        ConstantPool cp = this.cp;
        Constant c = cp.getConstant(index);
        if (c instanceof ConstantInteger) {
            value = ((ConstantInteger) c).getIntValue();
        } else if (c instanceof ConstantFloat) {
            value = ((ConstantFloat) c).getValue();
        } else if (c instanceof ConstantDouble) {
            value = ((ConstantDouble) c).getValue();
        } else if (c instanceof ConstantLong) {
            value = ((ConstantLong) c).getValue();
        } else if (c instanceof ConstantString) {
            value = ((ConstantString) c).getString(cp);
        } else {
            value = ((ConstantObject) c).getConstantValue(cp);
        }
    }

    public void setValue(String constantString) {
        value = constantString;
    }

    public void wipeValue() {
        value = null;
    }

    private void checkType(Type atype) {
        if (type == null)
            throw new ClassGenException("You haven't defined the type of the field yet");
        if (!isFinal())
            throw new ClassGenException("Only final fields may have an initial value!");
        if (!type.equals(atype))
            throw new ClassGenException("Types are not compatible: " + type + " vs. " + atype);
    }

    public Field getField() {
        String signature = getSignature();
        int nameIndex = cp.addUtf8(name);
        int signatureIndex = cp.addUtf8(signature);
        if (value != null) {
            checkType(type);
            int index = addConstant();
            addAttribute(new ConstantValue(cp.addUtf8("ConstantValue"), 2, index, cp));
        }
        addAnnotationsAsAttribute(cp);
        return new Field(modifiers, nameIndex, signatureIndex, getAttributesImmutable(), cp);
    }

    private int addConstant() {
        switch (type.getType()) {
            case Constants.T_INT:
            case Constants.T_CHAR:
            case Constants.T_BYTE:
            case Constants.T_BOOLEAN:
            case Constants.T_SHORT:
                return cp.addInteger((Integer) value);
            case Constants.T_FLOAT:
                return cp.addFloat((Float) value);
            case Constants.T_DOUBLE:
                return cp.addDouble((Double) value);
            case Constants.T_LONG:
                return cp.addLong((Long) value);
            case Constants.T_REFERENCE:
                return cp.addString(((String) value));
            default:
                throw new RuntimeException("Oops: Unhandled : " + type.getType());
        }
    }

    @Override
    public String getSignature() {
        return type.getSignature();
    }

    public String getInitialValue() {
        return (value == null ? null : value.toString());
    }

    public void setInitialStringValue(String value) {
        this.value = value;
    }

    @Override
    public final String toString() {
        String access = Utility.accessToString(modifiers);
        access = access.equals("") ? "" : (access + " ");
        String signature = type.toString();
        String name = getName();
        StringBuilder buf = new StringBuilder(access).append(signature).append(" ").append(name);
        String value = getInitialValue();
        if (value != null) {
            buf.append(" = ").append(value);
        }
        return buf.toString();
    }
}
