package org.aspectj.apache.bcel.generic;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.classfile.ConstantPool;

import java.io.DataOutputStream;
import java.io.IOException;

public class InstructionBranch extends Instruction implements InstructionTargeter {

    private static final int UNSET = -1;

    protected int targetIndex = UNSET;

    protected InstructionHandle targetInstruction;

    protected int positionOfThisInstruction;

    public InstructionBranch(short opcode, InstructionHandle target) {
        super(opcode);
        setTarget(target);
    }

    public InstructionBranch(short opcode, int index) {
        super(opcode);
        this.targetIndex = index;
    }

    public InstructionBranch(short opcode) {
        super(opcode);
    }

    public void dump(DataOutputStream out) throws IOException {
        int target = getTargetOffset();
        if (Math.abs(target) >= 32767 && opcode != GOTO_W && opcode != JSR_W) {
            throw new ClassGenException("Branch target offset too large for short.  Instruction: " + getName().toUpperCase() + "(" + opcode + ")");
        }
        out.writeByte(opcode);
        switch (opcode) {
            case GOTO_W:
            case JSR_W:
                out.writeInt(target);
                break;
            case IF_ACMPEQ:
            case IF_ACMPNE:
            case IF_ICMPEQ:
            case IF_ICMPGE:
            case IF_ICMPGT:
            case IF_ICMPLE:
            case IF_ICMPLT:
            case IF_ICMPNE:
            case IFEQ:
            case IFLE:
            case IFLT:
            case IFGT:
            case IFNE:
            case IFGE:
            case IFNULL:
            case IFNONNULL:
            case GOTO:
            case JSR:
                out.writeShort(target);
                break;
            default:
                throw new IllegalStateException("Don't know how to write out " + getName().toUpperCase());
        }
    }

    protected int getTargetOffset() {
        if (targetInstruction == null && targetIndex == UNSET) {
            throw new ClassGenException("Target of " + super.toString(true) + " is unknown");
        }
        if (targetInstruction == null) {
            return targetIndex;
        } else {
            return targetInstruction.getPosition() - positionOfThisInstruction;
        }
    }

    protected int updatePosition(int offset, int max_offset) {
        int i = getTargetOffset();
        positionOfThisInstruction += offset;
        if (Math.abs(i) >= 32767 - max_offset && opcode != JSR_W && opcode != GOTO_W) {
            if (opcode == JSR || opcode == GOTO) {
                if (opcode == JSR) {
                    opcode = JSR_W;
                } else {
                    opcode = GOTO_W;
                }
                return 2;
            } else {
                throw new IllegalStateException("Unable to pack method, jump (with opcode=" + opcode + ") is too far: " + Math.abs(i));
            }
        }
        return 0;
    }

    public String toString(boolean verbose) {
        String s = super.toString(verbose);
        String t = "null";
        if (verbose) {
            if (targetInstruction != null) {
                if (targetInstruction.getInstruction() == this) {
                    t = "<points to itself>";
                } else if (targetInstruction.getInstruction() == null) {
                    t = "<null destination>";
                } else {
                    t = targetInstruction.getInstruction().toString(false);
                }
            }
        } else {
            if (targetInstruction != null) {
                targetIndex = getTargetOffset();
                t = "" + (targetIndex + positionOfThisInstruction);
            }
        }
        return s + " -> " + t;
    }

    public final int getIndex() {
        return targetIndex;
    }

    public InstructionHandle getTarget() {
        return targetInstruction;
    }

    public void setTarget(InstructionHandle target) {
        notifyTarget(this.targetInstruction, target, this);
        this.targetInstruction = target;
    }

    static final void notifyTarget(InstructionHandle oldHandle, InstructionHandle newHandle, InstructionTargeter t) {
        if (oldHandle != null) {
            oldHandle.removeTargeter(t);
        }
        if (newHandle != null) {
            newHandle.addTargeter(t);
        }
    }

    public void updateTarget(InstructionHandle oldHandle, InstructionHandle newHandle) {
        if (targetInstruction == oldHandle) {
            setTarget(newHandle);
        } else {
            throw new ClassGenException("Not targeting " + oldHandle + ", but " + targetInstruction);
        }
    }

    public boolean containsTarget(InstructionHandle ih) {
        return targetInstruction == ih;
    }

    void dispose() {
        setTarget(null);
        targetIndex = -1;
        positionOfThisInstruction = -1;
    }

    public Type getType(ConstantPool cp) {
        if ((Constants.instFlags[opcode] & Constants.JSR_INSTRUCTION) != 0) {
            return new ReturnaddressType(physicalSuccessor());
        }
        return super.getType(cp);
    }

    public InstructionHandle physicalSuccessor() {
        InstructionHandle ih = this.targetInstruction;
        while (ih.getPrev() != null) {
            ih = ih.getPrev();
        }
        while (ih.getInstruction() != this) {
            ih = ih.getNext();
        }
        InstructionHandle toThis = ih;
        while (ih != null) {
            ih = ih.getNext();
            if (ih != null && ih.getInstruction() == this) {
                throw new RuntimeException("physicalSuccessor() called on a shared JsrInstruction.");
            }
        }
        return toThis.getNext();
    }

    public boolean isIfInstruction() {
        return (Constants.instFlags[opcode] & Constants.IF_INST) != 0;
    }

    public boolean equals(Object other) {
        return this == other;
    }

    public int hashCode() {
        int result = 17;
        result = opcode * 37 + result;
        return result;
    }
}
