package org.aspectj.apache.bcel.generic;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.classfile.Constant;
import org.aspectj.apache.bcel.classfile.ConstantPool;
import org.aspectj.apache.bcel.util.ByteSequence;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

public class InstructionList implements Serializable {

    private InstructionHandle start = null, end = null;

    private int length = 0;

    private int[] positions;

    public InstructionList() {
    }

    public InstructionList(Instruction i) {
        append(i);
    }

    public boolean isEmpty() {
        return start == null;
    }

    public static InstructionHandle findHandle(InstructionHandle[] ihs, int[] pos, int count, int target) {
        return findHandle(ihs, pos, count, target, false);
    }

    public static InstructionHandle findHandle(InstructionHandle[] ihs, int[] pos, int count, int target, boolean returnClosestIfNoExactMatch) {
        int l = 0, r = count - 1;
        int i, j;
        do {
            i = (l + r) / 2;
            j = pos[i];
            if (j == target) {
                return ihs[i];
            } else if (target < j) {
                r = i - 1;
            } else {
                l = i + 1;
            }
        } while (l <= r);
        if (returnClosestIfNoExactMatch) {
            i = (l + r) / 2;
            if (i < 0) {
                i = 0;
            }
            return ihs[i];
        }
        return null;
    }

    public InstructionHandle findHandle(int pos) {
        InstructionHandle[] ihs = getInstructionHandles();
        return findHandle(ihs, positions, length, pos);
    }

    public InstructionHandle[] getInstructionsAsArray() {
        return getInstructionHandles();
    }

    public InstructionHandle findHandle(int pos, InstructionHandle[] instructionArray) {
        return findHandle(instructionArray, positions, length, pos);
    }

    public InstructionHandle findHandle(int pos, InstructionHandle[] instructionArray, boolean useClosestApproximationIfNoExactFound) {
        return findHandle(instructionArray, positions, length, pos, useClosestApproximationIfNoExactFound);
    }

    public InstructionList(byte[] code) {
        ByteSequence bytes = new ByteSequence(code);
        InstructionHandle[] ihs = new InstructionHandle[code.length];
        int[] pos = new int[code.length];
        int count = 0;
        try {
            while (bytes.available() > 0) {
                int off = bytes.getIndex();
                pos[count] = off;
                Instruction i = Instruction.readInstruction(bytes);
                InstructionHandle ih;
                if (i instanceof InstructionBranch) {
                    ih = append((InstructionBranch) i);
                } else {
                    ih = append(i);
                }
                ih.setPosition(off);
                ihs[count] = ih;
                count++;
            }
        } catch (IOException e) {
            throw new ClassGenException(e.toString());
        }
        positions = new int[count];
        System.arraycopy(pos, 0, positions, 0, count);
        for (int i = 0; i < count; i++) {
            if (ihs[i] instanceof BranchHandle) {
                InstructionBranch bi = (InstructionBranch) ihs[i].instruction;
                int target = bi.positionOfThisInstruction + bi.getIndex();
                InstructionHandle ih = findHandle(ihs, pos, count, target);
                if (ih == null) {
                    throw new ClassGenException("Couldn't find target for branch: " + bi);
                }
                bi.setTarget(ih);
                if (bi instanceof InstructionSelect) {
                    InstructionSelect s = (InstructionSelect) bi;
                    int[] indices = s.getIndices();
                    for (int j = 0; j < indices.length; j++) {
                        target = bi.positionOfThisInstruction + indices[j];
                        ih = findHandle(ihs, pos, count, target);
                        if (ih == null) {
                            throw new ClassGenException("Couldn't find target for switch: " + bi);
                        }
                        s.setTarget(j, ih);
                    }
                }
            }
        }
    }

    public InstructionHandle append(InstructionHandle appendTo, InstructionList appendee) {
        assert appendee != null;
        if (appendee.isEmpty()) {
            return appendTo;
        }
        InstructionHandle next = appendTo.next;
        InstructionHandle ret = appendee.start;
        appendTo.next = appendee.start;
        appendee.start.prev = appendTo;
        appendee.end.next = next;
        if (next != null) {
            next.prev = appendee.end;
        } else {
            end = appendee.end;
        }
        length += appendee.length;
        appendee.clear();
        return ret;
    }

    public InstructionHandle append(Instruction i, InstructionList il) {
        InstructionHandle ih;
        if ((ih = findInstruction2(i)) == null) {
            throw new ClassGenException("Instruction " + i + " is not contained in this list.");
        }
        return append(ih, il);
    }

    public InstructionHandle append(InstructionList il) {
        assert il != null;
        if (il.isEmpty()) {
            return null;
        }
        if (isEmpty()) {
            start = il.start;
            end = il.end;
            length = il.length;
            il.clear();
            return start;
        } else {
            return append(end, il);
        }
    }

    private void append(InstructionHandle ih) {
        if (isEmpty()) {
            start = end = ih;
            ih.next = ih.prev = null;
        } else {
            end.next = ih;
            ih.prev = end;
            ih.next = null;
            end = ih;
        }
        length++;
    }

    public InstructionHandle append(Instruction i) {
        InstructionHandle ih = InstructionHandle.getInstructionHandle(i);
        append(ih);
        return ih;
    }

    public InstructionHandle appendDUP() {
        InstructionHandle ih = InstructionHandle.getInstructionHandle(InstructionConstants.DUP);
        append(ih);
        return ih;
    }

    public InstructionHandle appendNOP() {
        InstructionHandle ih = InstructionHandle.getInstructionHandle(InstructionConstants.NOP);
        append(ih);
        return ih;
    }

    public InstructionHandle appendPOP() {
        InstructionHandle ih = InstructionHandle.getInstructionHandle(InstructionConstants.POP);
        append(ih);
        return ih;
    }

    public BranchHandle append(InstructionBranch i) {
        BranchHandle ih = BranchHandle.getBranchHandle(i);
        append(ih);
        return ih;
    }

    public InstructionHandle append(Instruction i, Instruction j) {
        return append(i, new InstructionList(j));
    }

    public InstructionHandle append(InstructionHandle ih, Instruction i) {
        return append(ih, new InstructionList(i));
    }

    public BranchHandle append(InstructionHandle ih, InstructionBranch i) {
        BranchHandle bh = BranchHandle.getBranchHandle(i);
        InstructionList il = new InstructionList();
        il.append(bh);
        append(ih, il);
        return bh;
    }

    public InstructionHandle insert(InstructionHandle ih, InstructionList il) {
        if (il == null) {
            throw new ClassGenException("Inserting null InstructionList");
        }
        if (il.isEmpty()) {
            return ih;
        }
        InstructionHandle prev = ih.prev, ret = il.start;
        ih.prev = il.end;
        il.end.next = ih;
        il.start.prev = prev;
        if (prev != null) {
            prev.next = il.start;
        } else {
            start = il.start;
        }
        length += il.length;
        il.clear();
        return ret;
    }

    public InstructionHandle insert(InstructionList il) {
        if (isEmpty()) {
            append(il);
            return start;
        } else {
            return insert(start, il);
        }
    }

    private void insert(InstructionHandle ih) {
        if (isEmpty()) {
            start = end = ih;
            ih.next = ih.prev = null;
        } else {
            start.prev = ih;
            ih.next = start;
            ih.prev = null;
            start = ih;
        }
        length++;
    }

    public InstructionHandle insert(Instruction i, InstructionList il) {
        InstructionHandle ih;
        if ((ih = findInstruction1(i)) == null) {
            throw new ClassGenException("Instruction " + i + " is not contained in this list.");
        }
        return insert(ih, il);
    }

    public InstructionHandle insert(Instruction i) {
        InstructionHandle ih = InstructionHandle.getInstructionHandle(i);
        insert(ih);
        return ih;
    }

    public BranchHandle insert(InstructionBranch i) {
        BranchHandle ih = BranchHandle.getBranchHandle(i);
        insert(ih);
        return ih;
    }

    public InstructionHandle insert(Instruction i, Instruction j) {
        return insert(i, new InstructionList(j));
    }

    public InstructionHandle insert(InstructionHandle ih, Instruction i) {
        return insert(ih, new InstructionList(i));
    }

    public BranchHandle insert(InstructionHandle ih, InstructionBranch i) {
        BranchHandle bh = BranchHandle.getBranchHandle(i);
        InstructionList il = new InstructionList();
        il.append(bh);
        insert(ih, il);
        return bh;
    }

    public void move(InstructionHandle start, InstructionHandle end, InstructionHandle target) {
        if (start == null || end == null) {
            throw new ClassGenException("Invalid null handle: From " + start + " to " + end);
        }
        if (target == start || target == end) {
            throw new ClassGenException("Invalid range: From " + start + " to " + end + " contains target " + target);
        }
        for (InstructionHandle ih = start; ih != end.next; ih = ih.next) {
            if (ih == null) {
                throw new ClassGenException("Invalid range: From " + start + " to " + end);
            } else if (ih == target) {
                throw new ClassGenException("Invalid range: From " + start + " to " + end + " contains target " + target);
            }
        }
        InstructionHandle prev = start.prev, next = end.next;
        if (prev != null) {
            prev.next = next;
        } else {
            this.start = next;
        }
        if (next != null) {
            next.prev = prev;
        } else {
            this.end = prev;
        }
        start.prev = end.next = null;
        if (target == null) {
            end.next = this.start;
            this.start = start;
        } else {
            next = target.next;
            target.next = start;
            start.prev = target;
            end.next = next;
            if (next != null) {
                next.prev = end;
            }
        }
    }

    public void move(InstructionHandle ih, InstructionHandle target) {
        move(ih, ih, target);
    }

    private void remove(InstructionHandle prev, InstructionHandle next, boolean careAboutLostTargeters) throws TargetLostException {
        InstructionHandle first, last;
        if (prev == null && next == null) {
            first = last = start;
            start = end = null;
        } else {
            if (prev == null) {
                first = start;
                start = next;
            } else {
                first = prev.next;
                prev.next = next;
            }
            if (next == null) {
                last = end;
                end = prev;
            } else {
                last = next.prev;
                next.prev = prev;
            }
        }
        first.prev = null;
        last.next = null;
        if (!careAboutLostTargeters) {
            return;
        }
        List<InstructionHandle> target_vec = new ArrayList<>();
        for (InstructionHandle ih = first; ih != null; ih = ih.next) {
            ih.getInstruction().dispose();
        }
        StringBuilder buf = new StringBuilder("{ ");
        for (InstructionHandle ih = first; ih != null; ih = next) {
            next = ih.next;
            length--;
            Set<InstructionTargeter> targeters = ih.getTargeters();
            boolean isOK = false;
            for (InstructionTargeter instructionTargeter : targeters) {
                if (instructionTargeter.getClass().getName().endsWith("ShadowRange") || instructionTargeter.getClass().getName().endsWith("ExceptionRange") || instructionTargeter.getClass().getName().endsWith("LineNumberTag")) {
                    isOK = true;
                } else {
                    System.out.println(instructionTargeter.getClass());
                }
            }
            if (!isOK) {
                target_vec.add(ih);
                buf.append(ih.toString(true) + " ");
                ih.next = ih.prev = null;
            } else {
                ih.dispose();
            }
        }
        buf.append("}");
        if (!target_vec.isEmpty()) {
            InstructionHandle[] targeted = new InstructionHandle[target_vec.size()];
            target_vec.toArray(targeted);
            throw new TargetLostException(targeted, buf.toString());
        }
    }

    public void delete(InstructionHandle ih) throws TargetLostException {
        remove(ih.prev, ih.next, false);
    }

    public void delete(InstructionHandle from, InstructionHandle to) throws TargetLostException {
        remove(from.prev, to.next, false);
    }

    public void delete(Instruction from, Instruction to) throws TargetLostException {
        InstructionHandle from_ih, to_ih;
        if ((from_ih = findInstruction1(from)) == null) {
            throw new ClassGenException("Instruction " + from + " is not contained in this list.");
        }
        if ((to_ih = findInstruction2(to)) == null) {
            throw new ClassGenException("Instruction " + to + " is not contained in this list.");
        }
        delete(from_ih, to_ih);
    }

    private InstructionHandle findInstruction1(Instruction i) {
        for (InstructionHandle ih = start; ih != null; ih = ih.next) {
            if (ih.instruction == i) {
                return ih;
            }
        }
        return null;
    }

    private InstructionHandle findInstruction2(Instruction i) {
        for (InstructionHandle ih = end; ih != null; ih = ih.prev) {
            if (ih.instruction == i) {
                return ih;
            }
        }
        return null;
    }

    public boolean contains(InstructionHandle i) {
        if (i == null) {
            return false;
        }
        for (InstructionHandle ih = start; ih != null; ih = ih.next) {
            if (ih == i) {
                return true;
            }
        }
        return false;
    }

    public boolean contains(Instruction i) {
        return findInstruction1(i) != null;
    }

    public void setPositions() {
        setPositions(false);
    }

    public void setPositions(boolean check) {
        int maxAdditionalBytes = 0;
        int index = 0, count = 0;
        int[] pos = new int[length];
        if (check) {
            checkInstructionList();
        }
        for (InstructionHandle ih = start; ih != null; ih = ih.next) {
            Instruction i = ih.instruction;
            ih.setPosition(index);
            pos[count++] = index;
            switch (i.opcode) {
                case Constants.JSR:
                case Constants.GOTO:
                    maxAdditionalBytes += 2;
                    break;
                case Constants.TABLESWITCH:
                case Constants.LOOKUPSWITCH:
                    maxAdditionalBytes += 3;
                    break;
            }
            index += i.getLength();
        }
        boolean nonZeroOffset = false;
        int offset = 0;
        for (InstructionHandle ih = start; ih != null; ih = ih.next) {
            if (ih instanceof BranchHandle) {
                offset += ((BranchHandle) ih).updatePosition(offset, maxAdditionalBytes);
                if (offset != 0) {
                    nonZeroOffset = true;
                }
            }
        }
        if (nonZeroOffset) {
            index = count = 0;
            for (InstructionHandle ih = start; ih != null; ih = ih.next) {
                Instruction i = ih.instruction;
                ih.setPosition(index);
                pos[count++] = index;
                index += i.getLength();
            }
        }
        positions = new int[count];
        System.arraycopy(pos, 0, positions, 0, count);
    }

    private void checkInstructionList() {
        for (InstructionHandle ih = start; ih != null; ih = ih.next) {
            Instruction i = ih.instruction;
            if (i instanceof InstructionBranch) {
                Instruction inst = ((InstructionBranch) i).getTarget().instruction;
                if (!contains(inst)) {
                    throw new ClassGenException("Branch target of " + Constants.OPCODE_NAMES[i.opcode] + ":" + inst + " not in instruction list");
                }
                if (i instanceof InstructionSelect) {
                    InstructionHandle[] targets = ((InstructionSelect) i).getTargets();
                    for (InstructionHandle target : targets) {
                        inst = target.instruction;
                        if (!contains(inst)) {
                            throw new ClassGenException("Branch target of " + Constants.OPCODE_NAMES[i.opcode] + ":" + inst + " not in instruction list");
                        }
                    }
                }
                if (!(ih instanceof BranchHandle)) {
                    throw new ClassGenException("Branch instruction " + Constants.OPCODE_NAMES[i.opcode] + ":" + inst + " not contained in BranchHandle.");
                }
            }
        }
    }

    public byte[] getByteCode() {
        setPositions();
        ByteArrayOutputStream b = new ByteArrayOutputStream();
        DataOutputStream out = new DataOutputStream(b);
        try {
            for (InstructionHandle ih = start; ih != null; ih = ih.next) {
                Instruction i = ih.instruction;
                i.dump(out);
            }
        } catch (IOException e) {
            System.err.println(e);
            return null;
        }
        byte[] byteCode = b.toByteArray();
        if (byteCode.length > Constants.MAX_CODE_SIZE) {
            throw new ClassGenException("Code size too big: " + byteCode.length);
        }
        return byteCode;
    }

    public Instruction[] getInstructions() {
        ByteSequence bytes = new ByteSequence(getByteCode());
        List<Instruction> instructions = new ArrayList<>();
        try {
            while (bytes.available() > 0) {
                instructions.add(Instruction.readInstruction(bytes));
            }
        } catch (IOException e) {
            throw new ClassGenException(e.toString());
        }
        Instruction[] result = new Instruction[instructions.size()];
        instructions.toArray(result);
        return result;
    }

    @Override
    public String toString() {
        return toString(true);
    }

    public String toString(boolean verbose) {
        StringBuilder buf = new StringBuilder();
        for (InstructionHandle ih = start; ih != null; ih = ih.next) {
            buf.append(ih.toString(verbose) + "\n");
        }
        return buf.toString();
    }

    public Iterator iterator() {
        return new Iterator() {

            private InstructionHandle ih = start;

            public Object next() {
                InstructionHandle i = ih;
                ih = ih.next;
                return i;
            }

            public void remove() {
                throw new UnsupportedOperationException();
            }

            public boolean hasNext() {
                return ih != null;
            }
        };
    }

    public InstructionHandle[] getInstructionHandles() {
        InstructionHandle[] ihs = new InstructionHandle[length];
        InstructionHandle ih = start;
        for (int i = 0; i < length; i++) {
            ihs[i] = ih;
            ih = ih.next;
        }
        return ihs;
    }

    public int[] getInstructionPositions() {
        return positions;
    }

    public InstructionList copy() {
        HashMap<InstructionHandle, InstructionHandle> map = new HashMap<>();
        InstructionList il = new InstructionList();
        for (InstructionHandle ih = start; ih != null; ih = ih.next) {
            Instruction i = ih.instruction;
            Instruction c = i.copy();
            if (c instanceof InstructionBranch) {
                map.put(ih, il.append((InstructionBranch) c));
            } else {
                map.put(ih, il.append(c));
            }
        }
        InstructionHandle ih = start;
        InstructionHandle ch = il.start;
        while (ih != null) {
            Instruction i = ih.instruction;
            Instruction c = ch.instruction;
            if (i instanceof InstructionBranch) {
                InstructionBranch bi = (InstructionBranch) i;
                InstructionBranch bc = (InstructionBranch) c;
                InstructionHandle itarget = bi.getTarget();
                bc.setTarget(map.get(itarget));
                if (bi instanceof InstructionSelect) {
                    InstructionHandle[] itargets = ((InstructionSelect) bi).getTargets();
                    InstructionHandle[] ctargets = ((InstructionSelect) bc).getTargets();
                    for (int j = 0; j < itargets.length; j++) {
                        ctargets[j] = map.get(itargets[j]);
                    }
                }
            }
            ih = ih.next;
            ch = ch.next;
        }
        return il;
    }

    public void replaceConstantPool(ConstantPool old_cp, ConstantPool new_cp) {
        for (InstructionHandle ih = start; ih != null; ih = ih.next) {
            Instruction i = ih.instruction;
            if (i.isConstantPoolInstruction()) {
                InstructionCP ci = (InstructionCP) i;
                Constant c = old_cp.getConstant(ci.getIndex());
                ci.setIndex(new_cp.addConstant(c, old_cp));
            }
        }
    }

    private void clear() {
        start = end = null;
        length = 0;
    }

    public void dispose() {
        for (InstructionHandle ih = end; ih != null; ih = ih.prev) {
            ih.dispose();
        }
        clear();
    }

    public InstructionHandle getStart() {
        return start;
    }

    public InstructionHandle getEnd() {
        return end;
    }

    public int getLength() {
        return length;
    }

    public int size() {
        return length;
    }

    public void redirectBranches(InstructionHandle old_target, InstructionHandle new_target) {
        for (InstructionHandle ih = start; ih != null; ih = ih.next) {
            Instruction i = ih.getInstruction();
            if (i instanceof InstructionBranch) {
                InstructionBranch b = (InstructionBranch) i;
                InstructionHandle target = b.getTarget();
                if (target == old_target) {
                    b.setTarget(new_target);
                }
                if (b instanceof InstructionSelect) {
                    InstructionHandle[] targets = ((InstructionSelect) b).getTargets();
                    for (int j = 0; j < targets.length; j++) {
                        if (targets[j] == old_target) {
                            ((InstructionSelect) b).setTarget(j, new_target);
                        }
                    }
                }
            }
        }
    }

    public void redirectLocalVariables(LocalVariableGen[] lg, InstructionHandle old_target, InstructionHandle new_target) {
        for (LocalVariableGen localVariableGen : lg) {
            InstructionHandle start = localVariableGen.getStart();
            InstructionHandle end = localVariableGen.getEnd();
            if (start == old_target) {
                localVariableGen.setStart(new_target);
            }
            if (end == old_target) {
                localVariableGen.setEnd(new_target);
            }
        }
    }

    public void redirectExceptionHandlers(CodeExceptionGen[] exceptions, InstructionHandle old_target, InstructionHandle new_target) {
        for (CodeExceptionGen exception : exceptions) {
            if (exception.getStartPC() == old_target) {
                exception.setStartPC(new_target);
            }
            if (exception.getEndPC() == old_target) {
                exception.setEndPC(new_target);
            }
            if (exception.getHandlerPC() == old_target) {
                exception.setHandlerPC(new_target);
            }
        }
    }
}
