package org.aspectj.apache.bcel.generic;

import org.aspectj.apache.bcel.util.ByteSequence;

import java.io.DataOutputStream;
import java.io.IOException;

public abstract class InstructionSelect extends InstructionBranch {

    protected int[] match;

    protected int[] indices;

    protected InstructionHandle[] targets;

    protected int fixedLength;

    protected int matchLength;

    protected int padding = 0;

    protected short length;

    InstructionSelect(short opcode, int[] match, InstructionHandle[] targets, InstructionHandle target) {
        super(opcode, target);
        this.targets = targets;
        for (InstructionHandle instructionHandle : targets) {
            notifyTarget(null, instructionHandle, this);
        }
        this.match = match;
        if ((matchLength = match.length) != targets.length) {
            throw new ClassGenException("Match and target array have not the same length");
        }
        indices = new int[matchLength];
    }

    protected int getTargetOffset(InstructionHandle target) {
        if (target == null) {
            throw new ClassGenException("Target of " + super.toString(true) + " is invalid null handle");
        }
        int t = target.getPosition();
        if (t < 0) {
            throw new ClassGenException("Invalid branch target position offset for " + super.toString(true) + ":" + t + ":" + target);
        }
        return t - positionOfThisInstruction;
    }

    protected int updatePosition(int offset, int max_offset) {
        positionOfThisInstruction += offset;
        short old_length = length;
        padding = (4 - (positionOfThisInstruction + 1) % 4) % 4;
        length = (short) (fixedLength + padding);
        return length - old_length;
    }

    public void dump(DataOutputStream out) throws IOException {
        out.writeByte(opcode);
        for (int i = 0; i < padding; i++) {
            out.writeByte(0);
        }
        targetIndex = getTargetOffset();
        out.writeInt(targetIndex);
    }

    public InstructionSelect(short opcode, ByteSequence bytes) throws IOException {
        super(opcode);
        padding = (4 - bytes.getIndex() % 4) % 4;
        for (int i = 0; i < padding; i++) {
            bytes.readByte();
        }
        targetIndex = bytes.readInt();
    }

    public String toString(boolean verbose) {
        StringBuilder buf = new StringBuilder(super.toString(verbose));
        if (verbose) {
            for (int i = 0; i < matchLength; i++) {
                String s = "null";
                if (targets[i] != null) {
                    s = targets[i].getInstruction().toString();
                }
                buf.append("(" + match[i] + ", " + s + " = {" + indices[i] + "})");
            }
        } else {
            buf.append(" ...");
        }
        return buf.toString();
    }

    public void setTarget(int i, InstructionHandle target) {
        notifyTarget(targets[i], target, this);
        targets[i] = target;
    }

    public void updateTarget(InstructionHandle old_ih, InstructionHandle new_ih) {
        boolean targeted = false;
        if (targetInstruction == old_ih) {
            targeted = true;
            setTarget(new_ih);
        }
        for (int i = 0; i < targets.length; i++) {
            if (targets[i] == old_ih) {
                targeted = true;
                setTarget(i, new_ih);
            }
        }
        if (!targeted) {
            throw new ClassGenException("Not targeting " + old_ih);
        }
    }

    public boolean containsTarget(InstructionHandle ih) {
        if (targetInstruction == ih) {
            return true;
        }
        for (InstructionHandle target : targets) {
            if (target == ih) {
                return true;
            }
        }
        return false;
    }

    void dispose() {
        super.dispose();
        for (InstructionHandle target : targets) {
            target.removeTargeter(this);
        }
    }

    public int[] getMatchs() {
        return match;
    }

    public int[] getIndices() {
        return indices;
    }

    public boolean equals(Object other) {
        return this == other;
    }

    public int hashCode() {
        return opcode * 37;
    }

    public InstructionHandle[] getTargets() {
        return targets;
    }

    public int getLength() {
        return length;
    }
}
