package org.aspectj.apache.bcel.generic;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.classfile.Attribute;
import org.aspectj.apache.bcel.classfile.Code;
import org.aspectj.apache.bcel.classfile.CodeException;
import org.aspectj.apache.bcel.classfile.ConstantPool;
import org.aspectj.apache.bcel.classfile.ExceptionTable;
import org.aspectj.apache.bcel.classfile.LineNumber;
import org.aspectj.apache.bcel.classfile.LineNumberTable;
import org.aspectj.apache.bcel.classfile.LocalVariable;
import org.aspectj.apache.bcel.classfile.LocalVariableTable;
import org.aspectj.apache.bcel.classfile.Method;
import org.aspectj.apache.bcel.classfile.Utility;
import org.aspectj.apache.bcel.classfile.annotation.AnnotationGen;
import org.aspectj.apache.bcel.classfile.annotation.RuntimeAnnos;
import org.aspectj.apache.bcel.classfile.annotation.RuntimeParamAnnos;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Stack;

public class MethodGen extends FieldGenOrMethodGen {

    private String classname;

    private Type[] parameterTypes;

    private String[] parameterNames;

    private int maxLocals;

    private int maxStack;

    private InstructionList il;

    private boolean stripAttributes;

    private int highestLineNumber = 0;

    private List<LocalVariableGen> localVariablesList = new ArrayList<>();

    private List<LineNumberGen> lineNumbersList = new ArrayList<>();

    private ArrayList<CodeExceptionGen> exceptionsList = new ArrayList<>();

    private ArrayList<String> exceptionsThrown = new ArrayList<>();

    private List<Attribute> codeAttributesList = new ArrayList<>();

    private List<AnnotationGen>[] param_annotations;

    private boolean hasParameterAnnotations = false;

    private boolean haveUnpackedParameterAnnotations = false;

    public MethodGen(int access_flags, Type return_type, Type[] arg_types, String[] arg_names, String method_name, String class_name, InstructionList il, ConstantPool cp) {
        this.modifiers = access_flags;
        this.type = return_type;
        this.parameterTypes = arg_types;
        this.parameterNames = arg_names;
        this.name = method_name;
        this.classname = class_name;
        this.il = il;
        this.cp = cp;
    }

    public int getHighestlinenumber() {
        return highestLineNumber;
    }

    public MethodGen(Method m, String class_name, ConstantPool cp) {
        this(m, class_name, cp, false);
    }

    public MethodGen(Method m, String class_name, ConstantPool cp, boolean useTags) {
        this(m.getModifiers(), m.getReturnType(), m.getArgumentTypes(), null, m.getName(), class_name, ((m.getModifiers() & (Constants.ACC_ABSTRACT | Constants.ACC_NATIVE)) == 0) ? new InstructionList(m.getCode().getCode()) : null, cp);
        Attribute[] attributes = m.getAttributes();
        for (Attribute attribute : attributes) {
            Attribute a = attribute;
            if (a instanceof Code) {
                Code code = (Code) a;
                setMaxStack(code.getMaxStack());
                setMaxLocals(code.getMaxLocals());
                CodeException[] ces = code.getExceptionTable();
                InstructionHandle[] arrayOfInstructions = il.getInstructionsAsArray();
                if (ces != null) {
                    for (CodeException ce : ces) {
                        int type = ce.getCatchType();
                        ObjectType catchType = null;
                        if (type > 0) {
                            String cen = m.getConstantPool().getConstantString_CONSTANTClass(type);
                            catchType = new ObjectType(cen);
                        }
                        int end_pc = ce.getEndPC();
                        int length = m.getCode().getCode().length;
                        InstructionHandle end;
                        if (length == end_pc) {
                            end = il.getEnd();
                        } else {
                            end = il.findHandle(end_pc, arrayOfInstructions);
                            end = end.getPrev();
                        }
                        addExceptionHandler(il.findHandle(ce.getStartPC(), arrayOfInstructions), end, il.findHandle(ce.getHandlerPC(), arrayOfInstructions), catchType);
                    }
                }
                Attribute[] codeAttrs = code.getAttributes();
                for (Attribute codeAttr : codeAttrs) {
                    a = codeAttr;
                    if (a instanceof LineNumberTable) {
                        LineNumber[] ln = ((LineNumberTable) a).getLineNumberTable();
                        if (useTags) {
                            for (LineNumber l : ln) {
                                int lnum = l.getLineNumber();
                                if (lnum > highestLineNumber) {
                                    highestLineNumber = lnum;
                                }
                                LineNumberTag lt = new LineNumberTag(lnum);
                                il.findHandle(l.getStartPC(), arrayOfInstructions, true).addTargeter(lt);
                            }
                        } else {
                            for (LineNumber l : ln) {
                                addLineNumber(il.findHandle(l.getStartPC(), arrayOfInstructions, true), l.getLineNumber());
                            }
                        }
                    } else if (a instanceof LocalVariableTable) {
                        if (useTags) {
                            LocalVariable[] lv = ((LocalVariableTable) a).getLocalVariableTable();
                            for (LocalVariable l : lv) {
                                Type t = Type.getType(l.getSignature());
                                LocalVariableTag lvt = new LocalVariableTag(t, l.getSignature(), l.getName(), l.getIndex(), l.getStartPC());
                                InstructionHandle start = il.findHandle(l.getStartPC(), arrayOfInstructions, true);
                                byte b = t.getType();
                                if (b != Constants.T_ADDRESS) {
                                    int increment = t.getSize();
                                    if (l.getIndex() + increment > maxLocals) {
                                        maxLocals = l.getIndex() + increment;
                                    }
                                }
                                int end = l.getStartPC() + l.getLength();
                                do {
                                    start.addTargeter(lvt);
                                    start = start.getNext();
                                } while (start != null && start.getPosition() < end);
                            }
                        } else {
                            LocalVariable[] lv = ((LocalVariableTable) a).getLocalVariableTable();
                            removeLocalVariables();
                            for (LocalVariable l : lv) {
                                InstructionHandle start = il.findHandle(l.getStartPC(), arrayOfInstructions);
                                InstructionHandle end = il.findHandle(l.getStartPC() + l.getLength(), arrayOfInstructions);
                                if (end != null) {
                                    end = end.getPrev();
                                }
                                if (null == start) {
                                    start = il.getStart();
                                }
                                if (null == end) {
                                    end = il.getEnd();
                                }
                                addLocalVariable(l.getName(), Type.getType(l.getSignature()), l.getIndex(), start, end);
                            }
                        }
                    } else {
                        addCodeAttribute(a);
                    }
                }
            } else if (a instanceof ExceptionTable) {
                String[] names = ((ExceptionTable) a).getExceptionNames();
                for (String s : names) {
                    addException(s);
                }
            } else if (a instanceof RuntimeAnnos) {
                RuntimeAnnos runtimeAnnotations = (RuntimeAnnos) a;
                List<AnnotationGen> l = runtimeAnnotations.getAnnotations();
                annotationList.addAll(l);
            } else {
                addAttribute(a);
            }
        }
    }

    public LocalVariableGen addLocalVariable(String name, Type type, int slot, InstructionHandle start, InstructionHandle end) {
        int size = type.getSize();
        if (slot + size > maxLocals) {
            maxLocals = slot + size;
        }
        LocalVariableGen l = new LocalVariableGen(slot, name, type, start, end);
        int i = localVariablesList.indexOf(l);
        if (i >= 0) {
            localVariablesList.set(i, l);
        } else {
            localVariablesList.add(l);
        }
        return l;
    }

    public LocalVariableGen addLocalVariable(String name, Type type, InstructionHandle start, InstructionHandle end) {
        return addLocalVariable(name, type, maxLocals, start, end);
    }

    public void removeLocalVariable(LocalVariableGen l) {
        localVariablesList.remove(l);
    }

    public void removeLocalVariables() {
        localVariablesList.clear();
    }

    private static final void sort(LocalVariableGen[] vars, int l, int r) {
        int i = l, j = r;
        int m = vars[(l + r) / 2].getIndex();
        LocalVariableGen h;
        do {
            while (vars[i].getIndex() < m) {
                i++;
            }
            while (m < vars[j].getIndex()) {
                j--;
            }
            if (i <= j) {
                h = vars[i];
                vars[i] = vars[j];
                vars[j] = h;
                i++;
                j--;
            }
        } while (i <= j);
        if (l < j) {
            sort(vars, l, j);
        }
        if (i < r) {
            sort(vars, i, r);
        }
    }

    public LocalVariableGen[] getLocalVariables() {
        int size = localVariablesList.size();
        LocalVariableGen[] lg = new LocalVariableGen[size];
        localVariablesList.toArray(lg);
        for (int i = 0; i < size; i++) {
            if (lg[i].getStart() == null) {
                lg[i].setStart(il.getStart());
            }
            if (lg[i].getEnd() == null) {
                lg[i].setEnd(il.getEnd());
            }
        }
        if (size > 1) {
            sort(lg, 0, size - 1);
        }
        return lg;
    }

    public LocalVariableTable getLocalVariableTable(ConstantPool cp) {
        LocalVariableGen[] lg = getLocalVariables();
        int size = lg.length;
        LocalVariable[] lv = new LocalVariable[size];
        for (int i = 0; i < size; i++) {
            lv[i] = lg[i].getLocalVariable(cp);
        }
        return new LocalVariableTable(cp.addUtf8("LocalVariableTable"), 2 + lv.length * 10, lv, cp);
    }

    public LineNumberGen addLineNumber(InstructionHandle ih, int src_line) {
        LineNumberGen l = new LineNumberGen(ih, src_line);
        lineNumbersList.add(l);
        return l;
    }

    public void removeLineNumber(LineNumberGen l) {
        lineNumbersList.remove(l);
    }

    public void removeLineNumbers() {
        lineNumbersList.clear();
    }

    public LineNumberGen[] getLineNumbers() {
        LineNumberGen[] lg = new LineNumberGen[lineNumbersList.size()];
        lineNumbersList.toArray(lg);
        return lg;
    }

    public LineNumberTable getLineNumberTable(ConstantPool cp) {
        int size = lineNumbersList.size();
        LineNumber[] ln = new LineNumber[size];
        for (int i = 0; i < size; i++) {
            ln[i] = lineNumbersList.get(i).getLineNumber();
        }
        return new LineNumberTable(cp.addUtf8("LineNumberTable"), 2 + ln.length * 4, ln, cp);
    }

    public CodeExceptionGen addExceptionHandler(InstructionHandle start_pc, InstructionHandle end_pc, InstructionHandle handler_pc, ObjectType catch_type) {
        if ((start_pc == null) || (end_pc == null) || (handler_pc == null)) {
            throw new ClassGenException("Exception handler target is null instruction");
        }
        CodeExceptionGen c = new CodeExceptionGen(start_pc, end_pc, handler_pc, catch_type);
        exceptionsList.add(c);
        return c;
    }

    public void removeExceptionHandler(CodeExceptionGen c) {
        exceptionsList.remove(c);
    }

    public void removeExceptionHandlers() {
        exceptionsList.clear();
    }

    public CodeExceptionGen[] getExceptionHandlers() {
        CodeExceptionGen[] cg = new CodeExceptionGen[exceptionsList.size()];
        exceptionsList.toArray(cg);
        return cg;
    }

    private CodeException[] getCodeExceptions() {
        int size = exceptionsList.size();
        CodeException[] c_exc = new CodeException[size];
        try {
            for (int i = 0; i < size; i++) {
                CodeExceptionGen c = exceptionsList.get(i);
                c_exc[i] = c.getCodeException(cp);
            }
        } catch (ArrayIndexOutOfBoundsException e) {
        }
        return c_exc;
    }

    public void addException(String class_name) {
        exceptionsThrown.add(class_name);
    }

    public void removeException(String c) {
        exceptionsThrown.remove(c);
    }

    public void removeExceptions() {
        exceptionsThrown.clear();
    }

    public String[] getExceptions() {
        String[] e = new String[exceptionsThrown.size()];
        exceptionsThrown.toArray(e);
        return e;
    }

    private ExceptionTable getExceptionTable(ConstantPool cp) {
        int size = exceptionsThrown.size();
        int[] ex = new int[size];
        try {
            for (int i = 0; i < size; i++) {
                ex[i] = cp.addClass(exceptionsThrown.get(i));
            }
        } catch (ArrayIndexOutOfBoundsException e) {
        }
        return new ExceptionTable(cp.addUtf8("Exceptions"), 2 + 2 * size, ex, cp);
    }

    public void addCodeAttribute(Attribute a) {
        codeAttributesList.add(a);
    }

    public void addParameterAnnotationsAsAttribute(ConstantPool cp) {
        if (!hasParameterAnnotations) {
            return;
        }
        Attribute[] attrs = Utility.getParameterAnnotationAttributes(cp, param_annotations);
        if (attrs != null) {
            for (Attribute attr : attrs) {
                addAttribute(attr);
            }
        }
    }

    public void removeCodeAttribute(Attribute a) {
        codeAttributesList.remove(a);
    }

    public void removeCodeAttributes() {
        codeAttributesList.clear();
    }

    public Attribute[] getCodeAttributes() {
        Attribute[] attributes = new Attribute[codeAttributesList.size()];
        codeAttributesList.toArray(attributes);
        return attributes;
    }

    public Method getMethod() {
        String signature = getSignature();
        int name_index = cp.addUtf8(name);
        int signature_index = cp.addUtf8(signature);
        byte[] byte_code = null;
        if (il != null) {
            try {
                byte_code = il.getByteCode();
            } catch (Exception e) {
                throw new IllegalStateException("Unexpected problem whilst preparing bytecode for " + this.getClassName() + "." + this.getName() + this.getSignature(), e);
            }
        }
        LineNumberTable lnt = null;
        LocalVariableTable lvt = null;
        if ((localVariablesList.size() > 0) && !stripAttributes) {
            addCodeAttribute(lvt = getLocalVariableTable(cp));
        }
        if ((lineNumbersList.size() > 0) && !stripAttributes) {
            addCodeAttribute(lnt = getLineNumberTable(cp));
        }
        Attribute[] code_attrs = getCodeAttributes();
        int attrs_len = 0;
        for (Attribute code_attr : code_attrs) {
            attrs_len += (code_attr.getLength() + 6);
        }
        CodeException[] c_exc = getCodeExceptions();
        int exc_len = c_exc.length * 8;
        Code code = null;
        if ((il != null) && !isAbstract()) {
            List<Attribute> attributes = getAttributes();
            for (Attribute a : attributes) {
                if (a instanceof Code) {
                    removeAttribute(a);
                }
            }
            code = new Code(cp.addUtf8("Code"), 8 + byte_code.length + 2 + exc_len + 2 + attrs_len, maxStack, maxLocals, byte_code, c_exc, code_attrs, cp);
            addAttribute(code);
        }
        addAnnotationsAsAttribute(cp);
        addParameterAnnotationsAsAttribute(cp);
        ExceptionTable et = null;
        if (exceptionsThrown.size() > 0) {
            addAttribute(et = getExceptionTable(cp));
        }
        Method m = new Method(modifiers, name_index, signature_index, getAttributesImmutable(), cp);
        if (lvt != null) {
            removeCodeAttribute(lvt);
        }
        if (lnt != null) {
            removeCodeAttribute(lnt);
        }
        if (code != null) {
            removeAttribute(code);
        }
        if (et != null) {
            removeAttribute(et);
        }
        return m;
    }

    public void setMaxLocals(int m) {
        maxLocals = m;
    }

    public int getMaxLocals() {
        return maxLocals;
    }

    public void setMaxStack(int m) {
        maxStack = m;
    }

    public int getMaxStack() {
        return maxStack;
    }

    public String getClassName() {
        return classname;
    }

    public void setClassName(String class_name) {
        this.classname = class_name;
    }

    public void setReturnType(Type return_type) {
        setType(return_type);
    }

    public Type getReturnType() {
        return getType();
    }

    public void setArgumentTypes(Type[] arg_types) {
        this.parameterTypes = arg_types;
    }

    public Type[] getArgumentTypes() {
        return this.parameterTypes;
    }

    public void setArgumentType(int i, Type type) {
        parameterTypes[i] = type;
    }

    public Type getArgumentType(int i) {
        return parameterTypes[i];
    }

    public void setArgumentNames(String[] arg_names) {
        this.parameterNames = arg_names;
    }

    public String[] getArgumentNames() {
        if (parameterNames != null) {
            return parameterNames.clone();
        } else {
            return new String[0];
        }
    }

    public void setArgumentName(int i, String name) {
        parameterNames[i] = name;
    }

    public String getArgumentName(int i) {
        return parameterNames[i];
    }

    public InstructionList getInstructionList() {
        return il;
    }

    public void setInstructionList(InstructionList il) {
        this.il = il;
    }

    @Override
    public String getSignature() {
        return Utility.toMethodSignature(type, parameterTypes);
    }

    public void setMaxStack() {
        if (il != null) {
            maxStack = getMaxStack(cp, il, getExceptionHandlers());
        } else {
            maxStack = 0;
        }
    }

    public void setMaxLocals() {
        setMaxLocals(false);
    }

    public void setMaxLocals(boolean respectLocalVariableTable) {
        if (il != null) {
            int max = isStatic() ? 0 : 1;
            if (parameterTypes != null) {
                for (Type parameterType : parameterTypes) {
                    max += parameterType.getSize();
                }
            }
            for (InstructionHandle ih = il.getStart(); ih != null; ih = ih.getNext()) {
                Instruction ins = ih.getInstruction();
                if ((ins instanceof InstructionLV) || (ins instanceof RET)) {
                    int index = ins.getIndex() + ins.getType(cp).getSize();
                    if (index > max) {
                        max = index;
                    }
                }
            }
            if (!respectLocalVariableTable || max > maxLocals) {
                maxLocals = max;
            }
        } else {
            if (!respectLocalVariableTable) {
                maxLocals = 0;
            }
        }
    }

    public void stripAttributes(boolean flag) {
        stripAttributes = flag;
    }

    static final class BranchTarget {

        InstructionHandle target;

        int stackDepth;

        BranchTarget(InstructionHandle target, int stackDepth) {
            this.target = target;
            this.stackDepth = stackDepth;
        }
    }

    static final class BranchStack {

        Stack<BranchTarget> branchTargets = new Stack<>();

        Map<InstructionHandle, BranchTarget> visitedTargets = new Hashtable<>();

        public void push(InstructionHandle target, int stackDepth) {
            if (visited(target)) {
                return;
            }
            branchTargets.push(visit(target, stackDepth));
        }

        public BranchTarget pop() {
            if (!branchTargets.empty()) {
                BranchTarget bt = branchTargets.pop();
                return bt;
            }
            return null;
        }

        private final BranchTarget visit(InstructionHandle target, int stackDepth) {
            BranchTarget bt = new BranchTarget(target, stackDepth);
            visitedTargets.put(target, bt);
            return bt;
        }

        private final boolean visited(InstructionHandle target) {
            return (visitedTargets.get(target) != null);
        }
    }

    public static int getMaxStack(ConstantPool cp, InstructionList il, CodeExceptionGen[] et) {
        BranchStack branchTargets = new BranchStack();
        int stackDepth = 0;
        int maxStackDepth = 0;
        for (CodeExceptionGen codeExceptionGen : et) {
            InstructionHandle handlerPos = codeExceptionGen.getHandlerPC();
            if (handlerPos != null) {
                maxStackDepth = 1;
                branchTargets.push(handlerPos, 1);
            }
        }
        InstructionHandle ih = il.getStart();
        while (ih != null) {
            Instruction instruction = ih.getInstruction();
            short opcode = instruction.opcode;
            int prod = instruction.produceStack(cp);
            int con = instruction.consumeStack(cp);
            int delta = prod - con;
            stackDepth += delta;
            if (stackDepth > maxStackDepth) {
                maxStackDepth = stackDepth;
            }
            if (instruction instanceof InstructionBranch) {
                InstructionBranch branch = (InstructionBranch) instruction;
                if (instruction instanceof InstructionSelect) {
                    InstructionSelect select = (InstructionSelect) branch;
                    InstructionHandle[] targets = select.getTargets();
                    for (InstructionHandle target : targets) {
                        branchTargets.push(target, stackDepth);
                    }
                    ih = null;
                } else if (!(branch.isIfInstruction())) {
                    if (opcode == Constants.JSR || opcode == Constants.JSR_W) {
                        branchTargets.push(ih.getNext(), stackDepth - 1);
                    }
                    ih = null;
                }
                branchTargets.push(branch.getTarget(), stackDepth);
            } else {
                if (opcode == Constants.ATHROW || opcode == Constants.RET || (opcode >= Constants.IRETURN && opcode <= Constants.RETURN)) {
                    ih = null;
                }
            }
            if (ih != null) {
                ih = ih.getNext();
            }
            if (ih == null) {
                BranchTarget bt = branchTargets.pop();
                if (bt != null) {
                    ih = bt.target;
                    stackDepth = bt.stackDepth;
                }
            }
        }
        return maxStackDepth;
    }

    @Override
    public final String toString() {
        String access = Utility.accessToString(modifiers);
        String signature = Utility.toMethodSignature(type, parameterTypes);
        signature = Utility.methodSignatureToString(signature, name, access, true, getLocalVariableTable(cp));
        StringBuilder buf = new StringBuilder(signature);
        if (exceptionsThrown.size() > 0) {
            for (String s : exceptionsThrown) {
                buf.append("\n\t\tthrows " + s);
            }
        }
        return buf.toString();
    }

    public List<AnnotationGen> getAnnotationsOnParameter(int i) {
        ensureExistingParameterAnnotationsUnpacked();
        if (!hasParameterAnnotations || i > parameterTypes.length) {
            return null;
        }
        return param_annotations[i];
    }

    private void ensureExistingParameterAnnotationsUnpacked() {
        if (haveUnpackedParameterAnnotations) {
            return;
        }
        List<Attribute> attrs = getAttributes();
        RuntimeParamAnnos paramAnnVisAttr = null;
        RuntimeParamAnnos paramAnnInvisAttr = null;
        for (Attribute attribute : attrs) {
            if (attribute instanceof RuntimeParamAnnos) {
                if (!hasParameterAnnotations) {
                    param_annotations = new List[parameterTypes.length];
                    for (int j = 0; j < parameterTypes.length; j++) {
                        param_annotations[j] = new ArrayList<>();
                    }
                }
                hasParameterAnnotations = true;
                RuntimeParamAnnos rpa = (RuntimeParamAnnos) attribute;
                if (rpa.areVisible()) {
                    paramAnnVisAttr = rpa;
                } else {
                    paramAnnInvisAttr = rpa;
                }
                for (int j = 0; j < parameterTypes.length; j++) {
                    AnnotationGen[] annos = rpa.getAnnotationsOnParameter(j);
                    for (AnnotationGen anAnnotation : annos) {
                        param_annotations[j].add(anAnnotation);
                    }
                }
            }
        }
        if (paramAnnVisAttr != null) {
            removeAttribute(paramAnnVisAttr);
        }
        if (paramAnnInvisAttr != null) {
            removeAttribute(paramAnnInvisAttr);
        }
        haveUnpackedParameterAnnotations = true;
    }

    private List<AnnotationGen> makeMutableVersion(AnnotationGen[] mutableArray) {
        List<AnnotationGen> result = new ArrayList<>();
        for (AnnotationGen annotationGen : mutableArray) {
            result.add(new AnnotationGen(annotationGen, getConstantPool(), false));
        }
        return result;
    }

    public void addParameterAnnotation(int parameterIndex, AnnotationGen annotation) {
        ensureExistingParameterAnnotationsUnpacked();
        if (!hasParameterAnnotations) {
            param_annotations = new List[parameterTypes.length];
            hasParameterAnnotations = true;
        }
        List<AnnotationGen> existingAnnotations = param_annotations[parameterIndex];
        if (existingAnnotations != null) {
            existingAnnotations.add(annotation);
        } else {
            List<AnnotationGen> l = new ArrayList<>();
            l.add(annotation);
            param_annotations[parameterIndex] = l;
        }
    }
}
