package org.aspectj.apache.bcel.util;

import org.aspectj.apache.bcel.classfile.ClassParser;
import org.aspectj.apache.bcel.classfile.JavaClass;

import java.io.IOException;
import java.io.InputStream;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

public class NonCachingClassLoaderRepository implements Repository {

    private static java.lang.ClassLoader bootClassLoader = null;

    private final ClassLoaderReference loaderRef;

    private final Map<String, JavaClass> loadedClasses = new SoftHashMap();

    public static class SoftHashMap extends AbstractMap {

        private Map<Object, SpecialValue> map;

        private ReferenceQueue rq = new ReferenceQueue();

        public SoftHashMap(Map<Object, SpecialValue> map) {
            this.map = map;
        }

        public SoftHashMap() {
            this(new HashMap<>());
        }

        public SoftHashMap(Map<Object, SpecialValue> map, boolean b) {
            this(map);
        }

        class SpecialValue extends SoftReference {

            private final Object key;

            SpecialValue(Object k, Object v) {
                super(v, rq);
                this.key = k;
            }
        }

        private void processQueue() {
            SpecialValue sv = null;
            while ((sv = (SpecialValue) rq.poll()) != null) {
                map.remove(sv.key);
            }
        }

        @Override
        public Object get(Object key) {
            SpecialValue value = map.get(key);
            if (value == null)
                return null;
            if (value.get() == null) {
                map.remove(value.key);
                return null;
            } else {
                return value.get();
            }
        }

        @Override
        public Object put(Object k, Object v) {
            processQueue();
            return map.put(k, new SpecialValue(k, v));
        }

        @Override
        public Set entrySet() {
            return map.entrySet();
        }

        @Override
        public void clear() {
            processQueue();
            Set<Object> keys = map.keySet();
            for (Object name : keys) {
                map.remove(name);
            }
        }

        @Override
        public int size() {
            processQueue();
            return map.size();
        }

        @Override
        public Object remove(Object k) {
            processQueue();
            SpecialValue value = map.remove(k);
            if (value == null)
                return null;
            if (value.get() != null) {
                return value.get();
            }
            return null;
        }
    }

    public NonCachingClassLoaderRepository(java.lang.ClassLoader loader) {
        this.loaderRef = new DefaultClassLoaderReference((loader != null) ? loader : getBootClassLoader());
    }

    public NonCachingClassLoaderRepository(ClassLoaderReference loaderRef) {
        this.loaderRef = loaderRef;
    }

    private static synchronized java.lang.ClassLoader getBootClassLoader() {
        if (bootClassLoader == null) {
            bootClassLoader = new URLClassLoader(new URL[0]);
        }
        return bootClassLoader;
    }

    public void storeClass(JavaClass clazz) {
        synchronized (loadedClasses) {
            loadedClasses.put(clazz.getClassName(), clazz);
        }
        clazz.setRepository(this);
    }

    public void removeClass(JavaClass clazz) {
        synchronized (loadedClasses) {
            loadedClasses.remove(clazz.getClassName());
        }
    }

    public JavaClass findClass(String className) {
        synchronized (loadedClasses) {
            return loadedClasses.getOrDefault(className, null);
        }
    }

    public void clear() {
        synchronized (loadedClasses) {
            loadedClasses.clear();
        }
    }

    public JavaClass loadClass(String className) throws ClassNotFoundException {
        JavaClass javaClass = findClass(className);
        if (javaClass != null) {
            return javaClass;
        }
        javaClass = loadJavaClass(className);
        storeClass(javaClass);
        return javaClass;
    }

    public JavaClass loadClass(Class clazz) throws ClassNotFoundException {
        return loadClass(clazz.getName());
    }

    private JavaClass loadJavaClass(String className) throws ClassNotFoundException {
        String classFile = className.replace('.', '/');
        try {
            InputStream is = loaderRef.getClassLoader().getResourceAsStream(classFile + ".class");
            if (is == null) {
                throw new ClassNotFoundException(className + " not found.");
            }
            ClassParser parser = new ClassParser(is, className);
            return parser.parse();
        } catch (IOException e) {
            throw new ClassNotFoundException(e.toString());
        }
    }
}
