package org.aspectj.apache.bcel.util;

import org.aspectj.apache.bcel.classfile.ClassParser;
import org.aspectj.apache.bcel.classfile.JavaClass;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.WeakHashMap;

public class SyntheticRepository implements Repository {

    private static final String DEFAULT_PATH = ClassPath.getClassPath();

    private static HashMap<ClassPath, SyntheticRepository> _instances = new HashMap<>();

    private ClassPath _path = null;

    private WeakHashMap<String, JavaClass> _loadedClasses = new WeakHashMap<>();

    private SyntheticRepository(ClassPath path) {
        _path = path;
    }

    public static SyntheticRepository getInstance() {
        return getInstance(ClassPath.getSystemClassPath());
    }

    public static SyntheticRepository getInstance(ClassPath classPath) {
        SyntheticRepository rep = _instances.get(classPath);
        if (rep == null) {
            rep = new SyntheticRepository(classPath);
            _instances.put(classPath, rep);
        }
        return rep;
    }

    public void storeClass(JavaClass clazz) {
        _loadedClasses.put(clazz.getClassName(), clazz);
        clazz.setRepository(this);
    }

    public void removeClass(JavaClass clazz) {
        _loadedClasses.remove(clazz.getClassName());
    }

    public JavaClass findClass(String className) {
        return _loadedClasses.get(className);
    }

    public JavaClass loadClass(String className) throws ClassNotFoundException {
        if (className == null || className.equals("")) {
            throw new IllegalArgumentException("Invalid class name " + className);
        }
        className = className.replace('/', '.');
        try {
            return loadClass(_path.getInputStream(className), className);
        } catch (IOException e) {
            throw new ClassNotFoundException("Exception while looking for class " + className + ": " + e.toString());
        }
    }

    public JavaClass loadClass(Class clazz) throws ClassNotFoundException {
        String className = clazz.getName();
        String name = className;
        int i = name.lastIndexOf('.');
        if (i > 0) {
            name = name.substring(i + 1);
        }
        return loadClass(clazz.getResourceAsStream(name + ".class"), className);
    }

    private JavaClass loadClass(InputStream is, String className) throws ClassNotFoundException {
        JavaClass clazz = findClass(className);
        if (clazz != null) {
            return clazz;
        }
        try {
            if (is != null) {
                ClassParser parser = new ClassParser(is, className);
                clazz = parser.parse();
                storeClass(clazz);
                return clazz;
            }
        } catch (IOException e) {
            throw new ClassNotFoundException("Exception while looking for class " + className + ": " + e.toString());
        }
        throw new ClassNotFoundException("SyntheticRepository could not load " + className);
    }

    public void clear() {
        _loadedClasses.clear();
    }
}
