package org.aspectj.asm.internal;

import org.aspectj.asm.IProgramElement;
import org.aspectj.asm.IRelationship;
import org.aspectj.asm.IRelationship.Kind;
import org.aspectj.asm.IRelationshipMap;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

public class RelationshipMap extends HashMap<String, List<IRelationship>> implements IRelationshipMap {

    private static final long serialVersionUID = 496638323566589643L;

    public RelationshipMap() {
    }

    public List<IRelationship> get(String handle) {
        List<IRelationship> relationships = super.get(handle);
        if (relationships == null) {
            return null;
        } else {
            return relationships;
        }
    }

    public List<IRelationship> get(IProgramElement source) {
        return get(source.getHandleIdentifier());
    }

    public IRelationship get(String source, IRelationship.Kind kind, String relationshipName, boolean runtimeTest, boolean createIfMissing) {
        List<IRelationship> relationships = get(source);
        if (relationships == null) {
            if (!createIfMissing) {
                return null;
            }
            relationships = new ArrayList<>();
            IRelationship rel = new Relationship(relationshipName, kind, source, new ArrayList<>(), runtimeTest);
            relationships.add(rel);
            super.put(source, relationships);
            return rel;
        } else {
            for (IRelationship curr : relationships) {
                if (curr.getKind() == kind && curr.getName().equals(relationshipName) && curr.hasRuntimeTest() == runtimeTest) {
                    return curr;
                }
            }
            if (createIfMissing) {
                IRelationship rel = new Relationship(relationshipName, kind, source, new ArrayList<>(), runtimeTest);
                relationships.add(rel);
                return rel;
            }
        }
        return null;
    }

    public IRelationship get(IProgramElement source, IRelationship.Kind kind, String relationshipName, boolean runtimeTest, boolean createIfMissing) {
        return get(source.getHandleIdentifier(), kind, relationshipName, runtimeTest, createIfMissing);
    }

    public IRelationship get(IProgramElement source, Kind kind, String relationshipName) {
        return get(source, kind, relationshipName, false, true);
    }

    public boolean remove(String source, IRelationship relationship) {
        List<IRelationship> list = super.get(source);
        if (list != null) {
            return list.remove(relationship);
        }
        return false;
    }

    public void removeAll(String source) {
        super.remove(source);
    }

    public void put(String source, IRelationship relationship) {
        List<IRelationship> existingRelationships = super.get(source);
        if (existingRelationships == null) {
            existingRelationships = new ArrayList<>();
            existingRelationships.add(relationship);
            super.put(source, existingRelationships);
        } else {
            boolean matched = false;
            for (IRelationship existingRelationship : existingRelationships) {
                if (existingRelationship.getName().equals(relationship.getName()) && existingRelationship.getKind() == relationship.getKind()) {
                    existingRelationship.getTargets().addAll(relationship.getTargets());
                    matched = true;
                }
            }
            if (matched) {
                System.err.println("matched = true");
            }
            if (matched) {
                existingRelationships.add(relationship);
            }
        }
    }

    public void put(IProgramElement source, IRelationship relationship) {
        put(source.getHandleIdentifier(), relationship);
    }

    public void clear() {
        super.clear();
    }

    public Set<String> getEntries() {
        return keySet();
    }
}
